/*
 * Rahmenwerk-Plug-in "Protokolle und Auswertungen"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.pua.ganglinien.data;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;

import org.eclipse.core.runtime.Assert;

import com.bitctrl.lib.eclipse.databinding.realm.results.VoidResult;

import de.bsvrz.buv.plugin.pua.daten.RohDatenSatz;
import de.bsvrz.buv.plugin.pua.ganglinien.PuaGanglinienFormPage;
import de.bsvrz.buv.plugin.pua.ganglinien.internal.RahmenwerkService;
import de.bsvrz.buv.plugin.pua.ganglinien.model.LineProperties;
import de.bsvrz.buv.plugin.pua.ganglinien.model.observables.PUAGanglinienOO;
import de.bsvrz.buv.plugin.pua.ganglinien.model.observables.RootOO;
import de.bsvrz.buv.plugin.pua.ganglinien.twoDimMap.TwoDimensionalEMFMap;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.config.DoubleAttributeType;
import de.bsvrz.dav.daf.main.config.IntegerAttributeType;
import de.bsvrz.dav.daf.main.config.IntegerValueRange;
import de.bsvrz.dav.daf.main.config.IntegerValueState;
import de.bsvrz.pua.prot.client.dataobject.Column;
import de.bsvrz.pua.prot.client.dataobject.ProtocolAdministrationData;
import de.bsvrz.pua.prot.data.ProtocolData;
import de.bsvrz.pua.prot.util.ProcessingInformation.ApplyAggregation;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.losb.util.Tuple;

/**
 * Verarbeitet jeden Listen- und Aggregations-Datensatz.
 *
 * @author BitCtrl Systems GmbH, Enrico Schnepel
 *
 */
public class ProcessResultData implements VoidResult {

	/**
	 * die FormPage.
	 */
	private final PuaGanglinienFormPage puaGanglinienFormPage;

	/**
	 * Die Ergebnisdatensätze müssen je Aggregationstyp separat gespeichert werden.
	 */
	private final Map<AggregationsTyp, RohDatenSatz> resultDataMap = new HashMap<>();

	/**
	 * Zeitstempel des letzten Datensatzes. Wird benötigt, da einige Datensatze
	 * keinen Zeitstempel haben und hier der letzte Zeitstemepl übernommen wird.
	 */
	private long lastTimeStamp;

	/** der Aggregationstyp des Datensatzes. */
	private AggregationsTyp typ;

	/**
	 * Konstruktor.
	 *
	 * @param puaGanglinienFormPage die FormPage
	 */
	ProcessResultData(final PuaGanglinienFormPage puaGanglinienFormPage) {
		this.puaGanglinienFormPage = puaGanglinienFormPage;
	}

	@Override
	public void run() {
		final RootOO root = puaGanglinienFormPage.getRoot();
		final ProtocolAdministrationData adminData = root.getProtokoll().getAdministrationDataValue();
		final RohDatenSatz resultData = resultDataMap.get(typ);
		final ClientDavInterface dav = RahmenwerkService.getService().getRahmenWerk().getDavVerbindung();
		Assert.isTrue(dav != null);
		Assert.isTrue(adminData.columns.length == resultData.getDataList(dav.getDataModel()).size());
		long timeStamp = resultData.getTimeStamp().getTime();

		if (0 == timeStamp) {
			timeStamp = lastTimeStamp;
		} else {
			lastTimeStamp = timeStamp;
		}

		final PUAGanglinienOO puaGanglinien = root.getPuaGanglinien();
		final TwoDimensionalEMFMap<AggregationsTyp, Column, LineProperties> m = puaGanglinien.getMatrixValue();
		final Map<Column, Double> precisionMapValue = puaGanglinien.getPrecisionMapValue();
		// Für jede Spalte
		for (int i = 0; i < adminData.columns.length; i++) {
			final Column column = adminData.columns[i];
			Double range = precisionMapValue.get(column);
			if (null == range) {
				range = 1.0d;
			}
			final AggregationsTyp aggTyp = new AggregationsTyp();
			aggTyp.setKind(resultData.getOrder());
			aggTyp.setName(resultData.getDataName());
			final LineProperties lp = m.get(aggTyp, column);
			final NavigableMap<Long, Double> values = lp.getValues();

			// Sonderbehandlung für Intervalle
			if (ApplyAggregation.INTERVALL_SPALTE.equals(aggTyp.getKind())
					|| ApplyAggregation.INTERVALL_OBJEKT.equals(aggTyp.getKind())) {
				final List<Tuple<Long, Long>> periodList = root.getProtokoll().getProcessingParameterValue()
						.getPeriodList();
				boolean next = values.isEmpty();
				// finde das Intervall und setze den Timestamp auf das Ende des
				// Intervalls
				if (next) {
					// das erste Intervall
					if (!periodList.isEmpty()) {
						timeStamp = periodList.get(0).last;
					}
				} else {
					final Long lastKey = values.lastKey();
					for (final Tuple<Long, Long> tuple : periodList) {
						// gefunden!
						if (next) {
							timeStamp = tuple.last;
							break;
						}
						// das letzte Intervall ...
						if (tuple.last.equals(lastKey)) {
							next = true;
						}
					}
				}
			}
			// Setzen der Daten
			final ProtocolData resultValue = resultData.getData(dav.getDataModel(), i, true);
			if ((resultValue != null) && resultValue.isDefined() && resultValue.isPlain()) {
				try {
					double value = 0L;
					// if (aggTyp.getKind() == ApplyAggregation.LISTE) {
					if (resultValue.getAttributeType() instanceof DoubleAttributeType) {
						value = resultValue.asScaledValue().doubleValue();
					} else if (resultValue.getAttributeType() instanceof IntegerAttributeType) {
						final IntegerValueRange intRange = ((IntegerAttributeType) resultValue.getAttributeType())
								.getRange();
						if (intRange == null) {
							final List<IntegerValueState> states = ((IntegerAttributeType) resultValue
									.getAttributeType()).getStates();
							if (states == null) {
								value = resultValue.asUnscaledValue().intValue();
							} else {
								value = resultValue.asUnscaledValue().intValue() + 1;
							}
						} else {
							value = resultValue.asScaledValue().doubleValue();
						}
					} else {
						value = resultValue.asUnscaledValue().intValue();
					}
					// } else {
					// value = resultValue.asScaledValue().doubleValue();
					// }
					values.put(timeStamp, value);
				} catch (final ArithmeticException e) {
					Debug.getLogger().finer(e.getLocalizedMessage());
				}
				continue;
			}
			// bereits vorhandene Daten dürfen nicht zurückgesetzt werden
			if (!values.containsKey(timeStamp)) {
				values.put(timeStamp, null);
			}
		}
	}

	/**
	 * setzt den zu verarbeitenden Datensatz.
	 *
	 * @param resultData der Datensatz
	 */
	void setResultData(final RohDatenSatz resultData) {
		typ = new AggregationsTyp();
		typ.setKind(resultData.getOrder());
		typ.setName(resultData.getDataName());
		resultDataMap.put(typ, resultData);
	}
}
