/*
 * Rahmenwerk-Plugin fuer den Zugriff auf die Umfassende Datenanalyse (UDA)
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.uda.views;

import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.part.ViewPart;

import de.bsvrz.buv.plugin.uda.UdaToolsPlugin;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.MissingParameterException;
import de.bsvrz.ibv.uda.client.ClientModul;
import de.bsvrz.ibv.uda.client.ClientSkript;
import de.bsvrz.ibv.uda.client.ClientSkriptLauf;
import de.bsvrz.ibv.uda.client.ModulListener;
import de.bsvrz.ibv.uda.client.SkriptLaufListener;
import de.bsvrz.ibv.uda.client.SkriptListener;
import de.bsvrz.ibv.uda.client.UdaClient;
import de.bsvrz.ibv.uda.client.UdaConnectionListener;

/**
 * SkriptLaufUebersicht.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 *
 *
 */
public class SkriptLaufUebersicht extends ViewPart
		implements UdaConnectionListener, ModulListener, SkriptLaufListener, SkriptListener {

	/** View-ID. */
	public static final String ID = SkriptLaufUebersicht.class.getName();

	/** die Context-Hilfe-ID. */
	private static final String HILFE_ID = UdaToolsPlugin.PLUGIN_ID + "." + SkriptLaufUebersicht.class.getSimpleName();

	/** Skriptlauftabelle. */
	private TableViewer viewer;

	/**
	 * Verbindung zum UDA-Client.
	 */
	private UdaClient client;

	/**
	 * Standardkonstruktor.
	 */
	public SkriptLaufUebersicht() {
		super();
		try {
			UdaToolsPlugin.getDefault().getConnector().addConnectionListener(this);
		} catch (final MissingParameterException e) {
			e.printStackTrace();
		}
	}

	/**
	 * Aktualisiert diese Ansicht.
	 */
	private void aktualisieren() {
		Display.getDefault().asyncExec(new Runnable() {
			@Override
			public void run() {
				if (!viewer.getControl().isDisposed()) {
					viewer.refresh();
				}
			}
		});
	}

	/**
	 * Stellt eine Verbindung zum UDA-Modul her.
	 */
	@Override
	public void connected() {
		for (final ClientModul modul : client.getModule()) {
			modul.addClientListener(this);
			modul.addLaufListener(this);
		}
		aktualisieren();
	}

	@Override
	public void connectionClosed(final ClientDavInterface connection) {
		aktualisieren();
	}

	/**
	 * Erzeugt ein Kontextmenü für dieses Ansicht.
	 */
	private void createContextMenu() {
		final MenuManager menuManager = new MenuManager();
		menuManager.add(new Separator(IWorkbenchActionConstants.MB_ADDITIONS));
		viewer.getControl().setMenu(menuManager.createContextMenu(viewer.getControl()));
		getSite().registerContextMenu(SkriptLaufUebersicht.ID + ".contextMenu", menuManager, viewer);
	}

	@Override
	public void createPartControl(final Composite parent) {
		final GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 1;
		final Composite basisComposite = new Composite(parent, SWT.NONE);
		basisComposite.setLayout(gridLayout);

		viewer = new TableViewer(basisComposite, SWT.H_SCROLL | SWT.V_SCROLL | SWT.BORDER);

		final Table table = viewer.getTable();
		table.setLinesVisible(true);
		table.setHeaderVisible(true);

		final GridData gridData1 = new GridData();
		gridData1.grabExcessHorizontalSpace = true;
		gridData1.verticalAlignment = GridData.FILL;
		gridData1.grabExcessVerticalSpace = true;
		gridData1.horizontalAlignment = GridData.FILL;
		table.setLayoutData(gridData1);

		final SkriptLaufOverViewProvider provider = new SkriptLaufOverViewProvider();
		final String[] columnNames = provider.getColumnNames();
		final TableLayout layout = new TableLayout();
		table.setLayout(layout);
		for (final String columnName : columnNames) {
			final TableColumn column = new TableColumn(table, SWT.NORMAL);
			column.setText(columnName);
			layout.addColumnData(new ColumnWeightData(100, 120));
		}

		viewer.setContentProvider(provider);
		viewer.setLabelProvider(provider);

		try {
			client = UdaToolsPlugin.getDefault().getConnector();
		} catch (final MissingParameterException e) {
			throw new RuntimeException(e);
		}
		viewer.setInput(client);
		for (final ClientModul modul : client.getModule()) {
			modul.addClientListener(this);
			modul.addLaufListener(this);
		}

		createContextMenu();

		PlatformUI.getWorkbench().getHelpSystem().setHelp(parent, SkriptLaufUebersicht.HILFE_ID);
	}

	@Override
	public void dispose() {
		if (client != null) {
			for (final ClientModul modul : client.getModule()) {
				modul.removeClientListener(this);
				modul.removeLaufListener(this);
			}
		}

		try {
			UdaToolsPlugin.getDefault().getConnector().removeConnectionListener(this);
		} catch (final MissingParameterException e) {
			e.printStackTrace();
		}

		super.dispose();
	}

	@Override
	public void ergebnisAktualisiert(final ClientSkriptLauf lauf) {
		aktualisieren();
	}

	@Override
	public void meldungsListeAktualisiert(final ClientSkriptLauf skript) {
		//
	}

	@Override
	public void neuerModulStatus(final ClientModul modul, final boolean valid) {
		aktualisieren();
	}

	@Override
	public void skriptAktualisiert(final ClientSkript skript) {
		// wird hier nicht angezeigt.
	}

	@Override
	public void skriptAktualisiert(final ClientSkriptLauf skript) {
		aktualisieren();
	}

	@Override
	public void skriptAngelegt(final ClientModul modul, final ClientSkript skript) {
		aktualisieren();
	}

	@Override
	public void skriptEntfernt(final ClientModul modul, final ClientSkript skript) {
		aktualisieren();

	}

	@Override
	public void skriptLaufAngelegt(final ClientModul modul, final ClientSkriptLauf lauf) {
		aktualisieren();
	}

	@Override
	public void skriptLaufEntfernt(final ClientModul modul, final ClientSkriptLauf lauf) {
		aktualisieren();
	}

	@Override
	public void skriptZustandAktualisiert(final ClientSkriptLauf skript) {
		aktualisieren();
	}

	@Override
	public void setFocus() {
		if ((viewer != null) && (!viewer.getControl().isDisposed())) {
			viewer.getControl().setFocus();
		}
	}
}
