/*
 * Rahmenwerk-Plugin fuer den Zugriff auf die Umfassende Datenanalyse (UDA)
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.uda.wizards;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.ILabelProviderListener;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.window.Window;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Canvas;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;

import de.bsvrz.ibv.uda.verwaltung.protokoll.Protokoll;

/**
 * DAV-Protokolltabelle.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 */
public class ProtokollDefinitionTable extends Canvas
		implements IStructuredContentProvider, ITableLabelProvider, ISelectionChangedListener, DisposeListener {

	/** ausgewähltes DAV-Protokolle. */
	private Protokoll selectedProtokoll;

	/** Menge der DAV-Protokolle. */
	private final List<Protokoll> protokolle = new ArrayList<>();

	/**
	 * Tabellenviewer fuer Protokollmenge.
	 */
	private final TableViewer viewer;

	/**
	 * Spaltennamen.
	 */
	private final String[] columnNames = new String[] { "Typ", "Name", "Anzahl", "Größe" };

	/** Schaltfläche zum Bearbeiten eines Protokolleintrags. */
	private Button editButton;

	/** Schaltfläche zum Entfernen eines Protokolleintrags. */
	private Button deleteButton;

	/**
	 * Standardkonstruktor.
	 *
	 * @param parent
	 *            uebergeordnetes Composite.
	 * @param initialeProtokolle
	 *            Menge der Protokolle, die bereits angelegt sind.
	 */
	public ProtokollDefinitionTable(final Composite parent, final Collection<Protokoll> initialeProtokolle) {
		super(parent, SWT.NONE);
		setLayout(new GridLayout(2, false));

		if (initialeProtokolle != null) {
			protokolle.addAll(initialeProtokolle);
		}

		createButtonArea();

		viewer = new TableViewer(this, SWT.MULTI | SWT.FULL_SELECTION | SWT.H_SCROLL | SWT.V_SCROLL | SWT.BORDER);
		final Table table = viewer.getTable();

		viewer.addSelectionChangedListener(this);

		table.setLayoutData(new GridData(GridData.FILL_BOTH | GridData.GRAB_HORIZONTAL | GridData.GRAB_VERTICAL));
		table.setHeaderVisible(true);
		table.setLinesVisible(true);

		final TableLayout layout = new TableLayout();
		TableColumn column = null;

		for (final String columnName : columnNames) {
			column = new TableColumn(table, SWT.NORMAL);
			column.setText(columnName);
			layout.addColumnData(new ColumnWeightData(100));
		}

		table.setLayout(layout);

		viewer.setContentProvider(this);
		viewer.setLabelProvider(this);
		viewer.setInput(this);

		addDisposeListener(this);
	}

	@Override
	public void addListener(final ILabelProviderListener listener) {
		//
	}

	/**
	 * Erstellt die Schaltflaechen zur Manipulation der Tabelle.
	 */
	private void createButtonArea() {
		final Canvas buttons = new Canvas(this, SWT.NONE);
		buttons.setLayout(new GridLayout(1, true));

		final Button button = new Button(buttons, SWT.NONE);
		button.setText("Anlegen");
		button.addSelectionListener(new SelectionListener() {
			@Override
			public void widgetDefaultSelected(final SelectionEvent e) {
				final CreateProtokollWizard wizard = new CreateProtokollWizard(protokolle, null);
				final WizardDialog dialog = new WizardDialog(getShell(), wizard);
				if (dialog.open() == Window.OK) {
					viewer.refresh();
				}
			}

			@Override
			public void widgetSelected(final SelectionEvent e) {
				widgetDefaultSelected(e);
			}
		});

		editButton = new Button(buttons, SWT.NONE);
		editButton.setText("Bearbeiten");
		editButton.addSelectionListener(new SelectionListener() {
			@Override
			public void widgetDefaultSelected(final SelectionEvent e) {
				final CreateProtokollWizard wizard = new CreateProtokollWizard(protokolle, selectedProtokoll);
				final WizardDialog dialog = new WizardDialog(getShell(), wizard);
				if (dialog.open() == Window.OK) {
					viewer.refresh();
				}
			}

			@Override
			public void widgetSelected(final SelectionEvent e) {
				widgetDefaultSelected(e);
			}
		});
		editButton.setEnabled(false);

		deleteButton = new Button(buttons, SWT.NONE);
		deleteButton.setText("Löschen");
		deleteButton.addSelectionListener(new SelectionListener() {
			@Override
			public void widgetDefaultSelected(final SelectionEvent e) {
				final Iterator<?> iterator = ((IStructuredSelection) viewer.getSelection()).iterator();
				while (iterator.hasNext()) {
					protokolle.remove(iterator.next());
				}
				viewer.refresh();
			}

			@Override
			public void widgetSelected(final SelectionEvent e) {
				widgetDefaultSelected(e);
			}
		});
		deleteButton.setEnabled(false);
		button.pack();
	}

	@Override
	public void widgetDisposed(final DisposeEvent e) {
		protokolle.clear();
	}

	@Override
	public Image getColumnImage(final Object element, final int columnIndex) {
		return null;
	}

	@Override
	public String getColumnText(final Object element, final int columnIndex) {
		String result = "";
		final Protokoll protokoll = (Protokoll) element;
		switch (columnIndex) {
		case 0:
			result = protokoll.getTyp().toString();
			break;
		case 1:
			result = protokoll.getName();
			break;
		case 2:
			result = Long.toString(protokoll.getAnzahl());
			break;
		case 3:
			result = Long.toString(protokoll.getMaxSize());
			break;
		default:
			break;
		}
		return result;
	}

	@Override
	public Object[] getElements(final Object inputElement) {
		return protokolle.toArray();
	}

	/**
	 * Erfragt die Menge der Protokolle, die innerhalb der Tabelle stehen.
	 *
	 * @return die Menge der Protokolle, die innerhalb der Tabelle stehen.
	 */
	public Collection<Protokoll> getProtokolle() {
		return protokolle;
	}

	@Override
	public void inputChanged(final Viewer neuerViewer, final Object oldInput, final Object newInput) {
		//
	}

	@Override
	public boolean isLabelProperty(final Object element, final String property) {
		return false;
	}

	@Override
	public void removeListener(final ILabelProviderListener listener) {
		//
	}

	@Override
	public void selectionChanged(final SelectionChangedEvent event) {
		selectedProtokoll = null;
		deleteButton.setEnabled(false);
		editButton.setEnabled(false);

		if (event.getSelection() instanceof IStructuredSelection) {
			final Object selected = ((IStructuredSelection) event.getSelection()).getFirstElement();
			if (selected instanceof Protokoll) {
				selectedProtokoll = (Protokoll) selected;
				deleteButton.setEnabled(true);
				editButton.setEnabled(true);
			}
		}
	}
}
