/*
 * Rahmenwerk-Plugin fuer den Zugriff auf die Umfassende Datenanalyse (UDA)
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.uda.wizards;

import java.text.DateFormat;
import java.text.ParseException;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.jface.viewers.ComboViewer;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.ViewerComparator;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

import de.bsvrz.ibv.uda.client.ClientException;
import de.bsvrz.ibv.uda.client.ClientModul;
import de.bsvrz.ibv.uda.client.ClientSkript;
import de.bsvrz.ibv.uda.client.SkriptLaufDaten;
import de.bsvrz.ibv.uda.uda.data.SkriptLauf;
import de.bsvrz.ibv.uda.uda.data.StartTyp;
import de.bsvrz.ibv.uda.uda.data.StartZyklus;
import de.bsvrz.ibv.uda.verwaltung.protokoll.Protokoll;

/**
 * Skriptlaufdatenseite.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 *
 *
 */
public class SkriptLaufDataPage extends WizardPage {

	/**
	 * Regex fuer den Namen eines Parameters mit fuehrendem Minus und nachfolgendem
	 * Gleichheitszeichen: z.B. -abc=, -A2=, -a=, etc.
	 */
	private static final String REGEX_MINUS_NAME_GLEICH = "-{1}[a-zA-ZÄÖÜäöü]{1}[a-zA-Z0-9ÄÖÜäöüß]*={1}";

	/**
	 * Regex fuer den Wert eines Paramaters: z.B. "Hallo", "asdjh k", asjda,
	 * "-p1=Hallo", etc.
	 */
	private static final String REGEX_WERT = "((\".+?\")|([a-zA-Z0-9ÄÖÜäöüß]+))";

	/**
	 * Compiliertes Regex fuer <code>REGEX_MINUS_NAME_GLEICH + REGEX_WERT</code> .
	 */
	private static final Pattern PATTERN_PARAMETERSATZ = Pattern
			.compile(SkriptLaufDataPage.REGEX_MINUS_NAME_GLEICH + SkriptLaufDataPage.REGEX_WERT);

	/**
	 * Compiliertes Regex fuer <code>REGEX_MINUS_NAME_GLEICH</code>.
	 */
	private static final Pattern PATTER_NAME = Pattern.compile(SkriptLaufDataPage.REGEX_MINUS_NAME_GLEICH);

	/** Feld fuer Name des Skriptlaufs. */
	private Text nameField;

	/** Feld fuer Beschreibung. */
	private Text descField;

	/** Feld fuer Nutzernamen. */
	private Text userField;

	/** Passwortfeld. */
	private Text passwortField;

	/** Feld fuer Skriptselektor. */
	private ComboViewer skriptSelector;

	/** Das Element. */
	private final Object element;

	/** Feld fuer Name des Nutzers. */
	private Text editUserField;

	/** Feld fuer Name des Grundes. */
	private Text grundField;

	/** Auswahlfeld für Startart. */
	private ComboViewer startTypSelector;

	/** Protokoll. */
	private ProtokollDefinitionTable protokollListe;

	/** Feld fuer Startzeitpunkt. */
	private Text startZeitpunktField;

	/** Feld fuer allgemeine Datenrepräsentation eines Skriptlaufes. */
	private final SkriptLaufDaten skriptLaufDaten;

	/** Feld fuer Intervallauswahl. */
	private Text intervalField;

	/** Feld fuer Parameter. */
	private Text parameterField;

	/**
	 * Standardkonstruktor.
	 *
	 * @param modul
	 *            Verbindung zum UDA-Modul.
	 * @param lauf
	 *            der Skriptlauf.
	 */
	protected SkriptLaufDataPage(final ClientModul modul, final SkriptLauf lauf) {
		super("Skriptlaufdaten");
		element = modul;
		setTitle("Skriptlaufinformationen");
		setMessage("Geben Sie hier die Informationen zur Definition des Skriptlaufs an");

		skriptLaufDaten = new SkriptLaufDaten(lauf);
	}

	@Override
	public void createControl(final Composite parent) {
		final GridLayout pageLayout = new GridLayout();
		pageLayout.numColumns = 2;
		parent.setLayout(pageLayout);
		parent.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		Label label = new Label(parent, SWT.NONE);
		label.setText("Skript: ");
		skriptSelector = new ComboViewer(parent, SWT.NONE);
		skriptSelector.setContentProvider(new UdaSkriptProvider());
		skriptSelector.setComparator(new ViewerComparator());
		skriptSelector.setInput(element);
		skriptSelector.getCombo().setLayoutData(new GridData(GridData.FILL_BOTH | GridData.GRAB_HORIZONTAL));
		if (skriptLaufDaten.getSkript() != null) {
			skriptSelector.setSelection(new StructuredSelection(skriptLaufDaten.getSkript()));
		} else {
			skriptSelector.setSelection(null);
		}

		label = new Label(parent, SWT.NONE);
		label.setText("Skriptlaufname: ");
		nameField = new Text(parent, SWT.BORDER);
		nameField.setLayoutData(new GridData(GridData.FILL_HORIZONTAL | GridData.GRAB_HORIZONTAL));
		nameField.setText(skriptLaufDaten.getName());

		label = new Label(parent, SWT.NONE);
		label.setText("Beschreibung: ");
		descField = new Text(parent, SWT.BORDER);
		descField.setLayoutData(new GridData(GridData.FILL_BOTH | GridData.GRAB_HORIZONTAL));
		descField.setText(skriptLaufDaten.getBeschreibung());

		label = new Label(parent, SWT.NONE);
		label.setText("Parameter: ");
		parameterField = new Text(parent, SWT.BORDER);
		parameterField.setLayoutData(new GridData(GridData.FILL_BOTH | GridData.GRAB_HORIZONTAL));
		parameterField.setText(getParameterString(skriptLaufDaten.getParameter()));

		label = new Label(parent, SWT.NONE);
		label.setText("Benutzer: ");
		userField = new Text(parent, SWT.BORDER);
		userField.setLayoutData(new GridData(GridData.FILL_BOTH | GridData.GRAB_HORIZONTAL));
		userField.setText(skriptLaufDaten.getBenutzer());

		label = new Label(parent, SWT.NONE);
		label.setText("Passwort: ");
		passwortField = new Text(parent, SWT.BORDER | SWT.PASSWORD);
		passwortField.setLayoutData(new GridData(GridData.FILL_BOTH | GridData.GRAB_HORIZONTAL));
		passwortField.setText("");

		label = new Label(parent, SWT.NONE);
		label.setText("Protokolle: ");
		new Label(parent, SWT.NONE);

		protokollListe = new ProtokollDefinitionTable(parent, skriptLaufDaten.getProtokolle());
		final GridData data = new GridData(GridData.FILL_BOTH | GridData.GRAB_HORIZONTAL);
		data.heightHint = 150;
		data.horizontalSpan = 2;
		protokollListe.setLayoutData(data);

		label = new Label(parent, SWT.NONE);
		label.setText("Starttyp: ");
		startTypSelector = new ComboViewer(parent, SWT.NONE);
		startTypSelector.setContentProvider(new StartTypProvider());
		startTypSelector.setInput(element);
		startTypSelector.getCombo().setLayoutData(new GridData(GridData.FILL_BOTH | GridData.GRAB_HORIZONTAL));
		startTypSelector.setSelection(new StructuredSelection(skriptLaufDaten.getStartTyp()));

		label = new Label(parent, SWT.NONE);
		label.setText("Startzeitpunkt: ");
		startZeitpunktField = new Text(parent, SWT.NONE);
		startZeitpunktField.setLayoutData(new GridData(GridData.FILL_BOTH | GridData.GRAB_HORIZONTAL));
		startZeitpunktField.setText(
				DateFormat.getDateTimeInstance().format(new Date(skriptLaufDaten.getZyklus().getStartZeitPunkt())));

		label = new Label(parent, SWT.NONE);
		label.setText("Intervall (in ms): ");
		intervalField = new Text(parent, SWT.NONE);
		intervalField.setLayoutData(new GridData(GridData.FILL_BOTH | GridData.GRAB_HORIZONTAL));
		intervalField.setText(Long.toString(skriptLaufDaten.getZyklus().getIntervall()));

		label = new Label(parent, SWT.NONE);
		label.setText("Name des ändernden Benutzers: ");
		editUserField = new Text(parent, SWT.BORDER);
		editUserField.setLayoutData(new GridData(GridData.FILL_BOTH | GridData.GRAB_HORIZONTAL));
		editUserField.setText(System.getProperty("user.name"));

		label = new Label(parent, SWT.NONE);
		label.setText("Grund für die Änderung: ");
		grundField = new Text(parent, SWT.BORDER);
		grundField.setLayoutData(new GridData(GridData.FILL_BOTH | GridData.GRAB_HORIZONTAL));

		setControl(parent);
	}

	/**
	 * Erfragt den Benutzernamen.
	 *
	 * @return den Benutzernamen.
	 */
	public String getEditBenutzerName() {
		return editUserField.getText();
	}

	/**
	 * Erfragt den Grund.
	 *
	 * @return den Grund.
	 */
	public String getGrund() {
		return grundField.getText();
	}

	// /**
	// * Erffragt die Laufparamter.
	// *
	// * @return die Laufparamter.
	// */
	// private Map<String, String> getLaufParameter() {
	// Map<String, String> result = new HashMap<String, String>();
	// StringTokenizer tokenizer = new StringTokenizer(parameterField
	// .getText(), "|");
	// while (tokenizer.hasMoreTokens()) {
	// StringTokenizer parameterTokenizer = new StringTokenizer(tokenizer
	// .nextToken(), "=");
	// String name = null;
	// String wert = null;
	// if (parameterTokenizer.hasMoreTokens()) {
	// name = parameterTokenizer.nextToken().trim();
	// }
	// if (parameterTokenizer.hasMoreTokens()) {
	// wert = parameterTokenizer.nextToken().trim();
	// if (wert.startsWith("\"")) {
	// wert = wert.substring(1);
	// }
	// if (wert.endsWith("\"")) {
	// wert = wert.substring(0, wert.length() - 1);
	// }
	// }
	// if (name != null) {
	// result.put(name, wert);
	// }
	// }
	//
	// return result;
	// }

	/**
	 * Erfragt die Laufparamter.
	 *
	 * @return die Laufparamter.
	 */
	private Map<String, String> getLaufParameter() {
		final Map<String, String> result = new HashMap<>();

		final Matcher matcher = SkriptLaufDataPage.PATTERN_PARAMETERSATZ.matcher(parameterField.getText());

		while (matcher.find()) {
			final String parameterSatz = parameterField.getText().substring(matcher.start(), matcher.end());
			final Matcher nameRawMatcher = SkriptLaufDataPage.PATTER_NAME.matcher(parameterSatz);
			nameRawMatcher.find();
			final String name = parameterSatz.substring(nameRawMatcher.start() + 1, nameRawMatcher.end() - 1);
			final String wertRaw = parameterSatz.substring(nameRawMatcher.end());
			String wert = wertRaw;
			if (wertRaw.startsWith("\"")) {
				wert = wertRaw.substring(1, wertRaw.length() - 1);
			}
			result.put(name, wert);
		}

		return result;
	}

	/**
	 * Extrahiert die Kommandozeilen-Parameter des Skriptes.
	 *
	 * @param liste
	 *            Liste mit Skript-Paramtern.
	 * @return die Kommandozeilen-Parameter des Skriptes
	 */
	private String getParameterString(final Map<String, String> liste) {
		final StringBuffer buffer = new StringBuffer();
		for (final Entry<String, String> entry : liste.entrySet()) {
			if (buffer.length() > 0) {
				buffer.append('|');
			}
			buffer.append(entry.getKey());
			final String wert = entry.getValue();
			if (wert != null) {
				buffer.append('=');
				buffer.append(wert);
			}
		}
		return buffer.toString();
	}

	/**
	 * Erfragt den Skriptlauf.
	 *
	 * @return den Skriptlauf.
	 * @throws ClientException
	 *             bei Fehler innerhalb des Clientzugriffs auf die "Umfassende
	 *             Datenanalyse".
	 */
	public SkriptLaufDaten getSkriptLauf() throws ClientException {
		skriptLaufDaten.setBenutzer(userField.getText());
		skriptLaufDaten.setBeschreibung(descField.getText());
		final String anmeldeName = nameField.getText();
		if ((anmeldeName == null) || (anmeldeName.length() <= 0)) {
			throw new ClientException("Es muss ein Name für  die Datenverteileranmeldung angegeben werden");
		}
		skriptLaufDaten.setName(anmeldeName);

		final String passwd = passwortField.getText().trim();
		if (passwd.isEmpty()) {
			throw new ClientException("Es ist ein Passwort für die Dartenverteileranmeldung erforderlich");
		}
		skriptLaufDaten.setPasswd(passwd);

		skriptLaufDaten
				.setSkript((ClientSkript) ((IStructuredSelection) skriptSelector.getSelection()).getFirstElement());
		skriptLaufDaten
				.setStartTyp((StartTyp) ((IStructuredSelection) startTypSelector.getSelection()).getFirstElement());

		Date date;
		try {
			date = DateFormat.getDateTimeInstance().parse(startZeitpunktField.getText());
		} catch (final ParseException e) {
			startZeitpunktField.setText(DateFormat.getDateInstance().format(new Date(System.currentTimeMillis())));
			throw new ClientException("Das eingegebene Startdatum ist ungültig: " + e.getLocalizedMessage());
		}

		try {
			skriptLaufDaten.setZyklus(new StartZyklus(date.getTime(), Long.parseLong(intervalField.getText())));
		} catch (final NumberFormatException e) {
			throw new ClientException("Der eingegebene Ausführungszyklus ist ungültig: " + e.getLocalizedMessage());
		}

		skriptLaufDaten.clearProtokolle();
		for (final Protokoll protokoll : protokollListe.getProtokolle()) {
			skriptLaufDaten.addProtokoll(protokoll);
		}

		skriptLaufDaten.clearParameter();
		final Map<String, String> parameterListe = getLaufParameter();
		for (final Entry<String, String> entry : parameterListe.entrySet()) {
			skriptLaufDaten.setParameter(entry.getKey(), entry.getValue());
		}

		return skriptLaufDaten;
	}

}
