/*
 * Rahmenwerk-Plug-in "BitCtrl-Bibliotheken"
 *
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.rw.bitctrl.eclipse.databinding;

import org.eclipse.core.databinding.observable.value.ComputedValue;
import org.eclipse.core.databinding.observable.value.IObservableValue;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.progress.UIJob;

import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.attribute.AttRdsTMCRichtung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcglobal.attribute.AttTmcRichtung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.Strasse;
import de.bsvrz.sys.funclib.bitctrl.modell.util.cache.NetzCache;

/**
 * ComputedValue, welcher zur Bestimmung des Textes für ein
 * Richtungs-RadioButton.
 *
 * @author BitCtrl Systems GmbH, Andreas Meissner
 *
 */
public class RichtungTextComputedValue extends ComputedValue {

	/**
	 * Der Text, welcher Standard-mäßig bei positiver Straßenrichtung verwendet
	 * wird.
	 */
	public static final String DEFAULT_TEXT_NEGATIVE_RICHTUNG = "negativ (Gegenrichtung)";

	/**
	 * Der Text, welcher Standard-mäßig bei negativer Straßenrichtung verwendet
	 * wird.
	 */
	public static final String DEFAULT_TEXT_POSITIVE_RICHTUNG = "positiv (Fahrtrichtung)";

	/** Das ObservableValue für das selektierte Straßenobjekt. */
	private final IObservableValue strasseObservable;

	/** Die Richtung. */
	private final AttTmcRichtung richtung;

	/** Der Default-Text. */
	private final String defaultText;

	/** Das Display, in welchem der UIJob gestartet werden kann. */
	private final Display display;

	/**
	 * Das optionale Composite, dessen Layout nach Setzen des neuen
	 * Richtungs-Textes aktualisiert werden soll.
	 */
	private final Composite composite;

	/** Der Netz-Cache. */
	private final NetzCache netzCache;

	/**
	 * Der Konstruktor.
	 *
	 * @param netzCache
	 *            der Netz-Cache, darf nicht <code>null</code> sein
	 * @param strasseObservable
	 *            das ObservableValue für das selektierte Straßenobjekt, darf
	 *            nicht <code>null</code> sein
	 * @param richtung
	 *            die Richtung, darf nicht <code>null</code> sein
	 * @param composite
	 *            das optionale Composite, dessen Layout nach Setzen des neuen
	 *            Richtungs-Textes aktualisiert werden soll
	 * @param display
	 *            das Display, in welchem ein UIJob zum Aktualisieren des
	 *            Layouts eines Composites gestartet werden kann
	 */
	public RichtungTextComputedValue(final NetzCache netzCache,
			final IObservableValue strasseObservable,
			final AttTmcRichtung richtung, final Composite composite,
			final Display display) {

		assert netzCache != null;
		assert strasseObservable != null;
		assert richtung != null;

		this.display = display;
		this.composite = composite;
		this.netzCache = netzCache;
		this.strasseObservable = strasseObservable;
		this.richtung = richtung;
		defaultText = (richtung == AttTmcRichtung.ZUSTAND_1_POSITIV)
				? DEFAULT_TEXT_POSITIVE_RICHTUNG
				: DEFAULT_TEXT_NEGATIVE_RICHTUNG;
	}

	/**
	 * Der Konstruktor.
	 *
	 * @param netzCache
	 *            der Netz-Cache, darf nicht <code>null</code> sein
	 * @param strasseObservable
	 *            das ObservableValue für das selektierte Straßenobjekt, darf
	 *            nicht <code>null</code> sein
	 * @param richtung
	 *            die Richtung, darf nicht <code>null</code> sein
	 * @param composite
	 *            das optionale Composite, dessen Layout nach Setzen des neuen
	 *            Richtungs-Textes aktualisiert werden soll
	 * @param display
	 *            das Display, in welchem ein UIJob zum Aktualisieren des
	 *            Layouts eines Composites gestartet werden kann
	 */
	public RichtungTextComputedValue(final NetzCache netzCache,
			final IObservableValue strasseObservable,
			final AttRdsTMCRichtung richtung, final Composite composite,
			final Display display) {
		this(netzCache, strasseObservable,
				richtung == AttRdsTMCRichtung.ZUSTAND_0_POSITIV
						? AttTmcRichtung.ZUSTAND_1_POSITIV
						: AttTmcRichtung.ZUSTAND_1N_NEGATIV,
				composite, display);
	}

	@Override
	protected Object calculate() {
		final Strasse strasse = (Strasse) strasseObservable.getValue();
		final String richtungText = getRichtungText(netzCache, strasse,
				richtung);
		if (composite != null && display != null) {
			final UIJob job = new UIJob(display, "Aktualisiere Radio-Buttons") {
				@Override
				public IStatus runInUIThread(final IProgressMonitor monitor) {
					composite.layout(true, true);
					return Status.OK_STATUS;
				}
			};
			job.schedule(1);
		}
		if (richtungText != null) {
			return richtungText;
		}
		return defaultText;
	}

	/**
	 * Liefert den Richtungs-Text für eine gegebene Straßenrichtung zurück.
	 *
	 * @param netzCache
	 *            der Netz-Cache
	 * @param strasse
	 *            die Straße
	 * @param richtung
	 *            die Richtung
	 *
	 * @return den Richtungs-Text, kann <code>null</code> sein
	 */
	public static String getRichtungText(final NetzCache netzCache,
			final Strasse strasse, final AttTmcRichtung richtung) {
		final String vonStrassenKnoten = netzCache.getVonStrassenKnoten(strasse,
				richtung);
		final String nachStrassenKnoten = netzCache
				.getNachStrassenKnoten(strasse, richtung);
		if (vonStrassenKnoten != null && nachStrassenKnoten != null) {
			return new StringBuffer("von ").append(vonStrassenKnoten)
					.append(" nach ").append(nachStrassenKnoten).toString();
		}
		return null;
	}

	/**
	 * Liefert den Richtungs-Text für eine gegebene Straßenrichtung zurück.
	 *
	 * @param netzCache
	 *            der Netz-Cache
	 * @param strasse
	 *            die Straße
	 * @param richtung
	 *            die Richtung
	 *
	 * @return den Richtungs-Text, kann <code>null</code> sein
	 */
	public static String getRichtungText(final NetzCache netzCache,
			final Strasse strasse, final AttRdsTMCRichtung richtung) {
		return getRichtungText(netzCache, strasse,
				richtung == AttRdsTMCRichtung.ZUSTAND_0_POSITIV
						? AttTmcRichtung.ZUSTAND_1_POSITIV
						: AttTmcRichtung.ZUSTAND_1N_NEGATIV);
	}
}
