/*
 * Rahmenwerk-Plug-in "BitCtrl-Bibliotheken"
 *
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.rw.bitctrl.eclipse.databinding;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;
import java.util.ResourceBundle;

import org.eclipse.core.databinding.DataBindingContext;
import org.eclipse.core.databinding.UpdateValueStrategy;
import org.eclipse.core.databinding.observable.Observables;
import org.eclipse.core.databinding.observable.list.IObservableList;
import org.eclipse.core.databinding.observable.set.ListToSetAdapter;
import org.eclipse.core.databinding.observable.value.ComputedValue;
import org.eclipse.core.databinding.observable.value.IObservableValue;
import org.eclipse.core.databinding.observable.value.IValueChangeListener;
import org.eclipse.core.databinding.observable.value.WritableValue;
import org.eclipse.core.databinding.validation.IValidator;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.databinding.EMFObservables;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.jface.databinding.swt.typed.WidgetProperties;
import org.eclipse.jface.databinding.viewers.ObservableListContentProvider;
import org.eclipse.jface.databinding.viewers.ObservableMapLabelProvider;
import org.eclipse.jface.databinding.viewers.typed.ViewerProperties;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ViewerComparator;
import org.eclipse.nebula.jface.tablecomboviewer.TableComboViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.ui.forms.widgets.FormToolkit;

import com.bitctrl.lib.eclipse.databinding.observables.CollectionContainsValue;
import com.bitctrl.lib.eclipse.databinding.provider.MessagesLabelProvider;
import com.bitctrl.lib.eclipse.databinding.widgets.ObservableClearButton;
import com.bitctrl.messages.DynamicMessages;

import de.bsvrz.buv.rw.bitctrl.eclipse.widgets.AutoIncrementSpinner;
import de.bsvrz.buv.rw.bitctrl.eclipse.widgets.ColorPicker;
import net.randomice.emf.observables.runtime.IObservableObject;

/**
 * Ein Composite mit dynamisch erzeugten Dateneinstellungen in einer Zeile
 * angeordnet. für jedes Feature in einer {@link EClass} wird ein Control
 * erstellt. Die Konfiguration der Beschriftung erfolgt über ein
 * {@link ResourceBundle}.
 *
 * @author BitCtrl Systems GmbH, schnepel
 *
 */
public class SettingTable extends Composite {

	/**
	 * Es gibt kein spaltenweise füllbares GridLayout. Aus diesem Grund wird
	 * hier ein Runnable implementiert, das die zweite Zeile des Grids später
	 * erstellt.
	 *
	 * @author schnepel
	 */
	private final class DeferredSettingLineControlAdder implements Runnable {
		private final String featureKey;
		private final IObservableObject<? extends EObject> object;
		private final Label label;
		private final FormToolkit toolkit;
		private final IObservableList unknownMultiFeatures;
		private final String tooltip;
		private final EStructuralFeature feature;
		private final IObservableValue unknownModel;
		private final IObservableValue valueFeature;

		private DeferredSettingLineControlAdder(final String featureKey,
				final IObservableObject<? extends EObject> object,
				final Label label, final FormToolkit toolkit,
				final IObservableList unknownMultiFeatures,
				final String tooltip, final EStructuralFeature feature,
				final IObservableValue valueFeature,
				final IObservableValue unknownModel) {
			this.featureKey = featureKey;
			this.object = object;
			this.label = label;
			this.toolkit = toolkit;
			this.unknownMultiFeatures = unknownMultiFeatures;
			this.tooltip = tooltip;
			this.feature = feature;
			this.valueFeature = valueFeature;
			this.unknownModel = unknownModel;
		}

		@Override
		public void run() {
			final EClassifier type = feature.getEType();
			final Class<?> instanceClass = type.getInstanceClass();
			Control c = null;
			IObservableValue target = null;
			if (instanceClass.isEnum()) {
				final TableComboViewer v = new TableComboViewer(
						SettingTable.this, SWT.BORDER | SWT.READ_ONLY);
				v.setContentProvider(new ArrayContentProvider());
				try {
					final Field valuesField = instanceClass.getDeclaredField(
							Messages.SettingTable_EnumValuesField);
					final Object values = valuesField.get(null);
					v.setInput(((List<?>) values).toArray());
					target = ViewerProperties.singleSelection().observe(v);
					final ILabelProvider encapsulatedLabelProvider = new MessagesLabelProvider(
							dynamicMessages, featureKey + Messages.Underline);
					v.setLabelProvider(wrapComboLabelProvider(feature,
							encapsulatedLabelProvider));
					v.getTableCombo().setBackground(null);
					v.setComparator(
							new FirstElementsEnumViewerSorter(featureKey));
				} catch (final Exception e1) {
					e1.printStackTrace();
				}
				c = v.getControl();
			} else if (RGB.class.isAssignableFrom(instanceClass)) {
				final ColorPicker colorPicker = createRGBColorPicker(
						SettingTable.this, object, feature);
				target = colorPicker.getColor();
				c = colorPicker;
			} else if (Boolean.TYPE == instanceClass
					|| Boolean.class == instanceClass) {
				final Button b = toolkit.createButton(SettingTable.this,
						Messages.SettingTable_Yes_No, SWT.CHECK);
				target = WidgetProperties.buttonSelection().observe(b);
				c = b;
			} else if (instanceClass.isPrimitive()
					|| Number.class.isAssignableFrom(instanceClass)) {
				final AutoIncrementSpinner sp = new AutoIncrementSpinner(
						SettingTable.this, SWT.BORDER);
				target = sp.getObservableValue();
				bindMinOrMax(sp.observeMinimum(),
						Messages.SettingTable_MinimumPostfix);
				bindMinOrMax(sp.observeMaximum(),
						Messages.SettingTable_MaximumPostfix);

				c = sp;
			} else if (feature instanceof final EReference r) {
				if (r.isMany()) {
					c = toolkit.createLabel(SettingTable.this,
							"TODO:MultiValuedRef"); //$NON-NLS-1$
				} else {
					final TableComboViewer v = new TableComboViewer(
							SettingTable.this, SWT.BORDER | SWT.READ_ONLY);
					v.setContentProvider(new ObservableListContentProvider());
					final IObservableList listFeature = getListForReference(r);
					target = ViewerProperties.singleSelection().observe(v);
					final EStructuralFeature mapFeature = ((EClass) r
							.getEType()).getEStructuralFeature(
									dynamicMessages.getString(featureKey
											+ Messages.SettingTable_ListTitleFeaturePostfix));
					v.setLabelProvider(wrapComboLabelProvider(feature,
							new ObservableMapLabelProvider(
									EMFObservables.observeMap(
											new ListToSetAdapter(listFeature),
											mapFeature))));
					v.setInput(listFeature);
					v.setComparator(new ViewerComparator());
					c = v.getControl();
				}
			} else if (String.class.isAssignableFrom(instanceClass)) {
				c = toolkit.createText(SettingTable.this, "", SWT.NONE); //$NON-NLS-1$
				target = WidgetProperties.text(SWT.Modify).observe(c);
			}
			if (null == c) {
				c = toolkit.createLabel(SettingTable.this, ""); //$NON-NLS-1$
			}
			if (null != c) {
				final IValidator validator = getValidator(feature, c);
				if ((null != validator) && (target != null)) {
					final UpdateValueStrategy uvs = new UpdateValueStrategy()
							.setAfterConvertValidator(validator);
					dbc.bindValue(valueFeature,
							new WritableValue(target.getRealm()), uvs, uvs);
				}
				dbc.bindValue(target, valueFeature);
				c.setLayoutData(
						new GridData(SWT.FILL, SWT.CENTER, false, false, 2, 1));
				c.setToolTipText(tooltip);
				toolkit.adapt(c, true, true);
				c.setEnabled(false);
				ComputedValue enabledModel;
				if (null != unknownMultiFeatures) {
					enabledModel = new ComputedValue(
							object.getObservable().getRealm(), Boolean.TYPE) {
						@Override
						protected Boolean calculate() {
							return !((Boolean) unknownModel.getValue())
									&& object.getValue() != null
									&& isEnabled(feature);
						}
					};
					// bring UI into sync with model
					Display.getDefault().asyncExec(() -> {
						unknownModel.setValue(true);
						Display.getDefault()
								.asyncExec(() -> unknownModel.setValue(false));
					});
				} else {
					enabledModel = new ComputedValue(
							object.getObservable().getRealm(), Boolean.TYPE) {
						@Override
						protected Boolean calculate() {
							return object.getValue() != null;
						}
					};
				}
				dbc.bindValue(WidgetProperties.enabled().observe(label),
						enabledModel);
				dbc.bindValue(WidgetProperties.enabled().observe(c),
						enabledModel);
				if (null != valueRefreshListener) {
					valueFeature.addValueChangeListener(valueRefreshListener);
				}
			}
		}

		private void bindMinOrMax(final IObservableValue target,
				final String postfix) {
			final String minOrMaxString = dynamicMessages
					.getStringOrNull(featureKey + postfix);
			if (null != minOrMaxString) {
				IObservableValue model = null;
				try {
					model = Observables.constantObservableValue(
							Integer.parseInt(minOrMaxString), Integer.TYPE);
				} catch (final NumberFormatException e) {
					model = object.getValueFeature(object.eClass()
							.getEStructuralFeature(minOrMaxString));
				}
				if (null != model) {
					dbc.bindValue(target, model);
				}
			}
		}
	}

	private final class FirstElementsEnumViewerSorter extends ViewerComparator {
		private final String firstElements;

		public FirstElementsEnumViewerSorter(final String featureKey) {
			final String fe = dynamicMessages.getStringOrNull(
					featureKey + Messages.SettingTable_FirstElementsPostfix);
			firstElements = null == fe ? null
					: dynamicMessages.encloseWithCommas(fe);
		}

		@Override
		public int category(final Object element) {
			if (null == firstElements) {
				return super.category(element);
			}
			final String name = dynamicMessages
					.encloseWithCommas(((Enum<?>) element).name());
			final int index = firstElements.indexOf(name);
			if (index >= 0) {
				return index;
			}
			return firstElements.length() + super.category(element);
		}
	}

	private final DataBindingContext dbc;
	private final IValueChangeListener valueRefreshListener;
	private DynamicMessages dynamicMessages;

	/**
	 * Erstellt ein Composite mit einer Zeile von Dateneinstellungen.
	 *
	 * @param parent
	 *            das parent Composite
	 * @param dbc
	 *            ein {@link DataBindingContext}
	 * @param valueRefreshListener
	 *            wird aufgerufen, sobald sich ein Wert ändert
	 */
	public SettingTable(final Composite parent, final DataBindingContext dbc,
			final IValueChangeListener valueRefreshListener) {
		super(parent, SWT.NONE);
		this.dbc = dbc;
		this.valueRefreshListener = valueRefreshListener;
	}

	/**
	 * Darf überschrieben werden, um für ein bestimtes Feature einen besonderen
	 * {@link ILabelProvider} zu verwenden.
	 *
	 * @param feature
	 *            das Feature
	 * @param labelProvider
	 *            der Standard-{@link ILabelProvider}
	 * @return der {@link ILabelProvider}
	 */
	protected ILabelProvider wrapComboLabelProvider(
			final EStructuralFeature feature,
			final ILabelProvider labelProvider) {
		return labelProvider;
	}

	/**
	 * Darf überschrieben werden, um ein bestimtes Feature zu validieren.
	 *
	 * @param feature
	 *            das Feature
	 * @param c
	 *            das Coltrol zum dekorieren
	 * @return der {@link IValidator}
	 */
	protected IValidator getValidator(final EStructuralFeature feature,
			final Control c) {
		return null;
	}

	/**
	 * Erstellt die inneren Komponenten für die {@link SettingTable}.
	 *
	 * @param toolkit
	 *            das Toolkit
	 * @param object
	 *            das Objekt mit den Features
	 * @param defaults
	 *            nochmal vom selben Typ, aber mit Defaiult-Werten
	 * @param clazz
	 *            der Klasse des Objektes
	 * @param unknownMultiFeatures
	 *            welche Features sollen ausgeblendet werden
	 * @param bundle
	 *            für die Anzeige von Beschriftungen
	 */

	public void createSettingTable(final FormToolkit toolkit,
			final IObservableObject<?> object,
			final IObservableObject<?> defaults, final EClass clazz,
			final IObservableList unknownMultiFeatures,
			final ResourceBundle bundle) {
		dynamicMessages = DynamicMessages.getInstance(bundle);
		final EList<EStructuralFeature> features = clazz
				.getEStructuralFeatures();
		final GridLayout layout = new GridLayout(1, false);
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		layout.horizontalSpacing = 10;
		setLayout(layout);
		final String classKey = Messages.SettingTable_DynamicMessagesClassPrefix
				+ clazz.getName();
		final String excluded = dynamicMessages
				.encloseWithCommas(dynamicMessages.getStringOrNull(
						classKey + Messages.SettingTable_exclude));
		final String noClear = dynamicMessages.encloseWithCommas(dynamicMessages
				.getStringOrNull(classKey + Messages.SettingTable_no_clear));
		final List<Runnable> deferred = new ArrayList<>();
		// spaltenweise Verarbeitung der Attribute
		for (final EStructuralFeature feature : features) {
			if (excluded.contains(
					dynamicMessages.encloseWithCommas(feature.getName()))) {
				continue;
			}
			if (feature instanceof final EReference ref) {
				if (ref.isContainer()
						|| ref.isMany() && null != ref.getEOpposite()) {
					continue;
				}
			}
			layout.numColumns += 2;
			final String featureKey = classKey + Messages.Underline
					+ feature.getName();
			final Label label = toolkit.createLabel(this,
					dynamicMessages.getString(featureKey), SWT.NONE);
			final IObservableValue unknownModel;
			final IObservableValue valueFeature = object
					.getValueFeature(feature);
			if (noClear.contains(
					dynamicMessages.encloseWithCommas(feature.getName()))) {
				unknownModel = null;
				label.setLayoutData(
						new GridData(SWT.FILL, SWT.FILL, true, false, 2, 1));
			} else {
				label.setLayoutData(
						new GridData(SWT.FILL, SWT.FILL, true, false, 1, 1));
				Label clearButton;
				final IObservableValue defaultValue = null == defaults ? null
						: defaults.getValueFeature(feature);
				if (unknownMultiFeatures != null) {
					unknownModel = new CollectionContainsValue(
							unknownMultiFeatures, feature);
					clearButton = ObservableClearButton.create(this, SWT.FLAT,
							dbc, object.getValueFeature(feature), defaultValue,
							new ComputedValue(object.getObservable().getRealm(),
									Boolean.TYPE) {
								@Override
								protected Boolean calculate() {
									final Object dv = null == defaultValue
											? null
											: defaultValue.getValue();
									final Object v = valueFeature.getValue();
									return ((Boolean) unknownModel.getValue())
											|| v != dv
											|| (v != null && !v.equals(dv));
								}
							}, createResetToDefaults(valueFeature, defaultValue,
									feature));
				} else {
					unknownModel = null;
					clearButton = ObservableClearButton.create(this, SWT.FLAT,
							dbc, object.getValueFeature(feature), defaultValue);
				}
				clearButton.setLayoutData(
						new GridData(SWT.FILL, SWT.FILL, false, false, 1, 1));
			}
			final String tooltip = dynamicMessages.getStringOrNull(
					featureKey + Messages.SettingTable_TooltipTextPostfix);
			label.setToolTipText(tooltip);

			deferred.add(new DeferredSettingLineControlAdder(featureKey, object,
					label, toolkit, unknownMultiFeatures, tooltip, feature,
					valueFeature, unknownModel));
		}
		for (final Runnable r : deferred) {
			try {
				r.run();
			} catch (final Exception e) {
				e.printStackTrace();
			}
		}
		toolkit.paintBordersFor(this);
	}

	/**
	 * Darf überschrieben werden, damit das Runnable beim Zurücksetzen auf
	 * Standardwerte auszuführen.
	 *
	 * @param valueFeature
	 *            das {@link IObservableValue}
	 * @param defaultValue
	 *            der Default-Wert
	 * @param feature
	 *            das Feature
	 * @return ein Runnnable
	 */
	protected Runnable createResetToDefaults(
			final IObservableValue valueFeature,
			final IObservableValue defaultValue,
			final EStructuralFeature feature) {
		throw new IllegalArgumentException();
	}

	/**
	 * Gibt die Liste der möglichen Referenzwerte zurück, die in eine ComboBox
	 * gefüllt wird.
	 *
	 * @param r
	 *            die Referenz
	 * @return die Liste
	 */
	protected IObservableList getListForReference(final EReference r) {
		throw new IllegalArgumentException();
	}

	/**
	 * Wenn ein spezieller {@link ColorPicker} gewünscht ist, dann muss dies
	 * hier gemacht werden.
	 *
	 * @param parent
	 *            das Parent Composite
	 * @param object
	 *            das Haupt-Objekt
	 * @param feature
	 *            das Feature
	 * @return der {@link ColorPicker}
	 */
	protected ColorPicker createRGBColorPicker(final Composite parent,
			final IObservableObject<? extends EObject> object,
			final EStructuralFeature feature) {
		return new ColorPicker(parent, SWT.FLAT);
	}

	/**
	 * Ist ein bestimtes Feature aktiviert? Die Auswertung erfolgt in einem
	 * {@link ComputedValue}, so dass alle zugegriffenen Observables getrackt
	 * werden.
	 *
	 * @param feature
	 *            das Feature
	 * @return Wahr oder Falsch
	 */
	protected boolean isEnabled(final EStructuralFeature feature) {
		return true;
	}

}
