/*
 * Rahmenwerk-Plug-in "BitCtrl-Bibliotheken"
 *
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.rw.bitctrl.eclipse.dialogs;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.eclipse.core.runtime.Assert;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.dialogs.CheckedTreeSelectionDialog;
import org.eclipse.ui.dialogs.ElementListSelectionDialog;
import org.eclipse.ui.dialogs.ElementTreeSelectionDialog;
import org.eclipse.ui.dialogs.ListSelectionDialog;
import org.eclipse.ui.dialogs.SelectionDialog;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.sys.funclib.bitctrl.util.SystemObjectComparator;

/**
 * Bietet eine Auswahl an verschiedenen Auswahldialogen für Systemobjekte an.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 */
public final class SystemObjectSelectionDialog {

	/** Konstante für kein ausgewähltes Systemobjekt. */
	public static final String NO_SELECTION = "<Keine Auswahl>";

	/**
	 * Erzeugt einen Dialog der Systemobjekte als Liste zur Auswahl anbietet.
	 *
	 * @param parent
	 *            die übergeordnete Shell des Dialogs.
	 * @param systemObjectType
	 *            der Typ der auszuwählenden Systemobjekte.
	 * @param style
	 *            der Stil der Auswahl: {@link SWT#SINGLE} oder
	 *            {@link SWT#MULTI}
	 * @return der inititialisierte Auswahldialog.
	 * @see #getFirstSelectedObject(SelectionDialog)
	 * @see #getSelectedObjects(SelectionDialog)
	 */
	public static SelectionDialog createListSelectionDialog(final Shell parent,
			final SystemObjectType systemObjectType, final int style) {
		Assert.isLegal((style & SWT.SINGLE) == SWT.SINGLE
				|| (style & SWT.MULTI) == SWT.MULTI
				|| (style & SWT.NO) == SWT.NO);
		final boolean multipleSelection = (style & SWT.MULTI) == SWT.MULTI;
		final boolean noSelection = (style & SWT.NO) == SWT.NO;

		final List objects = new ArrayList<Object>(
				systemObjectType.getObjects());
		Collections.sort(objects, new SystemObjectComparator());

		if (noSelection) {
			objects.add(0, NO_SELECTION);
		}

		final ElementListSelectionDialog dialog = new ElementListSelectionDialog(
				parent, new LabelProvider());
		dialog.setMultipleSelection(multipleSelection);
		dialog.setElements(objects.toArray());
		init(dialog, systemObjectType, multipleSelection);
		return dialog;
	}

	/**
	 * Erzeugt einen Dialog der Systemobjekte als Liste zur Auswahl anbietet.
	 *
	 * @param parent
	 *            die übergeordnete Shell des Dialogs.
	 * @param systemObjectType
	 *            der Typ der auszuwählenden Systemobjekte.
	 * @return der inititialisierte Auswahldialog.
	 * @see #getFirstSelectedObject(SelectionDialog)
	 * @see #getSelectedObjects(SelectionDialog)
	 */
	public static SelectionDialog createMultiSelectionDialog(final Shell parent,
			final SystemObjectType systemObjectType) {
		final List<SystemObject> objects = new ArrayList<>(
				systemObjectType.getObjects());
		Collections.sort(objects, new SystemObjectComparator());
		final ListSelectionDialog dialog = new ListSelectionDialog(parent,
				objects, new ArrayContentProvider(), new LabelProvider(), null);
		init(dialog, systemObjectType, true);
		return dialog;
	}

	/**
	 * Erzeugt einen Dialog der Systemobjekte als Baum zur Auswahl anbietet. Der
	 * Baum hat zwei Ebenen, die erste sind die Systemobjekttypen die Untertypen
	 * des angegebenen sind, die zweite die zum Systemobjekte zum jeweiligen
	 * Systemobjekttyp.
	 *
	 * @param parent
	 *            die übergeordnete Shell des Dialogs.
	 * @param systemObjectType
	 *            der Typ der auszuwählenden Systemobjekte.
	 * @param style
	 *            der Stil der Auswahl: {@link SWT#SINGLE} oder
	 *            {@link SWT#MULTI}
	 * @return der inititialisierte Auswahldialog.
	 * @see #getFirstSelectedObject(SelectionDialog)
	 * @see #getSelectedObjects(SelectionDialog)
	 */
	public static SelectionDialog createTreeSelectionDialog(final Shell parent,
			final SystemObjectType systemObjectType, final int style) {
		Assert.isLegal(style == SWT.SINGLE || style == SWT.MULTI);
		final boolean multipleSelection = style == SWT.MULTI;

		final ElementTreeSelectionDialog dialog = new ElementTreeSelectionDialog(
				parent, new LabelProvider(),
				new SystemObjectTreeContentProvider());
		dialog.setAllowMultiple(multipleSelection);
		dialog.setInput(systemObjectType);
		init(dialog, systemObjectType, multipleSelection);
		return dialog;
	}

	/**
	 * Erzeugt einen Dialog der Systemobjekte als Baum zur Auswahl anbietet. Der
	 * Baum hat zwei Ebenen, die erste sind alle verfügbaren die
	 * Systemobjekttypen, die zweite die zum Systemobjekte zum jeweiligen
	 * Systemobjekttyp.
	 *
	 * @param parent
	 *            die übergeordnete Shell des Dialogs.
	 * @param dav
	 *            die zu nutzende Datenverteilerverbindung, über die die
	 *            verfügbaren Objekte abgerufen werden sollen.
	 * @return der inititialisierte Auswahldialog.
	 * @see #getFirstSelectedObject(SelectionDialog)
	 * @see #getSelectedObjects(SelectionDialog)
	 */
	public static SelectionDialog createTreeSelectionDialog(final Shell parent,
			final ClientDavInterface dav) {
		final CheckedTreeSelectionDialog dialog = new CheckedTreeSelectionDialog(
				parent, new LabelProvider(),
				new SystemObjectTreeContentProvider());
		dialog.setInput(dav);
		init(dialog, null, true);
		return dialog;
	}

	/**
	 * Initialisiert einen Auswahldialog für Systemobjekte.
	 *
	 * @param dialog
	 *            der zu initialisierende Dialog.
	 * @param systemObjectType
	 *            der Typ der Systemobjekte die zur Auswahl stehen.
	 * @param multipleSelection
	 *            Flag, ob mehrere Systemobjekte ausgewählt werden dürfen.
	 */
	public static void init(final SelectionDialog dialog,
			final SystemObjectType systemObjectType,
			final boolean multipleSelection) {
		final String title;
		String message;
		if (multipleSelection) {
			title = "Systemobjekte auswählen";
			message = "Bitte wählen Sie ein oder mehrere Systemobjekte";
			if (systemObjectType != null) {
				message += " vom Typ" + systemObjectType;
			}
			message += " aus.";
		} else {
			title = "Systemobjekt auswählen";
			message = "Bitte wählen Sie ein Systemobjekt";
			if (systemObjectType != null) {
				message += " vom Typ" + systemObjectType;
			}
			message += " aus.";
		}

		dialog.setTitle(title);
		dialog.setMessage(message);
	}

	/**
	 * Gibt das erste der ausgewählten Systemobjekte zurück.
	 *
	 * @param dialog
	 *            ein Dialog der Systemobjekte zur Auswahl anbietet.
	 * @return das erste ausgewählte Systemobjekt oder <code>null</code>, wenn
	 *         keines ausgewählt wurde.
	 */
	public static SystemObject getFirstSelectedObject(
			final SelectionDialog dialog) {
		final Object[] result = dialog.getResult();
		if (result != null) {
			if (result[0].equals(NO_SELECTION)) {
				return null;
			}
			return (SystemObject) result[0];
		}

		return null;
	}

	/**
	 * Gibt die ausgewählten Systemobjekte zurück.
	 *
	 * @param dialog
	 *            ein Dialog der Systemobjekte zur Auswahl anbietet.
	 * @return die ausgewählten Systemobjekte oder ein leeres Feld, wenn keine
	 *         ausgewählt wurde, aber nie <code>null</code>.
	 */
	public static List<SystemObject> getSelectedObjects(
			final SelectionDialog dialog) {
		final Object[] dialogResult = dialog.getResult();
		if (dialogResult != null) {
			final List<SystemObject> result = new ArrayList<>();
			for (final Object o : dialogResult) {
				if (o instanceof SystemObject) {
					result.add((SystemObject) o);
				}
			}
			return result;
		}

		return Collections.emptyList();
	}

	private SystemObjectSelectionDialog() {
		// utility class
	}

}
