/*
 * Rahmenwerk-Plug-in "BitCtrl-Bibliotheken"
 *
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.rw.bitctrl.eclipse.internal;

import java.util.HashMap;
import java.util.Map;

import org.eclipse.ui.AbstractSourceProvider;

import de.bsvrz.buv.rw.basislib.Rahmenwerk;
import de.bsvrz.buv.rw.basislib.dav.DavVerbindungsEvent;
import de.bsvrz.buv.rw.basislib.dav.DavVerbindungsListener;
import de.bsvrz.buv.rw.bitctrl.BitCtrlRahmenwerkConstants;
import de.bsvrz.buv.rw.bitctrl.internal.RahmenwerkService;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.util.benutzer.Benutzerverwaltung;

/**
 * Source Provider für die Variablen und Properties des Plug-ins.
 *
 * <p>
 * Versorgt werden die folgenden Properties:
 * <ul>
 * <li>{@link BitCtrlRahmenwerkConstants#VARIABLE_IS_AKTUELLER_BENUTZER_ADMIN}</li>
 * <li>{@link BitCtrlRahmenwerkConstants#VARIABLE_IS_DAV_VERBUNDEN}</li>
 * </ul>
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class SourceProvider extends AbstractSourceProvider
		implements DavVerbindungsListener {

	private final Map<Object, Object> currentState;

	/**
	 * Initialsisiert den Source Provider und setzt initiale Werte.
	 */
	public SourceProvider() {
		currentState = new HashMap<>();

		currentState.put(BitCtrlRahmenwerkConstants.VARIABLE_IS_DAV_VERBUNDEN,
				RahmenwerkService.getService().getRahmenWerk().isOnline());
		updateIsAdmin();
		RahmenwerkService.getService().getRahmenWerk()
				.addDavVerbindungsListener(this);
	}

	@Override
	public Map<Object, Object> getCurrentState() {
		return currentState;
	}

	@Override
	public String[] getProvidedSourceNames() {
		return new String[] {
				BitCtrlRahmenwerkConstants.VARIABLE_IS_AKTUELLER_BENUTZER_ADMIN };
	}

	@Override
	public void dispose() {
		RahmenwerkService.getService().getRahmenWerk()
				.removeDavVerbindungsListener(this);
	}

	/**
	 * Aktualisiert die Property
	 * {@link BitCtrlRahmenwerkConstants#VARIABLE_IS_AKTUELLER_BENUTZER_ADMIN}.
	 * Ist keine Verbindung zum Datenverteiler vorhanden, dann wird das Flag auf
	 * <code>true</code> gesetzt.
	 */
	private void updateIsAdmin() {
		final ObjektFactory objektFactory = RahmenwerkService.getService()
				.getObjektFactory();
		final Rahmenwerk rahmenwerk = RahmenwerkService.getService()
				.getRahmenWerk();
		if (objektFactory.isVerbunden() && rahmenwerk.isOnline()) {
			final String benutzer = rahmenwerk.getBenutzerName();
			final String passwort = rahmenwerk.getPasswort();
			final Benutzerverwaltung benutzerverwaltung = new Benutzerverwaltung(
					objektFactory);
			currentState.put(
					BitCtrlRahmenwerkConstants.VARIABLE_IS_AKTUELLER_BENUTZER_ADMIN,
					benutzerverwaltung.isDAVAdmin(benutzer, passwort));
		} else {
			currentState.put(
					BitCtrlRahmenwerkConstants.VARIABLE_IS_AKTUELLER_BENUTZER_ADMIN,
					true);
		}
	}

	@Override
	public void verbindungHergestellt(final DavVerbindungsEvent event) {
		currentState.put(BitCtrlRahmenwerkConstants.VARIABLE_IS_DAV_VERBUNDEN,
				true);

		updateIsAdmin();
	}

	@Override
	public void verbindungGetrennt(final DavVerbindungsEvent event) {
		currentState.put(BitCtrlRahmenwerkConstants.VARIABLE_IS_DAV_VERBUNDEN,
				false);

		updateIsAdmin();
	}

	@Override
	public boolean verbindungHalten(final DavVerbindungsEvent event) {
		return false;
	}
}
