/*
 * Rahmenwerk-Plug-in "BitCtrl-Bibliotheken"
 *
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.rw.bitctrl.eclipse.modell;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.viewers.DecoratingLabelProvider;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.ILabelDecorator;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.actions.ActionFactory;
import org.eclipse.ui.part.ViewPart;

import com.bitctrl.lib.eclipse.actions.RefreshViewerAction;
import com.bitctrl.lib.eclipse.emf.viewers.EmfLabelProvider;

import de.bsvrz.buv.rw.bitctrl.eclipse.modell.emf.EMFModellEinstellungen;
import de.bsvrz.buv.rw.bitctrl.eclipse.viewers.EinstellungenContentProvider;
import de.bsvrz.buv.rw.bitctrl.internal.RahmenwerkService;
import de.bsvrz.sys.funclib.bitctrl.daf.DavProvider;

/**
 * Basisklasse für View Parts die Rahmenwerkseinstellungen anzeigen.
 *
 * <p>
 * Der View Part pollt um Änderungen an den Einstellungen mitzubekommen.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public abstract class AbstractEinstellungenView extends ViewPart {

	/**
	 * Aktualisiert den Onlinezustand der Datenverteilerverbindung.
	 */
	private final class VerbundenPropertyChangeListener
			implements PropertyChangeListener {
		@Override
		public void propertyChange(final PropertyChangeEvent evt) {
			online = (Boolean) evt.getNewValue();
			updateOnline();
		}

	}

	private VerbundenPropertyChangeListener verbundenPropertyChangeListener;
	private DavProvider davProvider;
	private boolean online;

	private TreeViewer viewer;
	private MenuManager contextMenu;

	@Override
	public void init(final IViewSite site) throws PartInitException {
		super.init(site);

		davProvider = RahmenwerkService.getService().getObjektFactory();
		online = davProvider.isVerbunden();

		verbundenPropertyChangeListener = new VerbundenPropertyChangeListener();
		davProvider.addPropertyChangeListener(DavProvider.PROP_VERBUNDEN,
				verbundenPropertyChangeListener);
	}

	@Override
	public final void createPartControl(final Composite parent) {
		viewer = new TreeViewer(parent, SWT.MULTI | SWT.BORDER);
		viewer.setContentProvider(new EinstellungenContentProvider());
		final ILabelProvider labelProvider = new EmfLabelProvider();
		final ILabelDecorator decorator = PlatformUI.getWorkbench()
				.getDecoratorManager().getLabelDecorator();
		viewer.setLabelProvider(
				new DecoratingLabelProvider(labelProvider, decorator));

		getViewSite().getActionBars().setGlobalActionHandler(
				ActionFactory.REFRESH.getCommandId(),
				new RefreshViewerAction(viewer));

		configureViewer(viewer);

		viewer.addSelectionChangedListener(event -> {
			contextMenu.getMenu().setVisible(true);
			contextMenu.getMenu().setVisible(false);
		});

		makeActions();
		hookContextMenu();
		hookDoubleClickAction();
		contributeToActionBars();

		getSite().setSelectionProvider(viewer);
		updateOnline();
	}

	@Override
	public void setFocus() {
		viewer.getControl().setFocus();
	}

	private void hookContextMenu() {
		contextMenu = new MenuManager("#PopupMenu");
		contextMenu.setRemoveAllWhenShown(true);
		contextMenu.addMenuListener(
				AbstractEinstellungenView.this::fillContextMenu);

		final Menu menu = contextMenu.createContextMenu(viewer.getControl());
		viewer.getControl().setMenu(menu);
		getSite().registerContextMenu(contextMenu, viewer);
	}

	private void contributeToActionBars() {
		final IActionBars bars = getViewSite().getActionBars();
		fillLocalPullDown(bars.getMenuManager());
		fillLocalToolBar(bars.getToolBarManager());
	}

	/**
	 * Die Standardimplementierung fügt nur die Action Group
	 * {@value IWorkbenchActionConstants#MB_ADDITIONS} hinzu.
	 *
	 * <p>
	 * Abgeleitete Klassen sollten diese Methode erweitern und die Supermethode
	 * am Schluß aufrufen.
	 *
	 * @param manager
	 */
	protected void fillLocalPullDown(final IMenuManager manager) {
		manager.add(new Separator(IWorkbenchActionConstants.MB_ADDITIONS));
	}

	/**
	 * Die Standardimplementierung fügt nur die Action Groups
	 * {@value IWorkbenchActionConstants#MB_ADDITIONS} hinzu.
	 *
	 * <p>
	 * Abgeleitete Klassen sollten diese Methode erweitern und die Supermethode
	 * am Schluß aufrufen.
	 *
	 * @param manager
	 */
	protected void fillContextMenu(final IMenuManager manager) {
		manager.add(new Separator(IWorkbenchActionConstants.MB_ADDITIONS));
	}

	/**
	 * Die Standardimplementierung fügt nur die Action Group
	 * {@value IWorkbenchActionConstants#MB_ADDITIONS} hinzu.
	 *
	 * <p>
	 * Abgeleitete Klassen sollten diese Methode erweitern und die Supermethode
	 * am Schluß aufrufen.
	 *
	 * @param manager
	 */
	protected void fillLocalToolBar(final IToolBarManager manager) {
		manager.add(new Separator(IWorkbenchActionConstants.MB_ADDITIONS));
	}

	private void hookDoubleClickAction() {
		viewer.addDoubleClickListener(this::doDoubleClick);
	}

	/**
	 * Diese Methode wird beim Doppelklick auf ein Element im Viewer aufgerufen.
	 * In abgeleiteten Klassen überschreiben, um auf den Doppelklick zu
	 * reagieren.
	 *
	 * @param event
	 *            das Doppelklickevent.
	 */
	protected void doDoubleClick(final DoubleClickEvent event) {
		// tut nix
	}

	/**
	 * Erzeugt die Aktions die später verwendet werden sollen. Wird vor den
	 * <code>fill*</code>-Methoden aufgerufen.
	 *
	 * <p>
	 * Die Standardimplementierung tut nix.
	 */
	protected void makeActions() {
		// tut nix
	}

	/**
	 * Konfiguriert den Tree Viewer mit den Einstellungen. Hier muss mindestens
	 * mit {@link TreeViewer#setInput(Object)} der Input als Instanz von
	 * {@link EMFModellEinstellungen} gesetzt werden.
	 *
	 * @param treeViewer
	 */
	protected void configureViewer(final TreeViewer treeViewer) {
		// tut nix
	}

	/**
	 * Gibt den Viewer des View Part zurück.
	 *
	 * @return der Viewer.
	 */
	public StructuredViewer getViewer() {
		return viewer;
	}

	private void updateOnline() {
		getSite().getWorkbenchWindow().getShell().getDisplay().asyncExec(() -> {
			if (online) {
				geheOnline();
			} else {
				geheOffline();
			}
		});
	}

	/**
	 * Flag ob die View online geschalten ist. Dies ist nur dann der Fall, wenn
	 * eine Verbindung zum Datenverteiler besteht.
	 *
	 * @return <code>true</code>, wenn eine Datenverteilerverbindung besteht,
	 *         andernfalls <code>false</code>.
	 */
	public boolean isOnline() {
		return online;
	}

	/**
	 * Wird aufgerufen, wenn die Verbindung zum Datenverteiler hergestellt
	 * wurde.
	 */
	protected void geheOnline() {
		viewer.getControl().setEnabled(true);
		viewer.refresh();
		setContentDescription("");
	}

	/**
	 * Wird aufgerufen, wenn die Verbindung zum Datenverteiler getrennt wurde.
	 */
	protected void geheOffline() {
		if (!viewer.getControl().isDisposed()) {
			viewer.getControl().setEnabled(false);
			setContentDescription("Keine Verbindung zum Datenverteiler.");
		}
	}

	@Override
	public void dispose() {
		davProvider.removePropertyChangeListener(DavProvider.PROP_VERBUNDEN,
				verbundenPropertyChangeListener);
		super.dispose();
	}

}
