/*
 * Rahmenwerk-Plug-in "BitCtrl-Bibliotheken"
 *
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.rw.bitctrl.eclipse.modell;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import com.bitctrl.model.ModelElement;

import de.bsvrz.buv.rw.basislib.einstellungen.EinstellungChangeListener;
import de.bsvrz.buv.rw.basislib.einstellungen.Einstellungen;
import de.bsvrz.buv.rw.basislib.einstellungen.EinstellungsAdresse;
import de.bsvrz.buv.rw.basislib.einstellungen.EinstellungsEvent;
import de.bsvrz.buv.rw.basislib.einstellungen.SpeicherKey;
import de.bsvrz.buv.rw.basislib.einstellungen.UrlasserDialogAbgebrochenException;
import de.bsvrz.buv.rw.bitctrl.internal.BitCtrlRahmenwerkPlugin;
import de.bsvrz.buv.rw.bitctrl.internal.RahmenwerkService;
import de.bsvrz.puk.param.lib.daten.UrlasserInfo;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Basisklasse zum Laden und Speichern von Benutzereinstellungen.
 *
 * @author BitCtrl Systems GmbH, schnepel
 *
 * @param <N>
 *            der Named-Typ
 * @param <T>
 *            der Objekt-Typ
 */
public abstract class ModellEinstellungen<N, T> extends ModelElement {

	private static final long serialVersionUID = 1L;

	/**
	 * Property die getriggert wird, wenn eine Einstellung gesichert wurde. Der
	 * Wert ist der Name der Einstellung.
	 */
	public static final String PROP_EINSTELLUNG_CHANGED = "einstellungChanged";

	private final Class<T> type;

	private final Debug logger = Debug.getLogger();

	/**
	 * Initialisiert den Typ der zu speichernden Einstellungen.
	 *
	 * @param type
	 *            die Klasse der zu speichernden Einstellungen
	 */
	protected ModellEinstellungen(final Class<T> type) {
		this.type = type;

		final Einstellungen verwaltung = RahmenwerkService.getService()
				.getEinstellungen();
		verwaltung.addEinstellungsListener(new EinstellungChangeListener() {

			@Override
			public void einstellungEntfernt(final EinstellungsEvent event) {
				firePropertyChanged(PROP_EINSTELLUNG_CHANGED, null, null);

			}

			@Override
			public void einstellungAngelegt(final EinstellungsEvent event) {
				firePropertyChanged(PROP_EINSTELLUNG_CHANGED, null, null);
			}

			@Override
			public void einstellungAktualisiert(final EinstellungsEvent event) {
				firePropertyChanged(PROP_EINSTELLUNG_CHANGED, null, null);
			}

		});
	}

	/**
	 * Ermittelt das gespeicherte Objekt aus einer Einstellung.
	 *
	 * @param named
	 *            die Einstellung
	 * @return das Objekt
	 */
	protected abstract T getObject(N named);

	/**
	 * Setzt das Objekt in eine Einstellung.
	 *
	 * @param named
	 *            die Einstellung
	 * @param object
	 *            das Objekt
	 */
	protected abstract void setObject(N named, T object);

	/**
	 * Ermittelt den Namen einer Einstellung.
	 *
	 * @param named
	 *            die Einstellung
	 * @return der Name
	 */
	protected abstract String getName(N named);

	/**
	 * Setzt den Namen für eine Einstellung.
	 *
	 * @param named
	 *            die Einstellung
	 * @param name
	 *            der Name
	 */
	protected abstract void setName(N named, String name);

	/**
	 * Ermittelt den Zeitpunk der letzten Veränderung einer Einstellung.
	 *
	 * @param named
	 *            die Einstellung
	 * @return der Zeitpunkt
	 */
	protected abstract Date getLastModification(N named);

	/**
	 * Setzt den Zeitpunkt der letzten Veränderung in eine Einstellung.
	 *
	 * @param named
	 *            die Einstellung
	 * @param lastModification
	 *            der Zeitpunkt
	 */
	protected abstract void setLastModification(N named, Date lastModification);

	/**
	 * Erstellt ein neues Einstellungsobjekt.
	 *
	 * @return das Einstellungsobjekt
	 */
	protected abstract N createStorage();

	/**
	 * Kontainer fuer eine Zeile in den Einstellungsladen/-speichern
	 * Wizard-Seiten.
	 *
	 * @author BitCtrl Systems GmbH, schnepel
	 *
	 * @param <T>
	 */
	static class WizardPageItem<T> {
		private String name;
		private Date lastModification;
		private T object;

		public void setName(final String name) {
			this.name = name;
		}

		public String getName() {
			return name;
		}

		public void setLastModification(final Date lastModification) {
			this.lastModification = lastModification;
		}

		public Date getLastModification() {
			return lastModification;
		}

		public void setObject(final T object) {
			this.object = object;
		}

		public T getObject() {
			return object;
		}
	}

	/**
	 * Gibt alle zu der uebergebenen {@link EinstellungsArt} und des
	 * uebergebenen Typs passenden gespeicherten Objekte zurueck.
	 *
	 * @param art
	 *            die {@link EinstellungsArt}
	 * @return alle Objekte verlinkt unter dem verwendeten Namen
	 */
	final Map<String, WizardPageItem<T>> getModellEinstellungen(
			final SpeicherKey art) {
		final List<? extends N> contents = getModellEinstellungenList(art);
		final Map<String, WizardPageItem<T>> ret = new HashMap<>();
		if (null != contents) {
			for (final N named : contents) {
				final T object = getObject(named);
				if (type.isInstance(object)) {
					final WizardPageItem<T> r = new WizardPageItem<>();
					r.setName(getName(named));
					r.setLastModification(getLastModification(named));
					r.setObject(object);
					ret.put(r.getName(), r);
				}
			}
		}
		return ret;
	}

	/**
	 * Gibt alle zu dem uebergebenen {@link SpeicherKey} passenden gespeicherten
	 * Objekte zurueck.
	 *
	 * @param art
	 *            der {@link SpeicherKey}
	 * @return alle Objekte in einer Liste
	 */
	protected List<? extends N> getModellEinstellungenList(
			final SpeicherKey art) {

		final Einstellungen einstellungen = RahmenwerkService.getService()
				.getEinstellungen();

		final EinstellungsAdresse adresse = new EinstellungsAdresse(null,
				type.getName(), art.getOwnerType(), art.getPid(),
				art.getLocation());

		try {
			final String parameter = (String) einstellungen.getValue(adresse);
			final List<? extends N> contents = getModellEinstellungenInternal(
					parameter);
			return contents;
		} catch (final IOException ex) {
			logger.warning(ex.getLocalizedMessage());
		}
		return Collections.emptyList();
	}

	/**
	 * Gibt das zu der uebergebenen {@link SpeicherKey}, des uebergebenen Typs
	 * passenden unter dem übergebenen Namen gespeicherten Objekte zurueck.
	 *
	 * @param art
	 *            die {@link SpeicherKey}
	 * @param name
	 *            der Name unter dem das Objekt gespeichert ist
	 * @return das Objekt oder <code>null</code>, wenn kein Objekt mit dem Namen
	 *         existiert.
	 */
	public T getModellEinstellungen(final SpeicherKey art, final String name) {
		final List<? extends N> contents = getModellEinstellungenList(art);
		for (final N named : contents) {
			if (getName(named).equals(name)) {
				return getObject(named);
			}
		}
		return null;
	}

	/**
	 * Liefert alle Namen von Objekten, die unter einer bestimmten
	 * Einstellungsart gespeichert wurden.
	 *
	 * @param art
	 *            die Einstellungsart, für die Einstellungen gesucht werden.
	 * @return die Liste der Namen der gefundenen Einstellungen. Gibt es keine,
	 *         wird eine leere Liste zurückgegeben, aber nie <code>null</code>.
	 */
	public List<String> getModellEinstellungenNamen(final SpeicherKey art) {
		final List<String> result = new ArrayList<>();
		final List<? extends N> contents = getModellEinstellungenList(art);
		for (final N named : contents) {
			result.add(getName(named));
		}
		return result;
	}

	/**
	 * Liefert alle Modelleinstellungen, die in den Nutzereinstellungen gefunden
	 * wurden.
	 *
	 * @param art
	 *            der Speicherort, wo nach Einstellungen gesucht werden soll.
	 * @return die gefundenen Einstellungen. Gibt es keine, wird eine leere
	 *         Liste zurückgegeben, aber nie <code>null</code>.
	 */
	public List<T> getAlleModellEinstellungen(final SpeicherKey art) {
		final List<T> result = new ArrayList<>();
		for (final String name : getModellEinstellungenNamen(art)) {
			result.add(getModellEinstellungen(art, name));
		}
		result.removeIf(Objects::isNull);
		return result;
	}

	/**
	 * wandelt den übergebenen String in eine Liste von Einstellungen um.
	 *
	 * @param param
	 *            der String
	 * @return die Liste aller Einstellungen.
	 */
	protected abstract List<N> getModellEinstellungenInternal(String param);

	/**
	 * Speichert ein Objekt unter dem uebergebenen {@link SpeicherKey} und dem
	 * uebergebenen Namen ab.
	 *
	 * @param art
	 *            der {@link SpeicherKey}
	 * @param name
	 *            der Name, unter dem gespeichert werden soll
	 * @param object
	 *            das Objekt
	 * @throws IOException
	 *             speichern war nicht erfolgreich
	 * @throws UrlasserDialogAbgebrochenException
	 *             der potentiell angezeigte Urlasserdialog wurde vom Anwender
	 *             abgebrochen
	 */
	public void setModellEinstellungen(final SpeicherKey art, final String name,
			final T object) throws IOException {

		final Einstellungen verwaltung = RahmenwerkService.getService()
				.getEinstellungen();

		final EinstellungsAdresse adresse = new EinstellungsAdresse(null,
				type.getName(), art.getOwnerType(), art.getPid(),
				art.getLocation());
		String param = (String) verwaltung.getValue(adresse);
		final List<N> contents = getModellEinstellungenInternal(param);
		if (null != contents) {
			boolean hasSet = false;
			for (final N named : contents) {
				final String contentsName = getName(named);
				if (contentsName != null && contentsName.equals(name)) {
					setObject(named, object);
					setLastModification(named, new Date());
					hasSet = true;
					break;
				}
			}
			if (!hasSet) {
				final N named = createStorage();
				setName(named, name);
				setObject(named, object);
				setLastModification(named, new Date());
				contents.add(named);
			}
		}

		param = saveModellEinstellungenInternal(contents);
		verwaltung.setValue(adresse, param);
		firePropertyChanged(PROP_EINSTELLUNG_CHANGED, null, name);
	}

	/**
	 * wandelt die Liste der Einstellungen in einen String um.
	 *
	 * @param contents
	 *            die Liste der Einstellungen
	 * @return der String
	 */
	protected abstract String saveModellEinstellungenInternal(List<N> contents);

	/**
	 * Gibt die Klasse der gespeicherten Einstellungen zurück.
	 *
	 * @return die Klasse
	 */
	protected final Class<T> getTypeClass() {
		return type;
	}

	/**
	 * Sollte in den Implementierenden Klassen beim Eintreten einer Exception
	 * beim Lesen von Einstellungen aufgerufen werden, um ein einheitliches
	 * Fehlerhandling sicherzustellen. Der Fehler wird dem Benutzer in einem
	 * Fehlerdialog mitgeteilt und ins Error Log von Eclipse geschrieben.
	 *
	 * @param ex
	 *            der eingetretene Fehler in Form einer Exception.
	 * @param message
	 *            eine optionale Fehlermeldung. Kann <code>null</code> sein,
	 *            dann wird eine Standardtext ausgegeben.
	 */
	protected void handleReadError(final Exception ex, final String message) {
		final String error;
		if (message != null) {
			error = message;
		} else {
			error = "Die Benutzereinstellungen konnten wegen eines Fehlers beim Deserialisieren nicht gelesen werden.";
		}

		BitCtrlRahmenwerkPlugin.getDefault().getLogger()
				.error(error + ": " + ex.getLocalizedMessage());
	}

	/**
	 * Sollte in den Implementierenden Klassen beim Eintreten einer Exception
	 * beim Schreiben von Einstellungen aufgerufen werden, um ein einheitliches
	 * Fehlerhandling sicherzustellen. Der Fehler wird dem Benutzer in einem
	 * Fehlerdialog mitgeteilt und ins Error Log von Eclipse geschrieben.
	 *
	 * @param ex
	 *            der eingetretene Fehler in Form einer Exception.
	 * @param message
	 *            eine optionale Fehlermeldung. Kann <code>null</code> sein,
	 *            dann wird eine Standardtext ausgegeben.
	 */
	protected void handleWriteError(final Exception ex, final String message) {
		final String error;
		if (message != null) {
			error = message;
		} else {
			error = "Die Benutzereinstellungen konnten wegen eines Fehlers beim Serialisieren nicht geschrieben werden.";
		}

		BitCtrlRahmenwerkPlugin.getDefault().getLogger().error(error, ex);
	}

	/**
	 * Speichert ein Objekt unter dem uebergebenen {@link SpeicherKey} und dem
	 * uebergebenen Namen ab, falls Urlasserinformationen nötig sind, können
	 * diese gleich mit übergeben werden.
	 *
	 * @param art
	 *            der {@link SpeicherKey}
	 * @param name
	 *            der Name, unter dem gespeichert werden soll
	 * @param object
	 *            das Objekt
	 * @param urlasser
	 *            die Urlasserinformationen
	 * @throws IOException
	 *             speichern war nicht erfolgreich
	 */
	public void setModellEinstellungen(final SpeicherKey art, final String name,
			final T object, final UrlasserInfo urlasser) throws IOException {
		final Einstellungen verwaltung = RahmenwerkService.getService()
				.getEinstellungen();

		final EinstellungsAdresse adresse = new EinstellungsAdresse(null,
				type.getName(), art.getOwnerType(), art.getPid(),
				art.getLocation());
		String param = (String) verwaltung.getValue(adresse);
		final List<N> contents = getModellEinstellungenInternal(param);
		if (null != contents) {
			boolean hasSet = false;
			for (final N named : contents) {
				final String contentsName = getName(named);
				if (contentsName != null && contentsName.equals(name)) {
					setObject(named, object);
					setLastModification(named, new Date());
					hasSet = true;
					break;
				}
			}
			if (!hasSet) {
				final N named = createStorage();
				setName(named, name);
				setObject(named, object);
				setLastModification(named, new Date());
				contents.add(named);
			}
		}

		param = saveModellEinstellungenInternal(contents);
		verwaltung.setValue(adresse, param, urlasser);
		firePropertyChanged(PROP_EINSTELLUNG_CHANGED, null, name);

	}

}
