/*
 * Rahmenwerk-Plug-in "BitCtrl-Bibliotheken"
 *
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.rw.bitctrl.eclipse.modell;

import java.io.IOException;

import org.eclipse.core.databinding.observable.map.IObservableMap;
import org.eclipse.core.databinding.observable.map.WritableMap;
import org.eclipse.core.databinding.observable.value.IObservableValue;
import org.eclipse.core.databinding.observable.value.WritableValue;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.PlatformUI;

import com.bitctrl.lib.eclipse.actions.SaveAction;
import com.bitctrl.lib.eclipse.databinding.observables.ObservableMapKeySet;
import com.bitctrl.lib.eclipse.databinding.wizards.EnterOrSelectStringWizardPage;
import com.bitctrl.lib.eclipse.wizards.ComposedWizardDialog;

import de.bsvrz.buv.rw.basislib.Rahmenwerk;
import de.bsvrz.buv.rw.basislib.einstellungen.SpeicherKey;
import de.bsvrz.buv.rw.basislib.urlasser.UrlasserInfoDatenDialog;
import de.bsvrz.buv.rw.basislib.urlasser.UrlasserInfoDatenSender;
import de.bsvrz.buv.rw.bitctrl.eclipse.Messages;
import de.bsvrz.buv.rw.bitctrl.eclipse.wizards.SelectEinstellungsArtWizardPage;
import de.bsvrz.buv.rw.bitctrl.internal.BitCtrlRahmenwerkPlugin;
import de.bsvrz.buv.rw.bitctrl.internal.RahmenwerkService;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.puk.param.lib.daten.UrlasserInfo;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.util.benutzer.Benutzerverwaltung;

/**
 * Speichert ein Modell im Datenverteiler. Fragt den Anwender nach dem
 * {@link SpeicherKey} und einem Namen unter dem das Modell abgespeichert werden
 * soll. Wird verwendet, um Einstellungen benutzer-spezifisch zu speichern.
 *
 * @author BitCtrl Systems GmbH, schnepel
 *
 * @param <N>
 *            Typ der konkreten Named Klasse
 * @param <T>
 *            der zu speichernde Typ
 */
public abstract class ModellSaveAction<N, T> extends SaveAction {

	private final class ModellSaveUrlasserInfoDatenSender
			implements UrlasserInfoDatenSender {
		private final ModellEinstellungen<N, T> einstellungen;
		private final IObservableValue<SpeicherKey> artValue;
		private final String newName;

		private ModellSaveUrlasserInfoDatenSender(
				final ModellEinstellungen<N, T> einstellungen,
				final IObservableValue<SpeicherKey> artValue,
				final String newName) {
			this.einstellungen = einstellungen;
			this.artValue = artValue;
			this.newName = newName;
		}

		@Override
		public void execute(final ClientDavInterface verbindung,
				final UrlasserInfo urlasser) {
			try {
				einstellungen.setModellEinstellungen(artValue.getValue(),
						newName, getObject(newName));
				saveHook(newName);
			} catch (final IOException ex) {
				ErrorDialog.openError(
						PlatformUI.getWorkbench().getActiveWorkbenchWindow()
								.getShell(),
						"Benutzereinstellungen",
						"Die Einstellungen konnten nicht gesichert werden!",
						new Status(IStatus.ERROR,
								BitCtrlRahmenwerkPlugin.PLUGIN_ID,
								ex.getLocalizedMessage(), ex));
			}

		}
	}

	/**
	 * Konstruktor, der den ToolTipText setzt.
	 */
	protected ModellSaveAction() {
		setToolTipText("Einstellungen speichern");
	}

	@Override
	public final void run() {
		try {
			final Rahmenwerk rahmenwerk = RahmenwerkService.getService()
					.getRahmenWerk();
			if (!rahmenwerk.isOnline()) {
				return;
			}
			final ObjektFactory factory = RahmenwerkService.getService()
					.getObjektFactory();
			final Benutzerverwaltung benutzerverwaltung = new Benutzerverwaltung(
					factory);

			final boolean isAdmin = benutzerverwaltung.isDAVAdmin(
					rahmenwerk.getBenutzerName(), rahmenwerk.getPasswort());
			final IObservableValue<String> nameValue = new WritableValue<>(
					getCurrentName(), String.class);
			final IObservableMap cps = new WritableMap();
			final IObservableValue<SpeicherKey> artValue = new WritableValue<>(
					null, SpeicherKey.class);
			final ModellEinstellungen<N, T> einstellungen = createModelEinstellungen();
			artValue.addValueChangeListener(event -> {
				cps.clear();
				final SpeicherKey art = artValue.getValue();
				if (null != art) {
					cps.putAll(einstellungen.getModellEinstellungen(art));
				}
			});
			artValue.setValue(SpeicherKey.benutzerNetzweit());

			final EnterOrSelectStringWizardPage page2 = new EnterOrSelectStringWizardPage(
					Messages.EMFModelSaveAction_WizardHeader,
					Messages.EMFModelSaveAction_WizardDescription,
					Messages.EMFModelSaveAction_WizardPrompt1,
					Messages.EMFModelSaveAction_OR,
					Messages.EMFModelSaveAction_WizardPrompt2, nameValue,
					new ObservableMapKeySet(cps));

			final SelectEinstellungsArtWizardPage page1 = new SelectEinstellungsArtWizardPage(
					artValue);
			page1.setErlaubeNetzweit(isAdmin);

			// neue Shell erzeugen, damit der Wizard nicht im Hintergrund bleibt
			// und von der PreferencePage blockiert wird.
			final Display display = PlatformUI.createDisplay();
			final Shell shell = display.getActiveShell();
			final ComposedWizardDialog wizard = new ComposedWizardDialog(shell,
					Messages.EMFModelSaveAction_WizardCaption, page1, page2);
			if (wizard.open() == Window.OK) {
				final String newName = nameValue.getValue();
				final UrlasserInfoDatenDialog dialog = new UrlasserInfoDatenDialog(
						PlatformUI.getWorkbench().getActiveWorkbenchWindow()
								.getShell(),
						new ModellSaveUrlasserInfoDatenSender(einstellungen,
								artValue, newName));
				dialog.open();
			}
		} catch (final Exception e) {
			BitCtrlRahmenwerkPlugin.getDefault().getLog().log(new Status(
					IStatus.ERROR,
					de.bsvrz.buv.rw.bitctrl.internal.BitCtrlRahmenwerkPlugin.PLUGIN_ID,
					"Beim Speichern der Einstellungen ist ein Fehler aufgetreten",
					e));
		}
	}

	/**
	 * Ermittelt den aktuell verwendeten Namen.
	 *
	 * @return den Namen
	 */
	protected abstract String getCurrentName();

	/**
	 * Gibt das zu speichernde Objekt zurück.<br>
	 *
	 * <b>Hier sollte eine Kopie erstellt werden!</b>
	 *
	 * @param newName
	 *            der neue Name
	 * @return das Objekt
	 */
	protected abstract T getObject(String newName);

	/**
	 * Wird nach der erfolgten Speicherung aufgerufen.
	 *
	 * @param newName
	 *            der neue Name unter dem das Objekt abgelegt wurde.
	 */
	protected void saveHook(final String newName) {
		// stub
	}

	/**
	 * Erstellt ein Objekt vom Typ {@link ModellEinstellungen}.
	 *
	 * @return das Objekt
	 */
	protected abstract ModellEinstellungen<N, T> createModelEinstellungen();
}
