/*
 * Rahmenwerk-Plug-in "BitCtrl-Bibliotheken"
 *
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.rw.bitctrl.eclipse.modell.emf;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.eclipse.core.runtime.Assert;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ITreeSelection;
import org.eclipse.jface.viewers.TreePath;
import org.eclipse.ui.IWorkbenchPart;

import com.bitctrl.lib.eclipse.emf.eclipse.model.Named;

import de.bsvrz.buv.rw.basislib.einstellungen.EinstellungOwnerType;
import de.bsvrz.buv.rw.basislib.einstellungen.SpeicherKey;
import de.bsvrz.buv.rw.bitctrl.eclipse.util.RahmenwerkUtils;

/**
 * Basisklasse für Aktionen die EMF-Einstellungen manipulieren.
 *
 * Der Workbench Part der Aktion muss einen Selection Provider besitzen, der
 * eine {@link ITreeSelection} mit zwei Ebenen bereitstellt. Die erste Ebene ist
 * die {@link EinstellungsArt} und die zweite ist die Einstellung selbst. Die
 * Einstellung muss von {@link Named} abgeleitet sein.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 * @param <T>
 *            Der Typ der Einstellung der manipuliert werden soll.
 */
class BaseEinstellungAction<T extends Named> extends Action {

	private final IWorkbenchPart workbenchPart;
	private final EMFModellEinstellungen<T> einstellungen;

	/**
	 * Initialisiert die Aktion.
	 *
	 * @param workbenchPart
	 *            der Workbench Part für den die Aktion arbeitet. Darf nicht
	 *            <code>null</code> sein.
	 * @param einstellungen
	 *            der Einstellungsspeicher der manipuliert werden soll. Darf
	 *            nicht <code>null</code> sein.
	 */
	BaseEinstellungAction(final IWorkbenchPart workbenchPart,
			final EMFModellEinstellungen<T> einstellungen) {
		Assert.isNotNull(workbenchPart, "workbenchPart darf nicht null sein");
		Assert.isNotNull(einstellungen, "einstellungen darf nicht null sein");

		this.workbenchPart = workbenchPart;
		this.einstellungen = einstellungen;
	}

	/**
	 * Gibt den Workbench Part zurück, zu dem die Aktion gehört.
	 *
	 * @return der Workbench Part der Action.
	 */
	IWorkbenchPart getWorkbenchPart() {
		return workbenchPart;
	}

	/**
	 * Gibt den Einstellungsspeicher zurück, auf dem die Aktion arbeitet.
	 *
	 * @return der Einstellungspeicher auf dem die Action arbeitet.
	 */
	EMFModellEinstellungen<T> getEinstellungen() {
		return einstellungen;
	}

	/**
	 * Gibt die Liste von vom Benutzer im Baum ausgewählten Pfaden zu lesbaren
	 * Einstellungen zurück. Lesbar heißt hier <em>alle</em> Einstellungen bzw.
	 * alle Pfade die aus zwei Elementen bestehen und deren erstes Element eine
	 * {@link EinstellungsArt} und deren zweites Element ein {@link String} ist.
	 *
	 * @return alle lesbaren Einstellungen als Liste von Pfaden.
	 */
	List<TreePath> getReadableEinstellungen() {
		return getSelectedEinstellungen(false);
	}

	/**
	 * Gibt die Liste von vom Benutzer im Baum ausgewählten Pfaden zu
	 * schreibbaren Einstellungen zurück. Die Liste der Pfade ist eine Teilmenge
	 * von {@link #getReadableEinstellungen()}. Verfügt der angemeldete Benutzer
	 * über Administratorrechte, sieht er die selbe Liste. Hat der angemeldete
	 * Benutzer keine Administratorrechte, enthält die Liste nur die Pfade mit
	 * der Einstellunsgart {@link EinstellungsArt#benutzerLokal} und
	 * {@link EinstellungsArt#benutzerNetzweit}. Die beiden Einstellungsarten
	 * {@link EinstellungsArt#allgemeinLokal} und
	 * {@link EinstellungsArt#allgemeinNetzweit} darf nur der Administrator
	 * schreiben.
	 *
	 * @return alle schreibbaren Einstellungen als Liste von Pfaden.
	 * @see #getReadableEinstellungen()
	 */
	List<TreePath> getWritableEinstellungen() {
		return getSelectedEinstellungen(true);
	}

	/**
	 * @see #getReadableEinstellungen()
	 * @see #getWritableEinstellungen()
	 */
	private List<TreePath> getSelectedEinstellungen(
			final boolean onlyWritable) {
		final ISelection selection = workbenchPart.getSite()
				.getSelectionProvider().getSelection();
		if (selection.isEmpty() || !(selection instanceof ITreeSelection)) {
			return Collections.emptyList();
		}

		final List<TreePath> result = new ArrayList<>();
		for (final TreePath path : ((ITreeSelection) selection).getPaths()) {
			final Named einstellung = EinstellungenHelper.getEinstellung(path);
			final SpeicherKey einstellungsArt = EinstellungenHelper
					.getEinstellungsArt(path);

			if (onlyWritable && !RahmenwerkUtils.isAdmin() && (einstellungsArt
					.getOwnerType() == EinstellungOwnerType.SYSTEM)) {
				continue;
			}

			if (einstellung != null && einstellungsArt != null) {
				result.add(path);
			}
		}

		return result;
	}

}
