/*
 * Rahmenwerk-Plug-in "BitCtrl-Bibliotheken"
 *
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.rw.bitctrl.eclipse.modell.emf;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TreePath;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchCommandConstants;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.dialogs.ElementListSelectionDialog;

import com.bitctrl.lib.eclipse.emf.eclipse.model.Named;

import de.bsvrz.buv.rw.basislib.einstellungen.EinstellungOwnerType;
import de.bsvrz.buv.rw.basislib.einstellungen.SpeicherKey;
import de.bsvrz.buv.rw.bitctrl.eclipse.util.RahmenwerkUtils;
import de.bsvrz.buv.rw.bitctrl.internal.BitCtrlRahmenwerkPlugin;

/**
 * Aktion zum Kopieren (Duplizieren) von EMF-Einstellungen.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 * @param <T>
 *            Der Typ der Einstellung der manipuliert werden soll.
 * @see #run()
 */
public class CopyEinstellungAction<T extends Named>
		extends BaseEinstellungAction<T> {

	/**
	 * Initialisiert die Action.
	 *
	 * @param part
	 *            der Workbench Part für den die Action arbeitet. Darf nicht
	 *            <code>null</code> sein.
	 * @param einstellungen
	 *            der Einstellungsspeicher der manipiliert werden soll. Darf
	 *            nicht <code>null</code> sein.
	 */
	public CopyEinstellungAction(final IWorkbenchPart part,
			final EMFModellEinstellungen<T> einstellungen) {
		super(part, einstellungen);

		setId(IWorkbenchCommandConstants.EDIT_COPY);
		setActionDefinitionId(IWorkbenchCommandConstants.EDIT_COPY);
	}

	/**
	 * Führt das Kopieren (Duplizieren) der ausgewählten Benutzereinstellungen
	 * durch.
	 *
	 * <p>
	 * Eine Prüfung, ob die ausgewählten Einstellungen schreibar sind, ist nicht
	 * notwendig, da die Originaleinstellungen unangetastet bleiben.
	 *
	 * <p>
	 * Der Benutzer wählt als Ziel die Einstellungsart, wo die Kopie(n) abgelegt
	 * werden sollen. Zur Auswahl stehen nur die schreibbaren Einstellungsarten.
	 * Ist am Ziel bereits eine Einstellung mit dem selben Namen vorhanden wird
	 * "Kopie von " dem Namen der kopierten Einstellung vorangestellt. Ist auch
	 * bereits dieser Name schon vergeben, wird nach dem Muster "Kopie (2) von "
	 * solange probiert, bis ein noch nicht verwendeter Name gefunden wurde.
	 *
	 * <p>
	 * Evtl. geöffnete Editoren zu den Einstellungen werden nicht angetastet.
	 */
	@Override
	public void run() {
		final List<TreePath> readable = getReadableEinstellungen();

		Assert.isTrue(!readable.isEmpty(),
				"Kopieren (Duplizieren) kann nicht ausgeführt werden, wenn nichts markiert ist.");

		final EClass classname = getEinstellungen().getEClass();
		final String dialogTitle = classname.getName() + " kopieren";
		final String dialogMessage = "Bitte geben an wohin die Objekte kopiert werden sollen?";

		// Abhängig von den Benutzerrechten die Liste der Ziele aufstellen
		final List<SpeicherKey> einstellungsArten = new ArrayList<>();
		for (final SpeicherKey key : SpeicherKey.getDefaultKeys()) {
			if ((key.getOwnerType() == EinstellungOwnerType.SYSTEM)
					&& (!RahmenwerkUtils.isAdmin())) {
				continue;
			}
			einstellungsArten.add(key);
		}

		// Nach dem Ziel fragen
		final Shell shell = getWorkbenchPart().getSite().getShell();
		final ElementListSelectionDialog dlg = new ElementListSelectionDialog(
				shell, new LabelProvider());
		dlg.setTitle(dialogTitle);
		dlg.setMessage(dialogMessage);
		dlg.setMultipleSelection(false);
		dlg.setElements(einstellungsArten.toArray());

		// Objekte kopieren
		if (dlg.open() == Window.OK) {
			final Job job = new Job(dialogTitle) {

				@SuppressWarnings("unchecked")
				@Override
				protected IStatus run(final IProgressMonitor monitor) {
					monitor.beginTask(dialogTitle, readable.size());

					final SpeicherKey zielArt = (SpeicherKey) dlg
							.getFirstResult();

					for (final TreePath path : readable) {
						final Named einstellungQuelle = EinstellungenHelper
								.getEinstellung(path);

						final T einstellungZiel = (T) EcoreUtil
								.copy(einstellungQuelle);

						if (EcoreUtil.getID(einstellungZiel) != null) {
							// die Kopie hat eine andere ID
							EcoreUtil.setID(einstellungZiel,
									EcoreUtil.generateUUID());
						}
						final String name = einstellungZiel.getName();

						int index = 0;
						while (getEinstellungen().getModellEinstellungen(
								zielArt, einstellungZiel.getName()) != null) {
							if (index == 0) {
								final String neuerName = "Kopie von " + name;
								einstellungZiel.setName(neuerName);
							} else {
								final String neuerName = "Kopie (" + index
										+ ") von " + name;
								einstellungZiel.setName(neuerName);
							}
							++index;
						}

						try {
							getEinstellungen().setModellEinstellungen(zielArt,
									einstellungZiel.getName(), einstellungZiel);
						} catch (final IOException ex) {
							BitCtrlRahmenwerkPlugin.getDefault().getLog()
									.log(new Status(IStatus.WARNING,
											BitCtrlRahmenwerkPlugin.PLUGIN_ID,
											"Speichern des Objektes "
													+ einstellungZiel.getName()
													+ " fehlgeschlagen.",
											ex));
						}

						monitor.worked(1);
					}

					monitor.done();
					return Status.OK_STATUS;
				}

			};
			job.setUser(true);
			job.schedule();
		}
	}

}
