/*
 * Rahmenwerk-Plug-in "BitCtrl-Bibliotheken"
 *
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.rw.bitctrl.eclipse.modell.emf;

import java.util.List;

import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.viewers.TreePath;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchCommandConstants;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchPartSite;

import com.bitctrl.lib.eclipse.emf.eclipse.model.Named;
import com.bitctrl.util.CollectionUtilities;

/**
 * Aktion zum Löschen von EMF-Einstellungen.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 * @param <T>
 *            Der Typ der Einstellung der manipuliert werden soll.
 * @see #run()
 */
public class DeleteEinstellungAction<T extends Named>
		extends BaseEinstellungAction<T> {

	/**
	 * Spezieller Dialog zum Löschen von Einstellungen, der auch große Mengen
	 * von zu löschenden Objekten darstellen kann.
	 *
	 * @author BitCtrl Systems GmbH, ChHoesel
	 */
	private final class DeleteMessageDialog extends MessageDialog {
		private DeleteMessageDialog(final Shell parentShell,
				final String dialogTitle, final Image dialogTitleImage,
				final String dialogMessage, final int dialogImageType,
				final int defaultIndex, final String[] dialogButtonLabels) {
			super(parentShell, dialogTitle, dialogTitleImage, dialogMessage,
					dialogImageType, defaultIndex, dialogButtonLabels);
		}

		@Override
		protected int getShellStyle() {
			return super.getShellStyle() | SWT.SHEET;
		}

		@Override
		protected Control createMessageArea(final Composite composite) {
			final Image image = getImage();
			if (image != null) {
				imageLabel = new Label(composite, SWT.NULL);
				image.setBackground(imageLabel.getBackground());
				imageLabel.setImage(image);
				GridDataFactory.fillDefaults().align(SWT.CENTER, SWT.BEGINNING)
						.applyTo(imageLabel);
			}
			// create message
			if (message != null) {

				final ScrolledComposite scroll = new ScrolledComposite(
						composite, SWT.V_SCROLL | SWT.H_SCROLL);
				scroll.setLayout(new GridLayout());
				scroll.setLayoutData(GridDataFactory.fillDefaults()
						.grab(true, true).create());

				messageLabel = new Label(scroll, getMessageLabelStyle());
				messageLabel.setText(message);
				GridDataFactory.fillDefaults().align(SWT.FILL, SWT.BEGINNING)
						.grab(true, false)
						.hint(convertHorizontalDLUsToPixels(
								IDialogConstants.MINIMUM_MESSAGE_AREA_WIDTH),
								SWT.DEFAULT)
						.applyTo(messageLabel);
				scroll.setContent(messageLabel);
				scroll.setExpandHorizontal(true);
				scroll.setExpandVertical(true);
				scroll.setMinSize(
						messageLabel.computeSize(SWT.DEFAULT, SWT.DEFAULT));
			}
			return composite;
		}
	}

	/**
	 * Initialisiert die Action.
	 *
	 * @param part
	 *            der Workbench Part für den die Action arbeitet. Darf nicht
	 *            <code>null</code> sein.
	 * @param einstellungen
	 *            der Einstellungsspeicher der manipiliert werden soll. Darf
	 *            nicht <code>null</code> sein.
	 */
	public DeleteEinstellungAction(final IWorkbenchPart part,
			final EMFModellEinstellungen<T> einstellungen) {
		super(part, einstellungen);

		setId(IWorkbenchCommandConstants.EDIT_DELETE);
		setActionDefinitionId(IWorkbenchCommandConstants.EDIT_DELETE);
	}

	/**
	 * Führt das Löschen der ausgewählten Benutzereinstellungen durch.
	 *
	 * <p>
	 * Prüft als erstes, ob alle ausgewählten Einstellungen schreibar (und damit
	 * löschbar) sind. Die nicht löschbaren Einstellungen werden als Hinweis
	 * angezeigt und die löschbaren gelöscht.
	 *
	 * <p>
	 * Vor dem eigentlichen Löschen wird der Benutzer nochmal gefragt, ob er
	 * sich sicher ist.
	 *
	 * <p>
	 * Ist ein Editor für eine zu löschende Einstellung offen, wird dieser vor
	 * dem Löschen geschlossen.
	 */
	@Override
	public void run() {
		final List<TreePath> readable = getReadableEinstellungen();
		final List<TreePath> writable = getWritableEinstellungen();

		Assert.isTrue(!readable.isEmpty(),
				"Löschen kann nicht ausgeführt werden, wenn nichts markiert ist.");

		final Shell shell = getWorkbenchPart().getSite().getShell();

		// Ein Teil kann nicht gelöscht werden.
		if (readable.size() != writable.size()) {
			final List<TreePath> unwritable = CollectionUtilities
					.difference(readable, writable);
			MessageDialog.openInformation(shell, "Hinweis",
					"Sie können die folgenden Objekte nicht löschen, weil Sie nicht über die notwendigen Rechte verfügen:\n\n"
							+ EinstellungenHelper.getObjectList(unwritable));
		}

		// Es kann gar nichts gelöscht werden.
		if (writable.isEmpty()) {
			return;
		}

		// Löschen vom Benutzer bestätigen lassen und dann los ...
		final String classname = getEinstellungen().getEClass().getName();
		final String dialogTitle = classname + " löschen";

		final String dialogMessage = "Sollen die folgenden Objekte wirklich gelöscht werden? Die Aktion kann nicht rückgängig gemacht werden.\n\n"
				+ EinstellungenHelper.getObjectList(writable);
		final MessageDialog dailog = new DeleteMessageDialog(shell, dialogTitle,
				null, dialogMessage, MessageDialog.QUESTION, 0,
				new String[] { IDialogConstants.YES_LABEL,
						IDialogConstants.NO_LABEL });

		if (dailog.open() == Window.OK) {
			final Job job = new Job(dialogTitle) {

				@Override
				protected IStatus run(final IProgressMonitor monitor) {
					monitor.beginTask(dialogTitle, writable.size());

					final IWorkbenchPartSite site = getWorkbenchPart()
							.getSite();
					site.getShell().getDisplay().syncExec(() -> {
						for (final TreePath path : writable) {
							EinstellungenHelper.delete(getEinstellungen(),
									path);
							monitor.worked(1);
						}
					});

					monitor.done();
					return Status.OK_STATUS;
				}

			};
			job.setUser(true);
			job.schedule();
		}

	}
}
