/*
 * Rahmenwerk-Plug-in "BitCtrl-Bibliotheken"
 *
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.rw.bitctrl.eclipse.modell.emf;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.Resource.Diagnostic;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.xmi.impl.XMIResourceFactoryImpl;

import de.bsvrz.buv.rw.bitctrl.eclipse.modell.ModellEinstellungen;
import de.bsvrz.buv.rw.bitctrl.eclipse.modell.emf.storage.Named;
import de.bsvrz.buv.rw.bitctrl.eclipse.modell.emf.storage.StorageFactory;
import de.bsvrz.buv.rw.bitctrl.eclipse.modell.emf.storage.StoragePackage;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Enthaelt Hilfsmethoden fuer die Verwendung von EMF Modellen als
 * Einstellungsspeicher.
 *
 * @author BitCtrl Systems GmbH, schnepel
 *
 * @param <T>
 *            der gespeicherte Typ
 */
public class EMFModellEinstellungen<T extends EObject>
		extends ModellEinstellungen<Named, T> {

	private static final Debug LOGGER = Debug.getLogger();

	private static final long serialVersionUID = 1L;

	private static final String FILE_EXTENSION = "xmi";
	private static final String DUMMY_URL = "dummy:/dummy." + FILE_EXTENSION;

	static {
		final Resource.Factory.Registry reg = Resource.Factory.Registry.INSTANCE;
		final Map<String, Object> m = reg.getExtensionToFactoryMap();
		Object object = m.get(FILE_EXTENSION);
		if (null == object) {
			object = new XMIResourceFactoryImpl();
			m.put(FILE_EXTENSION, object);
		}
		// just load the class
		LOGGER.finest("Loaded Storageprovider "
				+ StoragePackage.eINSTANCE.getNsURI());
	}

	private final EClass eClass;

	/**
	 * Konstruktor.
	 *
	 * @param type
	 *            Klasse des gespeicherten Typs
	 * @deprecated {@link #EMFModellEinstellungen(Class, EClass)} verwenden
	 */
	@Deprecated
	public EMFModellEinstellungen(final Class<T> type) {
		this(type, null);
	}

	/**
	 * Initialisiert die EMF-Einstellungen.
	 *
	 * @param type
	 *            der Java-Typ der zu speichernden Objekte.
	 * @param eClass
	 *            der EMF-Typ der zu speichernden Objekte.
	 */
	public EMFModellEinstellungen(final Class<T> type, final EClass eClass) {
		super(type);
		this.eClass = eClass;
	}

	/**
	 * Gibt die Klasse der gespeicherten Objekt zurück.
	 *
	 * @return der EMF-Typ der gesicherten Objekte.
	 */
	public EClass getEClass() {
		return eClass;
	}

	@SuppressWarnings("unchecked")
	@Override
	protected T getObject(final Named named) {
		return (T) named.getObject();
	}

	@Override
	protected void setObject(final Named named, final T object) {
		named.setObject(object);
	}

	@Override
	protected String getName(final Named named) {
		return named.getName();
	}

	@Override
	protected void setName(final Named named, final String name) {
		named.setName(name);
	}

	@Override
	protected void setLastModification(final Named named,
			final Date lastModification) {
		named.setLastModification(lastModification);
	}

	@Override
	protected Date getLastModification(final Named named) {
		return named.getLastModification();
	}

	@Override
	protected Named createStorage() {
		return StorageFactory.eINSTANCE.createNamed();
	}

	@Override
	protected String saveModellEinstellungenInternal(
			final List<Named> contents) {
		final Iterator<Named> iterator = contents.iterator();
		while (iterator.hasNext()) {
			if (iterator.next().getObject() == null) {
				iterator.remove();
			}
		}

		final ByteArrayOutputStream os = new ByteArrayOutputStream();
		try {
			final ResourceSet resSet = new ResourceSetImpl();
			final Resource resource = resSet
					.createResource(URI.createURI(DUMMY_URL));
			resource.getContents().addAll(contents);
			resource.save(os, null);
		} catch (final IOException ex) {
			handleWriteError(ex, null);
		}
		final String param = os.toString();
		return param;
	}

	@SuppressWarnings({ "unchecked", "rawtypes" })
	@Override
	protected List<Named> getModellEinstellungenInternal(final String param) {
		final ResourceSet resSet = new ResourceSetImpl();
		final Resource resource = resSet
				.createResource(URI.createURI(DUMMY_URL));
		if (param != null) {
			final InputStream os = new ByteArrayInputStream(param.getBytes());
			try {
				resource.load(os, null);
			} catch (final IOException ex) {
				handleReadError(ex, null);
			}
		}

		return (List) resource.getContents();
	}

	/**
	 * Liefert alle Probleme und Fehler, die der EMF Parser findet.
	 *
	 * @param param
	 *            die serialisierte Einstellung
	 *
	 * @return die Menge der Fehler und Warnungen
	 *
	 */
	protected List<Diagnostic> getModellEinstellungenDiagnostic(
			final String param) {
		final ResourceSet resSet = new ResourceSetImpl();
		final Resource resource = resSet
				.createResource(URI.createURI(DUMMY_URL));
		if (param != null) {
			final InputStream os = new ByteArrayInputStream(param.getBytes());
			try {
				resource.load(os, null);
			} catch (final IOException ex) {

				final List<Diagnostic> result = new ArrayList<>();
				result.addAll(resource.getErrors());
				result.addAll(resource.getWarnings());
				return result;
			}
		}

		return Collections.emptyList();
	}

}
