/*
 * Rahmenwerk-Plug-in "BitCtrl-Bibliotheken"
 *
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.rw.bitctrl.eclipse.modell.emf;

import java.io.IOException;
import java.util.List;

import org.eclipse.jface.viewers.TreePath;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorReference;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PlatformUI;

import com.bitctrl.lib.eclipse.emf.eclipse.model.Named;

import de.bsvrz.buv.rw.basislib.einstellungen.SpeicherKey;

final class EinstellungenHelper {

	private EinstellungenHelper() {
		// utility class
	}

	static void delete(final EMFModellEinstellungen<?> einstellungen,
			final TreePath path) {
		IEditorPart editorPart = getEditorPart(path);
		final IWorkbenchPage page = PlatformUI.getWorkbench()
				.getActiveWorkbenchWindow().getActivePage();
		while (editorPart != null) {
			page.closeEditor(editorPart, false);
			editorPart = getEditorPart(path);
		}

		final SpeicherKey einstellungsArt = getEinstellungsArt(path);
		final String name = getEinstellung(path).getName();
		try {
			einstellungen.setModellEinstellungen(einstellungsArt, name, null,
					null);
		} catch (final IOException ex) {
			// TODO Auto-generated catch block
			ex.printStackTrace();
		}
	}

	/**
	 * Extrahiert die Einstellung aus einem Pfad und gibt sie zurück. Die
	 * Einstellung ist das letzte Segment des Pfades.
	 *
	 * @param path
	 *            ein Pfad, darf nicht <code>null</code> sein.
	 * @return die Einstellung oder <code>null</code>, wenn es keine gibt.
	 *         Letzteres ist z.&nbsp;B. der Fall, wenn anstelle eines
	 *         Einstellungsnamens eine Einstellungsart im Baum ausgewählt wurde.
	 */
	static Named getEinstellung(final TreePath path) {
		if (path.getLastSegment() instanceof Named) {
			return (Named) path.getLastSegment();
		}

		return null;
	}

	/**
	 * Extrahiert die Einstellungsart aus einem Pfad und gibt sie zurück. Die
	 * Einstellungsart ist das erste Segment om Pfad.
	 *
	 * @param path
	 *            ein Pfad, darf nicht <code>null</code> sein.
	 * @return die Einstellungsart oder <code>null</code>, wenn es keine gibt.
	 *         Letzteres ist z.&nbsp;B. der Fall, wenn es im Baum ein
	 *         Wurzelobjekt gibt. Dieses hat keine Einstellungsart, weil die
	 *         Einstellungsarten unterhalb der Wurzel liegen.
	 */
	static SpeicherKey getEinstellungsArt(final TreePath path) {
		if (path.getFirstSegment() instanceof SpeicherKey) {
			return (SpeicherKey) path.getFirstSegment();
		}

		return null;
	}

	/**
	 * Gibt den Editor zu einem Pfad zurück. Wenn der Pfad eine Einstellung
	 * addressiert, wird falls vorhanden ein geöffneter Editor dazu
	 * zurückgegeben. Sollte es mehrere Editoren geben, wird irgendeiner
	 * zurückgegeben.
	 *
	 * <p>
	 * Das Finden eines Editors funktioniert nur, wenn dessen
	 * {@link IEditorInput} jeweils einen Adapter für {@link EinstellungsArt}
	 * und für den Typ des bearbeiteten Objekts (abgeleitet von {@link Named})
	 * zurück gibt.
	 *
	 * @param path
	 *            ein Pfad, darf nicht <code>null</code> sein.
	 * @return ein geöffneter Editor oder <code>null</code>, wenn es keinen
	 *         gibt.
	 */
	private static IEditorPart getEditorPart(final TreePath path) {
		final Named einstellung = getEinstellung(path);
		final SpeicherKey art = getEinstellungsArt(path);

		if (einstellung == null || art == null) {
			return null;
		}

		final IWorkbenchPage page = PlatformUI.getWorkbench()
				.getActiveWorkbenchWindow().getActivePage();
		for (final IEditorReference e : page.getEditorReferences()) {
			// Ist der referenzierte Editor geöffnet?
			final IEditorPart editorPart = e.getEditor(false);
			if (editorPart == null) {
				continue;
			}

			// Beinhaltet der Editor Input eine Benutzereinstellung?
			final IEditorInput input = editorPart.getEditorInput();
			final Named editorEinsgtellung = input
					.getAdapter(einstellung.getClass());
			final SpeicherKey editorArt = input.getAdapter(SpeicherKey.class);
			if (editorEinsgtellung == null || editorArt == null) {
				continue;
			}

			// Ist es die gesuchte Benutzereinstellung?
			if (einstellung.getName().equals(editorEinsgtellung.getName())
					&& art.equals(editorArt)) {
				return editorPart;
			}
		}

		return null;
	}

	/**
	 * Gibt einen lesbaren String als zeilenweise Liste der Pfade zurück. Ein
	 * Pfad wird als "Einstellungsart: Einstellungsname" ausgegeben. Zwischen
	 * den Listenelementen wird jeweils ein Zeilenumbruch eingefügt.
	 *
	 * @param paths
	 *            die Liste der Pfade die als Stringliste ausgegeben werden
	 *            sollen. Darf nicht <code>null</code> sein.
	 * @return die Pfade als Stringliste.
	 */
	static String getObjectList(final List<TreePath> paths) {
		final StringBuilder result = new StringBuilder();
		for (final TreePath path : paths) {
			if (result.length() > 0) {
				result.append('\n');
			}

			result.append(EinstellungenHelper.getEinstellungsArt(path));
			result.append(": ");
			final Named named = EinstellungenHelper.getEinstellung(path);
			if (named != null) {
				result.append(named.getName());
			} else {
				result.append("unbekannt");
			}
		}
		return result.toString();
	}
}
