/*
 * Rahmenwerk-Plug-in "BitCtrl-Bibliotheken"
 *
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.rw.bitctrl.eclipse.modell.emf;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.IInputValidator;
import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TreePath;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchCommandConstants;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.dialogs.ElementListSelectionDialog;

import com.bitctrl.lib.eclipse.emf.eclipse.model.Named;
import com.bitctrl.util.CollectionUtilities;

import de.bsvrz.buv.rw.basislib.einstellungen.EinstellungOwnerType;
import de.bsvrz.buv.rw.basislib.einstellungen.SpeicherKey;
import de.bsvrz.buv.rw.basislib.einstellungen.UrlasserDialogAbgebrochenException;
import de.bsvrz.buv.rw.bitctrl.eclipse.util.RahmenwerkUtils;
import de.bsvrz.buv.rw.bitctrl.internal.BitCtrlRahmenwerkPlugin;

/**
 * Aktion zum Verschieben von Einstellungen.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 * @param <T>
 *            Der Typ der Einstellung der manipuliert werden soll.
 * @see #run()
 */
public class MoveEinstellungAction<T extends Named>
		extends BaseEinstellungAction<T> {

	private final class EinstellungInputValidator implements IInputValidator {
		private final String name;
		private final SpeicherKey zielArt;

		private EinstellungInputValidator(final String name,
				final SpeicherKey zielArt) {
			this.name = name;
			this.zielArt = zielArt;
		}

		@Override
		public String isValid(final String newText) {
			if (newText.isEmpty()) {
				return "Der Name muss mindestens ein Zeichen lang sein.";
			} else if (getEinstellungen().getModellEinstellungen(zielArt,
					newText) != null) {
				return "Es existiert bereits ein Objekt mit dem angegebenen Namen.";
			} else if (newText.equals(name)) {
				return "Der neue Name ist identisch mit dem alten.";
			}

			return null;
		}
	}

	/**
	 * Initialisiert die Action.
	 *
	 * @param part
	 *            der Workbench Part für den die Action arbeitet. Darf nicht
	 *            <code>null</code> sein.
	 * @param einstellungen
	 *            der Einstellungsspeicher der manipiliert werden soll. Darf
	 *            nicht <code>null</code> sein.
	 */
	public MoveEinstellungAction(final IWorkbenchPart part,
			final EMFModellEinstellungen<T> einstellungen) {
		super(part, einstellungen);

		setId(IWorkbenchCommandConstants.FILE_MOVE);
		setActionDefinitionId(IWorkbenchCommandConstants.FILE_MOVE);
	}

	/**
	 * Führt das Verschieben der ausgewählten Benutzereinstellungen durch.
	 *
	 * <p>
	 * Prüft als erstes, ob alle ausgewählten Einstellungen schreibar sind (und
	 * damit umbenannt werden können). Die nicht schreibaren Einstellungen
	 * werden als Hinweis angezeigt und die schreibaren umbenannt.
	 *
	 * <p>
	 * Der Benutzer wählt als Ziel die Einstellungsart, wohin die Einstellungen
	 * verschoben werden sollen. Zur Auswahl stehen nur die schreibbaren
	 * Einstellungsarten. Ist am Ziel bereits eine Einstellung mit dem selben
	 * Namen vorhanden wird der Benutzer gefragt, ob er diese überschreiben
	 * möchte. Wenn nicht kann er einen alternativen Namen angeben. Der neue
	 * Name muss mindestens ein Zeichen lang sein und darf nicht identisch mit
	 * einem vorhandenen Namen sein.
	 *
	 * <p>
	 * Ist ein Editor für eine zu verschiebende Einstellung offen, wird dieser
	 * vor dem verschieben geschlossen.
	 */
	@Override
	public void run() {
		final List<TreePath> readable = getReadableEinstellungen();
		final List<TreePath> writable = getWritableEinstellungen();

		Assert.isTrue(!readable.isEmpty(),
				"Verschieben kann nicht ausgeführt werden, wenn nichts markiert ist.");

		final Shell shell = getWorkbenchPart().getSite().getShell();

		// Ein Teil kann nicht verschoben werden.
		if (readable.size() != writable.size()) {
			final List<TreePath> unwritable = CollectionUtilities
					.difference(readable, writable);
			MessageDialog.openInformation(shell, "Hinweis",
					"Sie können die folgenden Objekte nicht verschieben, weil Sie nicht über die notwendigen Rechte verfügen:\n\n"
							+ EinstellungenHelper.getObjectList(unwritable));
		}

		// Es kann gar nichts verschoben werden.
		if (writable.isEmpty()) {
			return;
		}

		final String classname = getEinstellungen().getEClass().getName();
		final String dialogTitle = classname + " verschieben";
		final String dialogMessage = "Bitte geben an wohin die Objekte verschoben werden sollen?";

		// Abhängig von den Benutzerrechten die Liste der Ziele aufstellen
		final List<SpeicherKey> einstellungsArten = new ArrayList<>();
		for (final SpeicherKey key : SpeicherKey.getDefaultKeys()) {
			if ((key.getOwnerType() == EinstellungOwnerType.SYSTEM)
					&& (!RahmenwerkUtils.isAdmin())) {
				continue;
			}
			einstellungsArten.add(key);
		}

		// Nach dem Ziel fragen
		final ElementListSelectionDialog dlg = new ElementListSelectionDialog(
				shell, new LabelProvider());
		dlg.setTitle(dialogTitle);
		dlg.setMessage(dialogMessage);
		dlg.setMultipleSelection(false);
		dlg.setElements(einstellungsArten.toArray());

		// Objekte verschieben
		if (dlg.open() == Window.OK) {
			final Job job = new Job(dialogTitle) {

				@SuppressWarnings("unchecked")
				@Override
				protected IStatus run(final IProgressMonitor monitor) {
					monitor.beginTask(dialogTitle, writable.size());

					final SpeicherKey zielArt = (SpeicherKey) dlg
							.getFirstResult();

					for (final TreePath path : writable) {

						final T einstellungQuelle = (T) EinstellungenHelper
								.getEinstellung(path);
						final String name = einstellungQuelle.getName();

						/*
						 * Prüfen ob am Ziel bereits eine Einstellung mit dem
						 * selben existiert
						 */
						if (getEinstellungen().getModellEinstellungen(zielArt,
								einstellungQuelle.getName()) != null) {
							// Wenn ja, erst den neuen Namen erfragen
							final InputDialog renameDialog = new InputDialog(
									shell, dialogTitle,
									"Am Ziel existiert bereits eine Einstellung mit dem selben Namen. Bitte geben Sie einen neuen Namen für "
											+ name + " an",
									name, new EinstellungInputValidator(name,
											zielArt));
							shell.getDisplay().syncExec(() -> {
								if (renameDialog.open() == Window.OK) {
									doMove(monitor, path, zielArt,
											renameDialog.getValue());
								}
							});
						} else {
							// Wenn nein, Einstellungen verschieben
							doMove(monitor, path, zielArt, name);
						}

						try {
							getEinstellungen().setModellEinstellungen(zielArt,
									einstellungQuelle.getName(),
									einstellungQuelle);
						} catch (final IOException ex) {
							BitCtrlRahmenwerkPlugin.getDefault().getLog().warn(
									"Fehler beim Speichern von Modell-Einstellungen.",
									ex);
						}

						monitor.worked(1);
					}

					monitor.done();
					return Status.OK_STATUS;
				}
			};
			job.setUser(true);
			job.schedule();
		}
	}

	@SuppressWarnings("unchecked")
	private void doMove(final IProgressMonitor monitor, final TreePath path,
			final SpeicherKey zielArt, final String neuerName) {

		final T einstellungQuelle = (T) EinstellungenHelper
				.getEinstellung(path);
		final String classname = getEinstellungen().getEClass().getName();

		final T einstellungZiel = EcoreUtil.copy(einstellungQuelle);
		einstellungZiel.setName(neuerName);
		if (EcoreUtil.getID(einstellungZiel) != null) {
			EcoreUtil.setID(einstellungZiel, EcoreUtil.generateUUID());
		}
		monitor.subTask(
				"Sichere Objekt " + classname + " unter seinem neuen Namen "
						+ einstellungZiel.getName() + " ...");
		try {
			getEinstellungen().setModellEinstellungen(zielArt,
					einstellungZiel.getName(), einstellungZiel);

			/*
			 * XXX Workaround, weil sich Vorgabe- und Sollparameter beim Ändern
			 * der Einstellung überholen können.
			 */
			monitor.subTask("Warte auf Datenverteiler ...");
			try {
				Thread.sleep(3000);
			} catch (final InterruptedException ex) {
				// kann ignoriert werden
			}

			monitor.subTask("Lösche altes Objekt " + classname
					+ " mit dem Namen " + einstellungQuelle.getName() + " ...");
			getWorkbenchPart().getSite().getShell().getDisplay().syncExec(
					() -> EinstellungenHelper.delete(getEinstellungen(), path));

		} catch (final UrlasserDialogAbgebrochenException ex) {
			// Bediener hat den Urlasserdialog abgebrochen, das ist
			// ein
			// normaler
			// Vorgang
		} catch (final IOException ex) {
			ErrorDialog.openError(
					PlatformUI.getWorkbench().getActiveWorkbenchWindow()
							.getShell(),
					"FEHLER", "Objekt konnte nicht verschoben werden!",
					new Status(IStatus.ERROR, BitCtrlRahmenwerkPlugin.PLUGIN_ID,
							ex.getLocalizedMessage(), ex));
		}
	}
}
