/*
 * Rahmenwerk-Plug-in "BitCtrl-Bibliotheken"
 *
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.rw.bitctrl.eclipse.modell.emf;

import java.io.IOException;
import java.util.List;

import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.IInputValidator;
import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.TreePath;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchCommandConstants;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PlatformUI;

import com.bitctrl.lib.eclipse.emf.eclipse.model.Named;
import com.bitctrl.util.CollectionUtilities;

import de.bsvrz.buv.rw.basislib.einstellungen.SpeicherKey;
import de.bsvrz.buv.rw.basislib.einstellungen.UrlasserDialogAbgebrochenException;
import de.bsvrz.buv.rw.bitctrl.internal.BitCtrlRahmenwerkPlugin;

/**
 * Aktion zum Umbenennen von EMF-Einstellungen.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 * @param <T>
 *            Der Typ der Einstellung der manipuliert werden soll.
 * @see #run()
 */
public class RenameEinstellungAction<T extends Named>
		extends BaseEinstellungAction<T> {

	private final class RenameInputValidator implements IInputValidator {
		private final SpeicherKey art;
		private final String nameQuelle;

		private RenameInputValidator(final SpeicherKey art,
				final String nameQuelle) {
			this.art = art;
			this.nameQuelle = nameQuelle;
		}

		@Override
		public String isValid(final String newText) {
			if (newText.isEmpty()) {
				return "Der Name muss mindestens ein Zeichen lang sein.";
			} else if (getEinstellungen().getModellEinstellungen(art,
					newText) != null) {
				return "Es existiert bereits ein Objekt mit dem angegebenen Namen.";
			} else if (newText.equals(nameQuelle)) {
				return "Der neue Name ist identisch mit dem alten.";
			}

			return null;
		}
	}

	/**
	 * Initialisiert die Action.
	 *
	 * @param part
	 *            der Workbench Part für den die Action arbeitet. Darf nicht
	 *            <code>null</code> sein.
	 * @param einstellungen
	 *            der Einstellungsspeicher der manipiliert werden soll. Darf
	 *            nicht <code>null</code> sein.
	 */
	public RenameEinstellungAction(final IWorkbenchPart part,
			final EMFModellEinstellungen<T> einstellungen) {
		super(part, einstellungen);

		setId(IWorkbenchCommandConstants.FILE_RENAME);
		setActionDefinitionId(IWorkbenchCommandConstants.FILE_RENAME);
	}

	/**
	 * Führt das Umbenennen der ausgewählten Benutzereinstellungen durch.
	 *
	 * <p>
	 * Prüft als erstes, ob alle ausgewählten Einstellungen schreibar sind (und
	 * damit umbenannt werden können). Die nicht schreibaren Einstellungen
	 * werden als Hinweis angezeigt und die schreibaren umbenannt.
	 *
	 * <p>
	 * Für jedes markierte Objekt, wird nach dem neuen Namen gefragt. Der neue
	 * Name muss mindestens ein Zeichen lang sein und darf nicht identisch mit
	 * dem neuen Namen sein.
	 *
	 * <p>
	 * Ist ein Editor für eine zu löschende Einstellung offen, wird dieser vor
	 * dem Löschen geschlossen.
	 */
	@SuppressWarnings("unchecked")
	@Override
	public void run() {
		final List<TreePath> readable = getReadableEinstellungen();
		final List<TreePath> writable = getWritableEinstellungen();

		Assert.isTrue(!readable.isEmpty(),
				"Umbenennen kann nicht ausgeführt werden, wenn nichts markiert ist.");

		final Shell shell = getWorkbenchPart().getSite().getShell();

		// Ein Teil kann nicht umbenannt werden.
		if (readable.size() != writable.size()) {
			final List<TreePath> unwritable = CollectionUtilities
					.difference(readable, writable);
			MessageDialog.openInformation(shell, "Hinweis",
					"Sie können die folgenden Objekte nicht umbenennen, weil Sie nicht über die notwendigen Rechte verfügen:\n\n"
							+ EinstellungenHelper.getObjectList(unwritable));
		}

		// Es kann gar nichts umbenannt werden.
		if (writable.isEmpty()) {
			return;
		}

		// Für jedes ausgewählte Objekt nach dem neuen Namen fragen und ändern.
		final String classname = getEinstellungen().getEClass().getName();
		final String dialogTitle = classname + " umbenennen";
		for (final TreePath path : writable) {

			final T einstellungQuelle = (T) EinstellungenHelper
					.getEinstellung(path);
			final String nameQuelle = einstellungQuelle.getName();
			final SpeicherKey art = EinstellungenHelper
					.getEinstellungsArt(path);

			final T einstellungZiel = EcoreUtil.copy(einstellungQuelle);
			if (EcoreUtil.getID(einstellungZiel) != null) {
				EcoreUtil.setID(einstellungZiel, EcoreUtil.generateUUID());
			}

			final String dialogMessage = "Bitte geben Sie den neuen Namen für "
					+ nameQuelle + " an.";

			// Neuen Namen erfragen
			final InputDialog dlg = new InputDialog(shell, dialogTitle,
					dialogMessage, nameQuelle,
					new RenameInputValidator(art, nameQuelle));
			if (dlg.open() == Window.OK) {
				einstellungZiel.setName(dlg.getValue());

				final Job job = new Job(dialogTitle) {

					@Override
					protected IStatus run(final IProgressMonitor monitor) {
						monitor.beginTask(dialogTitle + " von " + nameQuelle
								+ " nach " + einstellungZiel.getName(), 3);

						monitor.subTask("Sichere Objekt " + classname
								+ " unter seinem neuen Namen "
								+ einstellungZiel.getName() + " ...");
						try {
							getEinstellungen().setModellEinstellungen(art,
									einstellungZiel.getName(), einstellungZiel);
							monitor.worked(1);

							/*
							 * XXX Workaround, weil sich Vorgabe- und
							 * Sollparameter beim Ändern der Einstellung
							 * überholen können.
							 */
							monitor.subTask("Warte auf Datenverteiler ...");
							try {
								Thread.sleep(3000);
							} catch (final InterruptedException ex) {
								// kann ignoriert werden
							}
							monitor.worked(1);

							monitor.subTask("Lösche altes Objekt " + classname
									+ " mit dem Namen " + nameQuelle + " ...");
							getWorkbenchPart().getSite().getShell().getDisplay()
									.syncExec(() -> EinstellungenHelper
											.delete(getEinstellungen(), path));

							monitor.worked(1);

						} catch (final UrlasserDialogAbgebrochenException ex) {
							// Bediener hat den Urlasserdialog abgebrochen, das
							// ist
							// ein
							// normaler
							// Vorgang
						} catch (final IOException ex) {
							ErrorDialog.openError(PlatformUI.getWorkbench()
									.getActiveWorkbenchWindow().getShell(),
									"FEHLER",
									"Objekt konnte nicht umbenannt werden!",
									new Status(IStatus.ERROR,
											BitCtrlRahmenwerkPlugin.PLUGIN_ID,
											ex.getLocalizedMessage(), ex));
						}

						monitor.done();
						return Status.OK_STATUS;
					}
				};
				job.setUser(true);
				job.schedule();
			}
		}
	}

}
