/*
 * Rahmenwerk-Plug-in "BitCtrl-Bibliotheken"
 *
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.rw.bitctrl.eclipse.modell.java;

import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import com.google.gson.JsonSerializationContext;
import com.google.gson.JsonSerializer;

import de.bsvrz.buv.rw.bitctrl.eclipse.modell.ModellEinstellungen;
import de.bsvrz.buv.rw.bitctrl.eclipse.modell.java.storage.Named;
import de.bsvrz.buv.rw.bitctrl.eclipse.modell.java.storage.NamedObjectList;

/**
 * Enthaelt Hilfsmethoden fuer die Verwendung von Java Modellen als
 * Einstellungsspeicher.
 *
 * @author BitCtrl Systems GmbH, schnepel
 *
 * @param <T>
 *            der gespeicherte Typ
 */
public class JavaModellEinstellungen<T> extends ModellEinstellungen<Named, T> {

	private static final long serialVersionUID = 1L;
	private final List<Class<?>> contextClasses = new ArrayList<>();

	/**
	 * Adapter zum Serialisieren und Deserialisieren von {@link Named}-Objekten.
	 */
	private static class NamedObjectAdapter
			implements JsonSerializer<Named>, JsonDeserializer<Named> {

		private final Map<String, Class<?>> typeRegistry = new HashMap<>();

		// Methode zum Registrieren von Typen
		public void registerType(final String typeName, final Class<?> type) {
			typeRegistry.put(typeName, type);
		}

		@Override
		public JsonElement serialize(final Named wrapper, final Type typeOfSrc,
				final JsonSerializationContext context) {
			final JsonObject jsonObject = new JsonObject();
			final String typeName = wrapper.getObject().getClass().getName();
			jsonObject.addProperty("type", typeName);

			// Serialisiere das Objekt
			jsonObject.add("properties",
					context.serialize(wrapper.getObject()));
			jsonObject.addProperty("name", wrapper.getName());
			jsonObject.addProperty("lastModification",
					wrapper.getLastModification().getTime());

			return jsonObject;
		}

		@Override
		public Named deserialize(final JsonElement json, final Type typeOfT,
				final JsonDeserializationContext context)
				throws JsonParseException {
			final JsonObject jsonObject = json.getAsJsonObject();
			final String typeName = jsonObject.get("type").getAsString();

			final JsonElement properties = jsonObject.get("properties");

			// Finde die Klasse im Registry
			final Class<?> clazz = typeRegistry.get(typeName);
			if (clazz == null) {
				throw new JsonParseException("Unbekannter Typ: " + typeName);
			}

			// Deserialisiere das Objekt in den richtigen Typ
			final Named result = new Named();
			result.setObject(context.deserialize(properties, clazz));
			result.setName(jsonObject.get("name").getAsString());
			result.setLastModification(
					new Date(jsonObject.get("lastModification").getAsLong()));
			return result;
		}
	}

	/**
	 * Konstruktor.
	 *
	 * @param type
	 *            Klasse des gespeicherten Typs
	 */
	public JavaModellEinstellungen(final Class<T> type) {
		super(type);
		addContextClass(NamedObjectList.class);
		addContextClass(type);
	}

	public void addContextClass(final Class<?> clazz) {
		contextClasses.add(clazz);
	}

	@Override
	protected T getObject(final Named named) {
		return (T) named.getObject();
	}

	@Override
	protected void setObject(final Named named, final T object) {
		named.setObject(object);
	}

	@Override
	protected String getName(final Named named) {
		return named.getName();
	}

	@Override
	protected void setName(final Named named, final String name) {
		named.setName(name);
	}

	@Override
	protected void setLastModification(final Named named,
			final Date lastModification) {
		named.setLastModification(lastModification);
	}

	@Override
	protected Date getLastModification(final Named named) {
		return named.getLastModification();
	}

	@Override
	protected Named createStorage() {
		return new Named();
	}

	@Override
	protected List<Named> getModellEinstellungenInternal(final String param) {
		if (param != null) {

			try {
				final Gson gson = new GsonBuilder()
						.registerTypeAdapter(Named.class, createAdapter())
						.create();

				final NamedObjectList namedObjectList = gson.fromJson(param,
						NamedObjectList.class);
				return namedObjectList.getContents();
			} catch (final Exception ex) {
				handleReadError(ex, null);
			}
		}
		return new ArrayList<>();
	}

	@Override
	protected String saveModellEinstellungenInternal(
			final List<Named> contents) {
		final Iterator<Named> iterator = contents.iterator();
		while (iterator.hasNext()) {
			if (iterator.next().getObject() == null) {
				iterator.remove();
			}
		}

		try {
			final NamedObjectList namedObjectList = new NamedObjectList();
			namedObjectList.setContents(contents);
			// Gson-Instanz erstellen
			final Gson gson = new GsonBuilder()
					.registerTypeAdapter(Named.class, createAdapter()).create();
			return gson.toJson(namedObjectList);
		} catch (final Exception ex) {
			handleWriteError(ex, null);
		}

		return "";
	}

	private NamedObjectAdapter createAdapter() {
		final NamedObjectAdapter adapter = new NamedObjectAdapter();
		adapter.registerType(Named.class.getName(), Named.class);
		for (final Class<?> c : contextClasses) {
			adapter.registerType(c.getName(), c);
		}
		return adapter;
	}

}
