/*
 * Rahmenwerk-Plug-in "BitCtrl-Bibliotheken"
 *
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.rw.bitctrl.eclipse.viewers;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.Viewer;

import de.bsvrz.buv.rw.basislib.einstellungen.SpeicherKey;
import de.bsvrz.buv.rw.bitctrl.eclipse.modell.ModellEinstellungen;

/**
 * Content Provider der Nutzereinstellungen als Baum darstellt. Der Baum hat nur
 * zwei Ebenen, die erste Ebene ist der Speicherort (siehe {@link SpeicherKey}
 * und die zweite die Einstellungen an diesen Ort.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class EinstellungenContentProvider implements ITreeContentProvider {

	private final Map<SpeicherKey, List<?>> speicherKey2einstellungen = new HashMap<>();
	private final ViewerRefresher viewerRefresher = new ViewerRefresher();
	private Viewer viewer;

	@Override
	public Object[] getElements(final Object inputElement) {
		final ModellEinstellungen<?, ?> einstellungen = (ModellEinstellungen<?, ?>) inputElement;
		speicherKey2einstellungen.clear();
		for (final SpeicherKey key : SpeicherKey.getDefaultKeys()) {
			speicherKey2einstellungen.put(key,
					einstellungen.getAlleModellEinstellungen(key));
		}

		return SpeicherKey.getDefaultKeys().toArray();
	}

	@Override
	public boolean hasChildren(final Object element) {
		if (speicherKey2einstellungen == null) {
			return false;
		}

		return getChildren(element).length > 0;
	}

	@Override
	public Object[] getChildren(final Object parentElement) {
		if (parentElement instanceof SpeicherKey) {
			return speicherKey2einstellungen.get(parentElement).toArray();
		}

		return new Object[0];
	}

	@Override
	public Object getParent(final Object element) {
		Object parent = null;
		for (final Entry<SpeicherKey, List<?>> entry : speicherKey2einstellungen
				.entrySet()) {
			if (entry.getValue().contains(element)) {
				if (parent != null) {
					// Parent ist nicht eindeutig, also nicht bestimmbar
					parent = null;
					break;
				}

				parent = entry.getKey();
			}
		}

		return parent;
	}

	/**
	 * Der Input muss von Typ {@link ModellEinstellungen} sein.
	 */
	@Override
	public void inputChanged(final Viewer v, final Object oldInput,
			final Object newInput) {
		viewer = v;
		if (oldInput != null) {
			final ModellEinstellungen<?, ?> einstellungen = (ModellEinstellungen<?, ?>) oldInput;
			einstellungen.removePropertyChangeListener(viewerRefresher);
		}
		if (newInput != null) {
			final ModellEinstellungen<?, ?> einstellungen = (ModellEinstellungen<?, ?>) newInput;
			einstellungen.addPropertyChangeListener(viewerRefresher);
		}
	}

	@Override
	public void dispose() {
		// tut nix
	}

	private final class ViewerRefresher implements PropertyChangeListener {

		@Override
		public void propertyChange(final PropertyChangeEvent e) {
			if (!viewer.getControl().isDisposed()) {
				viewer.getControl().getDisplay().asyncExec(() -> {
					if (!viewer.getControl().isDisposed()) {
						viewer.refresh();
					}
				});
			}

		}

	}

}
