/*
 * Rahmenwerk-Plug-in "BitCtrl-Bibliotheken"
 *
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.rw.bitctrl.eclipse.views;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.dialogs.IMessageProvider;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.ViewerComparator;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.dialogs.FilteredTree;
import org.eclipse.ui.forms.widgets.Form;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.part.ViewPart;

import com.bitctrl.lib.eclipse.BitCtrlEclipseCommonPlugin;
import com.bitctrl.lib.eclipse.Images;
import com.bitctrl.lib.eclipse.help.DefaultHelpContext;
import com.bitctrl.lib.eclipse.viewer.ViewerTools;

import de.bsvrz.buv.rw.bitctrl.eclipse.Messages;
import de.bsvrz.buv.rw.bitctrl.eclipse.MultipleSelection;
import de.bsvrz.buv.rw.bitctrl.eclipse.PatternOrSubstringFilter;
import de.bsvrz.buv.rw.bitctrl.eclipse.SystemObjectTreeContentProvider;
import de.bsvrz.buv.rw.bitctrl.eclipse.SystemObjectTypeLabelProvider;
import de.bsvrz.buv.rw.bitctrl.internal.RahmenwerkService;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;

/**
 * <p>
 * View to let the user select one or multiple objects of type SystemObject.
 * Overwrite {@link #handleDoubleClick(DoubleClickEvent, List)} for adding
 * custom behavior or define a context menu entry using your class name as menu
 * id.
 * </p>
 * <p>
 * Icons are provided using the {@link SystemObjectTypeLabelProvider}. Please
 * provide an icon named after the type of the SystemObject as PNG or GIF in the
 * folder "icons" of the plug-in de.bsvrz.buv.rw.bitctrl.
 * </p>
 * <p>
 * Popup menues created via &lt;objectContribution/&gt; must define a filter
 * (isOfType=[PidOfType])
 * </p>
 *
 * @author schnepel
 */
public abstract class SystemObjectAuswahlView extends ViewPart
		implements ISelectionListener {

	private final class TreeSelectionChangedListener
			implements ISelectionChangedListener {
		@Override
		public void selectionChanged(final SelectionChangedEvent event) {
			currentSelection.clear();
			currentSelection.addAll(event.getStructuredSelection().toList());
			final Iterator<SystemObject> it = currentSelection.iterator();
			while (it.hasNext()) {
				final SystemObject object = it.next();
				if (object instanceof SystemObjectType) {
					it.remove();
				}
			}
		}
	}

	/**
	 * Refreshes the View whenever there is a change in the ObjectFactory.
	 *
	 * @author BitCtrl Systems GmbH, schnepel
	 *
	 */
	private final class ViewRefresher
			implements PropertyChangeListener, Runnable {

		@Override
		public void propertyChange(final PropertyChangeEvent evt) {
			if (null != filteredTree) {
				form.getDisplay().asyncExec(this);
			}
		}

		@Override
		public void run() {
			if (form != null && !form.isDisposed()) {
				form.setMessage(null);
				try {
					final List<SystemObject> systemObjects = new ArrayList<>();
					final ObjektFactory objektFactory = RahmenwerkService
							.getService().getObjektFactory();

					if (objektFactory.isVerbunden()) {
						final ClientDavInterface daVVerbindung = objektFactory
								.getDav();
						final DataModel dataModel = daVVerbindung
								.getDataModel();
						for (final String rootType : rootTypes) {
							systemObjects.add(dataModel.getObject(rootType));
						}
					} else {
						form.setMessage(
								Messages.SystemObjectAuswahl_ERROR_DAV_NOT_INITIALIZED,
								IMessageProvider.ERROR);
					}
					final TreeViewer viewer = filteredTree.getViewer();
					viewer.setInput(systemObjects.toArray());
					viewer.refresh();
				} catch (final RuntimeException re) {
					final String message = form.getMessage();
					form.setMessage(null == message ? ""
							: (message + "\n") + re.getLocalizedMessage());
				}
			}
		}

	}

	private final MultipleSelection multi;
	private final String[] rootTypes;
	/**
	 * the {@link FilteredTree} behind the Part.
	 */
	protected FilteredTree filteredTree;
	private final List<SystemObject> currentSelection = new ArrayList<>();
	private Form form;
	private ViewRefresher viewRefresher;
	private Action linkAction;

	/**
	 * @param multipleSelection
	 * @param rootTypes
	 */
	protected SystemObjectAuswahlView(final MultipleSelection multipleSelection,
			final String... rootTypes) {
		super();
		multi = multipleSelection;
		this.rootTypes = rootTypes;
		setPartName(getClass().getName());
	}

	@Override
	public void createPartControl(final Composite parent) {
		final FormToolkit toolkit = new FormToolkit(parent.getDisplay());
		form = toolkit.createForm(parent);
		toolkit.decorateFormHeading(form);
		final Label headLabel = toolkit.createLabel(form.getHead(),
				Messages.SystemObjectAuswahl_Description, SWT.WRAP);
		headLabel.setBackground(null);
		form.setHeadClient(headLabel);
		final Composite formBody = form.getBody();
		PlatformUI.getWorkbench().getHelpSystem().setHelp(formBody,
				getHelpId());
		formBody.setLayout(new GridLayout());

		filteredTree = new FilteredTree(formBody,
				SWT.BORDER | multi.getSwtConstant(),
				new PatternOrSubstringFilter(), true, true);
		toolkit.adapt(filteredTree);
		filteredTree.getViewer()
				.setContentProvider(new SystemObjectTreeContentProvider());
		filteredTree.setInitialText(Messages.SystemObjectAuswahl_DefaultFilter);
		filteredTree.getViewer().setComparator(new ViewerComparator());
		final ISelectionChangedListener selectionListener = new TreeSelectionChangedListener();
		filteredTree.getViewer().addSelectionChangedListener(selectionListener);
		filteredTree.addDisposeListener(e -> filteredTree.getViewer()
				.removeSelectionChangedListener(selectionListener));
		filteredTree.getViewer().addDoubleClickListener(
				event -> handleDoubleClick(event, currentSelection));
		filteredTree.getViewer()
				.setLabelProvider(new SystemObjectTypeLabelProvider());

		getSite().setSelectionProvider(filteredTree.getViewer());

		viewRefresher = new ViewRefresher();
		RahmenwerkService.getService().getObjektFactory()
				.addPropertyChangeListener(viewRefresher);
		viewRefresher.run();

		linkAction = new Action("Selektion verknüpfen", IAction.AS_CHECK_BOX) {
			@Override
			public void run() {
				selectionChanged(null, lastSelection);
			}
		};
		linkAction.setImageDescriptor(BitCtrlEclipseCommonPlugin.getDefault()
				.getImageDescriptor(Images.IMG_ELCL_SYNCED));
		getViewSite().getActionBars().getToolBarManager().add(linkAction);
		getSite().getWorkbenchWindow().getSelectionService()
				.addSelectionListener(this);
	}

	protected String getHelpId() {
		return DefaultHelpContext.getDefaultContextId(getClass());
	}

	/**
	 * Wird aufgerufen, sobald ein Doppelklick auf ein Element ausgeführt wird.
	 *
	 * @param event
	 * @param currentSelection
	 */
	protected void handleDoubleClick(final DoubleClickEvent event,
			@SuppressWarnings("hiding") final List<SystemObject> currentSelection) {
		//
	}

	@Override
	public void setFocus() {
		filteredTree.setFocus();
	}

	protected MenuManager setupPopupMenu(final Composite parent) {
		final MenuManager menuManager = new MenuManager("#PopupMenu");
		filteredTree.getViewer().getControl()
				.setMenu(menuManager.createContextMenu(parent));
		getViewSite().registerContextMenu(menuManager,
				filteredTree.getViewer());
		return menuManager;
	}

	@Override
	public void dispose() {
		RahmenwerkService.getService().getObjektFactory()
				.removePropertyChangeListener(viewRefresher);
		super.dispose();
	}

	private IStructuredSelection lastSelection;

	@Override
	public void selectionChanged(final IWorkbenchPart part,
			final ISelection selection) {
		if (equals(part)) {
			return;
		}

		final Set<SystemObject> systemObjects = new HashSet<>(
				ViewerTools.getTypedSelection(selection, SystemObject.class));
		if (systemObjects.isEmpty()) {
			return;
		}

		if (!filteredTree.getViewer().getTree().isDisposed()) {
			final TreeItem[] treeSel = filteredTree.getViewer().getTree()
					.getSelection();
			final Set<SystemObject> treeData = new HashSet<>();
			for (final TreeItem treeItem : treeSel) {
				treeData.add((SystemObject) treeItem.getData());
			}

			lastSelection = (IStructuredSelection) selection;

			if (systemObjects.equals(treeData)) {
				return;
			}

			if (linkAction.isChecked()) {
				filteredTree.getViewer().setSelection(
						new StructuredSelection(new ArrayList<>(systemObjects)),
						true);
			}
		}
	}
}
