/*
 * Rahmenwerk-Plug-in "BitCtrl-Bibliotheken"
 *
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.rw.bitctrl.eclipse.widgets;

import org.eclipse.core.databinding.DataBindingContext;
import org.eclipse.core.databinding.observable.value.IObservableValue;
import org.eclipse.core.databinding.observable.value.IValueChangeListener;
import org.eclipse.core.databinding.observable.value.ValueChangeEvent;
import org.eclipse.core.databinding.observable.value.WritableValue;
import org.eclipse.jface.databinding.swt.typed.WidgetProperties;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Spinner;

/**
 * Adjusts the step and page increment values automatically according to the
 * actual value. Numbers are rounded to a multiple of 2 5 or 10 with a precision
 * of 2 digits.
 *
 * TODO: Algorithm needs to be reworked.
 *
 * @author schnepel
 */
public class AutoIncrementSpinner extends Composite implements DisposeListener {

	private final class Model2Target implements IValueChangeListener {
		@Override
		public void handleValueChange(final ValueChangeEvent event) {
			Integer integer = (Integer) event.diff.getNewValue();
			if (null != integer) {
				if (spinner.getSelection() == integer) {
					integer = null;
				}
			} else {
				integer = 0;
			}
			if (null != integer) {
				spinner.setValues(integer, Integer.MIN_VALUE, Integer.MAX_VALUE,
						0, SPINNER_PAGE_INCREMENT, SPINNER_SINGLE_INCREMENT);
			}
		}
	}

	private final class Target2Model implements IValueChangeListener {
		@Override
		public void handleValueChange(final ValueChangeEvent event) {
			Integer newValue = (Integer) event.diff.getNewValue();
			final Integer oldValue = (Integer) observableModel.getValue();
			if (oldValue != null && newValue != null) {
				newValue = (int) dynamicRound(oldValue, newValue);
				final Integer max = (Integer) maximum.getValue();
				final Integer min = (Integer) minimum.getValue();
				if (null != max) {
					newValue = Math.min(newValue, max);
				}
				if (null != min) {
					newValue = Math.max(newValue, min);
				}
				if (oldValue.equals(newValue)) {
					observableTarget.setValue(newValue);
				}
			}
			observableModel.setValue(newValue);
		}

		private long dynamicRound(final long oldValue, final long newValue) {
			int oldSign = (int) Math.signum(oldValue);
			if (0 == oldSign) {
				oldSign = 1;
			}
			final int grow = oldSign * (int) Math.signum(newValue - oldValue);
			if (0 == grow) {
				return newValue;
			}
			long distance = (long) Math.abs(((double) newValue) - oldValue);
			if (SPINNER_PAGE_INCREMENT != distance
					&& SPINNER_SINGLE_INCREMENT != distance) {
				return newValue;
			}
			if (SPINNER_PAGE_INCREMENT == distance) {
				distance = 1;
			} else {
				distance = 10;
			}
			if (0 == oldValue) {
				return grow * distance;
			}
			final long oldVal = Math.abs(oldValue);
			final long ticks = 100;
			final double[] values = new double[] { 1, 2, 5, 10 };
			final double[] multiplier = new double[values.length - 1];
			final int mCount = multiplier.length;
			for (int i = 0; i < mCount; i++) {
				multiplier[i] = values[i + 1] / values[i];
			}
			int mIndex = 0;
			double stepping;
			stepping = Math.max(1,
					Math.pow(10, Math.floor(Math.log10(oldVal)) - 2));
			long newVal = calcNewVal(oldVal, grow, distance, stepping);
			// grow stepping to be larger than value / ticks
			for (; growFactor(grow, oldVal, newVal) >= ticks
					/ distance; mIndex = (mIndex + 1) % mCount) {
				stepping *= multiplier[mIndex];
				newVal = calcNewVal(oldVal, grow, distance, stepping);
				if (stepping > 10000000) {
					break;
				}
			}
			return newVal * oldSign;
		}

		private double growFactor(final int grow, final long oldVal,
				final long newVal) {
			return 1 == grow ? oldVal / (newVal - (double) oldVal)
					: newVal / (oldVal - (double) newVal);
		}

		private long calcNewVal(final long oldVal, final int grow,
				final long distance, final double stepping) {
			final double dx = (oldVal + grow * distance * stepping)
					/ (distance * stepping);
			double x;
			if (1 == grow) {
				x = Math.floor(dx);
			} else {
				x = Math.ceil(dx);
			}
			return Math.min(Integer.MAX_VALUE,
					(long) (x * distance * stepping));
		}
	}

	static final int SPINNER_SINGLE_INCREMENT = 978;
	static final int SPINNER_PAGE_INCREMENT = 987;

	private final WritableValue minimum = new WritableValue(
			Integer.MIN_VALUE + 1, Integer.TYPE);
	private final WritableValue maximum = new WritableValue(Integer.MAX_VALUE,
			Integer.TYPE);

	private final IObservableValue observableModel = new WritableValue(0,
			Integer.TYPE);
	private final Spinner spinner;
	private final IValueChangeListener target2Model;
	private final IValueChangeListener model2Target;
	private final IObservableValue observableTarget;

	public AutoIncrementSpinner(final Composite parent, final int style) {
		super(parent, SWT.NONE);
		setLayout(new FillLayout());
		spinner = new Spinner(this, style);
		target2Model = new Target2Model();
		model2Target = new Model2Target();
		observableTarget = WidgetProperties.spinnerSelection().observe(spinner);
		observableTarget.addValueChangeListener(target2Model);
		observableModel.addValueChangeListener(model2Target);
		final DataBindingContext dbc = new DataBindingContext();
		dbc.bindValue(WidgetProperties.minimum().observe(spinner), minimum);
		dbc.bindValue(WidgetProperties.maximum().observe(spinner), maximum);
		spinner.setValues(0, Integer.MIN_VALUE, Integer.MAX_VALUE, 0,
				SPINNER_PAGE_INCREMENT, SPINNER_SINGLE_INCREMENT);
		addDisposeListener(this);
	}

	@Override
	public void widgetDisposed(final DisposeEvent e) {
		observableTarget.removeValueChangeListener(target2Model);
		observableModel.removeValueChangeListener(model2Target);
	}

	public IObservableValue getObservableValue() {
		return observableModel;
	}

	public int getSelection() {
		return (Integer) observableModel.getValue();
	}

	public void setSelection(final int value) {
		observableModel.setValue(value);
	}

	public int getMinimum() {
		return (Integer) minimum.getValue();
	}

	public void setMinimum(final int value) {
		if (getMaximum() < value) {
			setMaximum(value);
		}
		if (getSelection() < value) {
			setSelection(value);
		}
		minimum.setValue(value);
	}

	public IObservableValue observeMinimum() {
		return minimum;
	}

	public int getMaximum() {
		return (Integer) maximum.getValue();
	}

	public void setMaximum(final int value) {
		if (getMinimum() > value) {
			setMinimum(value);
		}
		if (getSelection() > value) {
			setSelection(value);
		}
		maximum.setValue(value);
	}

	public IObservableValue observeMaximum() {
		return maximum;
	}
}
