/*
 * Rahmenwerk-Plug-in "BitCtrl-Bibliotheken"
 *
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.rw.bitctrl.eclipse.widgets;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.databinding.DataBindingContext;
import org.eclipse.core.databinding.observable.Observables;
import org.eclipse.core.databinding.observable.value.IObservableValue;
import org.eclipse.core.databinding.observable.value.WritableValue;
import org.eclipse.jface.databinding.swt.typed.WidgetProperties;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.resource.LocalResourceManager;
import org.eclipse.jface.resource.ResourceManager;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.ColorDialog;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;

import com.bitctrl.lib.eclipse.BitCtrlEclipseCommonPlugin;
import com.bitctrl.lib.eclipse.Images;
import com.bitctrl.lib.eclipse.databinding.observables.SilencePolicy;
import com.bitctrl.lib.eclipse.databinding.observables.SilentComputedValue;
import com.bitctrl.lib.eclipse.databinding.realm.RealmRunnable;
import com.bitctrl.lib.eclipse.resources.WidgetImageRegistry;

public class ColorPicker extends Composite implements ISelectionProvider {
	private static final RGB BLACK = new RGB(0, 0, 0);
	private static final RGB WHITE = new RGB(255, 255, 255);

	private final WritableValue<RGB> color = new WritableValue<>(null,
			RGB.class);
	private final Label colorText;
	private final Label colorAction;
	private final IObservableValue<Boolean> enabledState = new WritableValue<>(
			true, Boolean.TYPE);
	private final List<ISelectionChangedListener> selectionListeners = new ArrayList<>();
	private ResourceManager resourceManager;

	/**
	 * Konstruktor zum Erzeugen eines {@link ColorPicker}.
	 *
	 * @param parent
	 *            das übergeordnete Element
	 * @param style
	 *            der Stil für die Darstellung der Farbauswahl
	 */

	public ColorPicker(final Composite parent, final int style) {
		this(parent, style, (RGB) null);
	}

	/**
	 * Konstruktor zum Erzeugen eines {@link ColorPicker}.
	 *
	 * @param parent
	 *            das übergeordnete Element
	 * @param style
	 *            der Stil für die Darstellung der Farbauswahl
	 * @param defaultRGB
	 *            die initiale Farbe
	 */
	public ColorPicker(final Composite parent, final int style,
			final RGB defaultRGB) {
		this(parent, style, Observables.constantObservableValue(defaultRGB));
	}

	/**
	 * Konstruktor zum Erzeugen eines {@link ColorPicker}.
	 *
	 * @param parent
	 *            das übergeordnete Element
	 * @param style
	 *            der Stil für die Darstellung der Farbauswahl
	 * @param defaultRGB
	 *            die initiale Farbe
	 */
	public ColorPicker(final Composite parent, final int style,
			final IObservableValue<RGB> defaultRGB) {
		super(parent, SWT.NONE);

		resourceManager = new LocalResourceManager(
				JFaceResources.getResources(), this);

		final GridLayout layout = new GridLayout(2, false);
		layout.verticalSpacing = 0;
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		setLayout(layout);

		colorText = new Label(this, style | SWT.CENTER);
		final GridData layoutData = new GridData(SWT.FILL, SWT.CENTER, true,
				false);
		layoutData.heightHint = 17;
		colorText.setLayoutData(layoutData);
		colorText.setText("undef.");

		colorAction = new Label(this, style);
		colorAction.setImage(WidgetImageRegistry.getInstance(this)
				.getImage(BitCtrlEclipseCommonPlugin.getDefault()
						.getImageDescriptor(Images.IMG_ETOOL_EDIT)));
		colorAction.setLayoutData(
				new GridData(SWT.FILL, SWT.CENTER, false, false));

		final DataBindingContext dbc = new DataBindingContext();
		@SuppressWarnings("unchecked")
		final IObservableValue<RGB> bgColor = new SilentComputedValue(
				defaultRGB.getRealm(), RGB.class, SilencePolicy.EqualsSign) {
			@SuppressWarnings("null")
			@Override
			protected RGB calculate() {
				RGB rgb = color.getValue();
				if (null == rgb) {
					if (defaultRGB != null) {
						rgb = defaultRGB.getValue();
					}
					if (null == rgb) {
						rgb = BLACK;
					}
				}
				if (Boolean.FALSE.equals(enabledState.getValue())) {
					rgb = new RGB((rgb.red + 192) / 2, (rgb.green + 192) / 2,
							(rgb.blue + 192) / 2);
				}
				return rgb;
			}
		};

		dbc.bindValue(WidgetProperties.background().observe(colorText),
				new SilentComputedValue(defaultRGB.getRealm(), Color.class,
						SilencePolicy.EqualsSign) {
					@Override
					protected Color calculate() {
						return resourceManager.createColor(bgColor.getValue());
					}
				});
		dbc.bindValue(WidgetProperties.foreground().observe(colorText),
				new SilentComputedValue(defaultRGB.getRealm(), Color.class,
						SilencePolicy.EqualsSign) {
					@Override
					protected Color calculate() {
						RGB rgb = color.getValue();
						if (null == rgb) {
							rgb = bgColor.getValue();
							final int grey = (rgb.red * 30 + rgb.green * 59
									+ rgb.blue * 11) / 100;
							rgb = grey < 128 ? WHITE : BLACK;
						}
						return resourceManager.createColor(rgb);
					}
				});
		final MouseAdapter mouseListener = new MouseAdapter() {
			@Override
			public void mouseUp(final MouseEvent e) {
				final Point position = getDisplay().map(ColorPicker.this, null,
						0, 0);
				final Shell shell = new Shell(getShell());
				position.y = Math.max(0, position.y - 319);
				position.x = Math.max(0, position.x - 20);
				shell.setLocation(position);
				final ColorDialog cd = new ColorDialog(shell);
				cd.setText("Linienfarbe wählen...");
				cd.setRGB(color.getValue());
				final RGB newColor = cd.open();
				if (newColor == null) {
					return;
				}
				color.setValue(newColor);
				notifySelectionListeners();
			}
		};
		colorText.addMouseListener(mouseListener);
		colorAction.addMouseListener(mouseListener);
	}

	private void notifySelectionListeners() {
		for (final ISelectionChangedListener listener : selectionListeners) {
			listener.selectionChanged(
					new SelectionChangedEvent(this, getSelection()));
		}
	}

	/**
	 * liefert die Property für die aktuelle ausgewählte Farbe.
	 *
	 * @return die Property
	 */
	public IObservableValue<RGB> getColor() {
		return color;
	}

	/**
	 * liefert den RGB-Wert der ausgewählten Farbe.
	 *
	 * @return den RGB-Wert
	 */
	public RGB getColorValue() {
		return color.getValue();
	}

	public void setColor(final RGB color) {
		RealmRunnable.syncExec(this.color.getRealm(),
				() -> ColorPicker.this.color.setValue(color));
	}

	@Override
	public void setEnabled(final boolean enabled) {
		super.setEnabled(enabled);
		colorText.setEnabled(enabled);
		colorAction.setEnabled(enabled);
		enabledState.setValue(enabled);
	}

	@Override
	public void addSelectionChangedListener(
			final ISelectionChangedListener listener) {
		selectionListeners.add(listener);
	}

	@Override
	public ISelection getSelection() {
		return new StructuredSelection(getColorValue());
	}

	@Override
	public void removeSelectionChangedListener(
			final ISelectionChangedListener listener) {
		selectionListeners.remove(listener);
	}

	@Override
	public void setSelection(final ISelection selection) {
		setColor((RGB) ((StructuredSelection) selection).getFirstElement());
	}
}
