/*
 * Rahmenwerk-Plug-in "BitCtrl-Bibliotheken"
 *
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.rw.bitctrl.eclipse.wizards;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;

import de.bsvrz.buv.rw.basislib.einstellungen.EinstellungLocation;
import de.bsvrz.buv.rw.basislib.einstellungen.SpeicherKey;
import de.bsvrz.buv.rw.bitctrl.eclipse.util.RahmenwerkUtils;

/**
 * Erlaubt dem Nutzer die Auswahl wo Einstellungen gesichert werden sollen. Es
 * kann ein oder mehrere Einstellungsarten ausgewählt werden.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 */
public class EinstellungsArtWizardPage extends WizardPage {

	private Collection<SpeicherKey> preSelection;
	private Map<SpeicherKey, Button> einstellungsArten;
	private boolean multiple;

	/**
	 * Initialisert die Page.
	 *
	 * @param pageName
	 *            der Name der Page im Wizard.
	 */
	public EinstellungsArtWizardPage(final String pageName) {
		super(pageName);
		setTitle("Einstellungsart");
		setDescription("Bitte wählen Sie die gewünschten Einstellungsarten.");
		setPageComplete(false);
	}

	/**
	 * Legt das Flag fest, ob mehrere Einstellungsarten ausgewählt werden
	 * dürfen.
	 *
	 * @param multiple
	 */
	public void setMultiple(final boolean multiple) {
		this.multiple = multiple;
	}

	/**
	 * Gibt das Flag zurück, ob mehrere Einstellungsarten ausgewählt werden
	 * dürfen.
	 *
	 * @return <code>true</code>, wenn die Mehrfachauswahl erlaubt sein soll.
	 */
	public boolean isMultiple() {
		return multiple;
	}

	/**
	 * Gibt die vorausgewählten Einstellungsarten zurück.
	 *
	 * @return die Vorauswahl der Einstellungsarten.
	 */
	public Collection<SpeicherKey> getPreSelection() {
		if (preSelection == null) {
			return Collections.emptyList();
		}
		return preSelection;
	}

	/**
	 * Legt die vorausgewählten Einstellungsarten fest.
	 *
	 * @param preSelection
	 */
	public void setPreSelection(final Collection<SpeicherKey> preSelection) {
		this.preSelection = preSelection;
	}

	@Override
	public void createControl(final Composite parent) {
		einstellungsArten = new HashMap<>();

		final Composite container = new Composite(parent, SWT.NONE);
		container.setLayout(new GridLayout(1, false));

		final SelectionListener selectionListener = new SelectionAdapter() {

			@Override
			public void widgetSelected(final SelectionEvent e) {
				updatePageComplete();
			}

		};

		for (final SpeicherKey einstellungsArt : SpeicherKey.getDefaultKeys()) {
			if (!RahmenwerkUtils.isAdmin() && einstellungsArt
					.getLocation() == EinstellungLocation.NETZWERKWEIT) {
				continue;
			}

			final int flags;
			if (isMultiple()) {
				flags = SWT.CHECK;
			} else {
				flags = SWT.RADIO;
			}

			final Button button = new Button(container, flags);
			button.setText(einstellungsArt.getTypBezeichnung());

			if (getPreSelection().contains(einstellungsArt)) {
				button.setSelection(true);
			}
			button.addSelectionListener(selectionListener);

			einstellungsArten.put(einstellungsArt, button);
		}

		updatePageComplete();
		setControl(container);
	}

	/**
	 * Gibt die vom Nutzer ausgewählten Einstellungsarten zurück.
	 *
	 * @return die ausgewählten Einstellungsarten.
	 */
	public Set<SpeicherKey> getEinstellungsArten() {
		final Set<SpeicherKey> result = new HashSet<>();

		for (final Entry<SpeicherKey, Button> entry : einstellungsArten
				.entrySet()) {
			if (entry.getValue().getSelection()) {
				result.add(entry.getKey());
			}
		}

		return result;
	}

	/**
	 * Gibt die erste (oder einzige) vom Nutzer gewählte Einstellungsart zurück.
	 *
	 * @return eine oder die gewählte Einstellungsart.
	 */
	public SpeicherKey getFirstEinstellungsArt() {
		final List<SpeicherKey> result = new ArrayList<>();

		for (final Entry<SpeicherKey, Button> entry : einstellungsArten
				.entrySet()) {
			if (entry.getValue().getSelection()) {
				result.add(entry.getKey());
			}
		}

		return result.get(0);
	}

	private void updatePageComplete() {
		boolean complete = false;

		for (final Button b : einstellungsArten.values()) {
			complete |= b.getSelection();
		}

		setPageComplete(complete);
	}
}
