/*
 * Rahmenwerk-Plug-in "BitCtrl-Bibliotheken"
 *
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.rw.bitctrl.eclipse.wizards;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Map.Entry;

import org.eclipse.core.databinding.observable.value.IObservableValue;
import org.eclipse.core.runtime.Assert;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;

import de.bsvrz.buv.rw.basislib.Rahmenwerk;
import de.bsvrz.buv.rw.basislib.einstellungen.EinstellungLocation;
import de.bsvrz.buv.rw.basislib.einstellungen.EinstellungOwnerType;
import de.bsvrz.buv.rw.basislib.einstellungen.SpeicherKey;
import de.bsvrz.buv.rw.bitctrl.eclipse.Messages;
import de.bsvrz.buv.rw.bitctrl.internal.RahmenwerkService;

/**
 * {@link WizardPage}, auf der der Nutzer den Ort ( {@link SpeicherKey} )
 * irgendetwas gespeichert werden soll.
 */
public class SelectEinstellungsArtWizardPage extends WizardPage {

	/**
	 * Flag, durch das entschieden wird, ob auch die Netzweit-Optionen mit
	 * angezeigt werden.
	 */
	private boolean erlaubeNetzweit = true;

	/** die ausgewählte Einstellungsart. */
	private SpeicherKey einstellungsArt;

	private Group artGroup;

	private final IObservableValue observer;

	public SelectEinstellungsArtWizardPage(final SpeicherKey defaultArt) {
		super(SelectEinstellungsArtWizardPage.class.getName());
		einstellungsArt = defaultArt;
		observer = null;
	}

	public SelectEinstellungsArtWizardPage(
			final IObservableValue observableValue) {
		super(SelectEinstellungsArtWizardPage.class.getName());
		Assert.isTrue(SpeicherKey.class.equals(observableValue.getValueType()));
		observer = observableValue;
		einstellungsArt = (SpeicherKey) observableValue.getValue();
	}

	@Override
	public void createControl(final Composite parent) {
		setTitle(Messages.SelectEinstellungsArtWizardPage_Title);
		setDescription(Messages.SelectEinstellungsArtWizardPage_Description);

		final Composite control = new Composite(parent, SWT.NONE);
		final GridLayout layout = new GridLayout();
		layout.numColumns = 1;
		control.setLayout(layout);

		new Label(control, SWT.NONE)
				.setText(Messages.SelectEinstellungsArtWizardPage_Prompt);

		artGroup = new Group(control, SWT.SHADOW_ETCHED_OUT);
		artGroup.setText("Einstellungsart");
		artGroup.setLayout(new GridLayout(1, false));

		final Map<SpeicherKey, String> arten = new LinkedHashMap<>();
		final Rahmenwerk rahmenwerk = RahmenwerkService.getService()
				.getRahmenWerk();
		if (rahmenwerk.isOnline() && erlaubeNetzweit) {
			arten.put(
					new SpeicherKey(EinstellungOwnerType.BENUTZER,
							EinstellungLocation.LOKAL,
							rahmenwerk.getBenutzer().getPid()),
					Messages.EinstellungsArt_benutzerLokal);
			arten.put(
					new SpeicherKey(EinstellungOwnerType.BENUTZER,
							EinstellungLocation.NETZWERKWEIT,
							rahmenwerk.getBenutzer().getPid()),
					Messages.EinstellungsArt_benutzerNetzweit);
			arten.put(
					new SpeicherKey(EinstellungOwnerType.SYSTEM,
							EinstellungLocation.LOKAL),
					Messages.EinstellungsArt_allgemeinLokal);
			arten.put(
					new SpeicherKey(EinstellungOwnerType.SYSTEM,
							EinstellungLocation.NETZWERKWEIT),
					Messages.EinstellungsArt_allgemeinNetzweit);
		} else {
			arten.put(
					new SpeicherKey(EinstellungOwnerType.SYSTEM,
							EinstellungLocation.LOKAL),
					Messages.EinstellungsArt_allgemeinLokal);
		}

		for (final Entry<SpeicherKey, String> entry : arten.entrySet()) {
			final Button button = new Button(artGroup, SWT.RADIO);
			button.setText(entry.getValue());
			button.setData(entry.getKey());
			button.setSelection(entry.getKey().equals(einstellungsArt));
			button.addSelectionListener(new SelectionAdapter() {
				@Override
				public void widgetSelected(final SelectionEvent e) {
					validatePage();
				}
			});
		}
		validatePage();
		setControl(control);
	}

	protected void validatePage() {
		if (artGroup != null) {
			for (final Control child : artGroup.getChildren()) {
				if ((child instanceof Button)
						&& ((Button) child).getSelection()) {
					final Object data = child.getData();
					if (data instanceof SpeicherKey) {
						einstellungsArt = (SpeicherKey) data;
						if (observer != null) {
							observer.setValue(einstellungsArt);
						}
						break;
					}
				}
			}
		}
		setPageComplete(einstellungsArt != null);
	}

	public SpeicherKey getArt() {
		return einstellungsArt;
	}

	/**
	 * @return the erlaubeNetzweit
	 */
	public final boolean isErlaubeNetzweit() {
		return erlaubeNetzweit;
	}

	/**
	 * Setzt ein Flag, durch das entschieden wird, ob auch die Netzweit-Optionen
	 * mit angezeigt werden oder nicht.
	 *
	 * @param erlaubeNetzweit
	 *            the erlaubeNetzweit to set
	 */
	public final void setErlaubeNetzweit(final boolean erlaubeNetzweit) {
		this.erlaubeNetzweit = erlaubeNetzweit;
	}
}
