/*
 * Rahmenwerk-Plug-in "BitCtrl-Bibliotheken"
 *
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.rw.bitctrl.eclipse.wizards;

import java.util.List;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

import de.bsvrz.buv.rw.bitctrl.eclipse.MultipleSelection;
import de.bsvrz.buv.rw.bitctrl.eclipse.SystemObjectAuswahlType;
import de.bsvrz.dav.daf.main.config.SystemObject;

/**
 * Definiert eine Wizard-Seite, auf der nicht nur der Tree zur Auswahl eines
 * {@link SystemObject}s ist, sonder darunter noch ein Textfeld+'Suchen'-Button,
 * in dem der Nutzer was eingeben kann
 *
 * @author uhlmann
 */
public abstract class SystemObjectAuswahlMitSuchenFeldWizardPage
		extends SystemObjectAuswahlWizardPage {

	/**
	 * Widget, in dem der Nutzer einen Text eingeben kann.
	 */
	private Text editText;

	/**
	 * Hier wird immer der aktuelle Nutzertext reinkopiert, so dass die Methode
	 * {@link #getNutzerText()} von jedem Thread aus aufgerufen werden kann.
	 */
	private String nutzerText;

	private final String editLabelString;

	private final Button[] suchenButtons;

	private final String[] buttonLabels;

	/**
	 * Diese Methode wird aufgerufen, wenn der Nutzer auf den 'Suchen'-Button
	 * klickt.
	 * <p>
	 * Ueber this.get*()-Methoden kann auf verschiedene Dinge, wie z.B. die
	 * aktuelle Auswahl von SystemObjects, zugegriffen werden.
	 * </p>
	 *
	 * @param index
	 *            Index der betätigten Schaltfläche
	 *
	 * @return der Text, der dann ins Edit-Feld kopiert wird
	 */
	protected abstract String suchen(int index);

	/**
	 * Konstruktor initialisiert Felder
	 *
	 * @param title
	 *            Dialogtitel
	 * @param multi
	 *            Mehrfach- oder Einfachauswahl
	 * @param selectionType
	 *            Auswahltyp
	 * @param editLabelString
	 * @param preSelectedNutzerText
	 * @param preSelectedObjects
	 * @param buttonLabels
	 * @param rootTypes
	 */
	public SystemObjectAuswahlMitSuchenFeldWizardPage(final String title,
			final MultipleSelection multi,
			final SystemObjectAuswahlType selectionType,
			final String editLabelString, final String preSelectedNutzerText,
			final List<SystemObject> preSelectedObjects,
			final String[] buttonLabels, final String... rootTypes) {
		super(title, multi, selectionType, preSelectedObjects, rootTypes);
		this.editLabelString = editLabelString;
		nutzerText = preSelectedNutzerText;
		this.buttonLabels = buttonLabels;
		suchenButtons = new Button[buttonLabels.length];
	}

	@Override
	public void createControl(final Composite parent) {
		GridData gd;
		final int nColumns = 2 + suchenButtons.length;
		final Composite editComposite = new Composite(parent, SWT.NONE);
		editComposite.setLayout(new GridLayout(nColumns, false));
		super.createControl(editComposite);
		gd = new GridData(SWT.FILL, SWT.FILL, true, true);
		gd.horizontalSpan = nColumns;
		getControl().setLayoutData(gd);
		final Label l = new Label(editComposite, SWT.NONE);
		l.setText(editLabelString);
		gd = new GridData(SWT.FILL, SWT.FILL, false, false);
		l.setLayoutData(gd);
		editText = new Text(editComposite, SWT.BORDER);
		if (nutzerText != null) {
			// Bewusst VOR addModifyListener!
			editText.setText(nutzerText);
		}
		editText.addModifyListener(
				e -> setNutzerText(((Text) e.widget).getText()));
		gd = new GridData(SWT.FILL, SWT.FILL, true, false);
		editText.setLayoutData(gd);

		int buttonLoop;
		for (buttonLoop = 0; buttonLoop < suchenButtons.length; ++buttonLoop) {
			final Button suchenButton = new Button(editComposite, SWT.PUSH);
			suchenButton.setData("buttonIndex", buttonLoop);
			suchenButton.setText(buttonLabels[buttonLoop]);
			gd = new GridData(SWT.FILL, SWT.FILL, true, false);
			gd.widthHint = 80 - 5 * Math.min(6, suchenButtons.length);
			suchenButton.setLayoutData(gd);
			suchenButton.addSelectionListener(new SelectionAdapter() {

				@Override
				public void widgetSelected(final SelectionEvent e) {
					// This will also trigger the ModifyListener
					editText.setText(
							suchen((Integer) e.widget.getData("buttonIndex")));
				}

			});
		}
	}

	private synchronized void setNutzerText(final String nutzerText) {
		this.nutzerText = nutzerText;
	}

	/**
	 * Liefert den vom Nutzer eingetippten oder über Suchdialoge spezifizierten
	 * Text
	 *
	 * @return der Text
	 */
	public synchronized String getNutzerText() {
		return nutzerText;
	}
}
