/*
 * Rahmenwerk-Plug-in "BitCtrl-Bibliotheken"
 *
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.rw.bitctrl.eclipse.wizards;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.runtime.Assert;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.ViewerComparator;
import org.eclipse.jface.wizard.IWizard;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.dialogs.FilteredTree;

import com.bitctrl.lib.eclipse.wizards.ComposedWizardDialog;
import com.bitctrl.lib.eclipse.wizards.ComposedWizardDialog.CloseWizardHandler;

import de.bsvrz.buv.rw.bitctrl.eclipse.Messages;
import de.bsvrz.buv.rw.bitctrl.eclipse.MultipleSelection;
import de.bsvrz.buv.rw.bitctrl.eclipse.PatternOrSubstringFilter;
import de.bsvrz.buv.rw.bitctrl.eclipse.SystemObjectAuswahlType;
import de.bsvrz.buv.rw.bitctrl.eclipse.SystemObjectTreeContentProvider;
import de.bsvrz.buv.rw.bitctrl.eclipse.SystemObjectTypeLabelProvider;
import de.bsvrz.buv.rw.bitctrl.internal.RahmenwerkService;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;

/**
 * Wizard page to let the user select one or multiple objects of type
 * SystemObject.
 *
 * Icons are provided using the {@link SystemObjectTypeLabelProvider}. Please
 * provide an icon named after the type of the SystemObject as PNG or GIF in the
 * folder icons of the plug-in de.bsvrz.buv.rw.bitctrl.
 *
 * @author schnepel
 */
public class SystemObjectAuswahlWizardPage extends WizardPage
		implements CloseWizardHandler {

	private final class FilteredTreeDoubleClickListener
			implements IDoubleClickListener {
		@Override
		public void doubleClick(final DoubleClickEvent event) {
			final IWizard w = getWizard();
			if (w != null && w.canFinish() && w
					.getContainer() instanceof final ComposedWizardDialog window) {
				window.finishFromCode();
			}

		}
	}

	private final class FilteredTreeSelectionChangedListener
			implements ISelectionChangedListener {
		@Override
		public void selectionChanged(final SelectionChangedEvent event) {
			result.clear();
			result.addAll(event.getStructuredSelection().toList());
			final Iterator<SystemObject> it = result.iterator();
			while (it.hasNext()) {
				final SystemObject object = it.next();
				switch (selectionType) {
				case OnlyInstances:
					if (object instanceof SystemObjectType) {
						it.remove();
					}
					break;
				case OnlyTypes:
					if (!(object instanceof SystemObjectType)) {
						it.remove();
					}
					break;
				case TypesOrInstances:
					break;
				}
			}
			validatePage();
		}
	}

	private FilteredTree filteredTree;
	private final String[] rootTypes;
	private final List<SystemObject> result = new ArrayList<>();
	private final MultipleSelection multi;
	private final SystemObjectAuswahlType selectionType;

	/**
	 * Konstruktor.
	 *
	 * @param title
	 *            Kopfzeile des Dialoges
	 * @param multi
	 *            Sollen mehrere Elemente ausgewählt werden können?
	 * @param selectionType
	 *            Sollen Typen und/oder Instanzen auagewählt werden können?
	 * @param preSelectedObjects
	 *            Liste der vorselektierten Objekte.
	 * @param rootTypes
	 *            Die erlaubten Typen für die Baumdarstellung
	 */
	public SystemObjectAuswahlWizardPage(final String title,
			final MultipleSelection multi,
			final SystemObjectAuswahlType selectionType,
			final List<SystemObject> preSelectedObjects,
			final String... rootTypes) {
		super(SystemObjectAuswahlWizardPage.class.getName());
		Assert.isNotNull(selectionType);
		this.multi = multi;
		this.rootTypes = rootTypes;
		this.selectionType = selectionType;
		if (null != preSelectedObjects) {
			result.addAll(preSelectedObjects);
		}
		setTitle(title);
		setDescription(Messages.SystemObjectAuswahl_Description);
	}

	@Override
	public void createControl(final Composite parent) {
		final List<SystemObject> systemObjects = new ArrayList<>();
		final ClientDavInterface daVVerbindung = RahmenwerkService.getService()
				.getObjektFactory().getDav();
		if (null != daVVerbindung) {
			final DataModel dataModel = daVVerbindung.getDataModel();
			for (final String rootType : rootTypes) {
				systemObjects.add(dataModel.getObject(rootType));
			}
		} else {
			setErrorMessage(
					Messages.SystemObjectAuswahl_ERROR_DAV_NOT_INITIALIZED);
		}
		filteredTree = new FilteredTree(parent,
				SWT.BORDER | multi.getSwtConstant(),
				new PatternOrSubstringFilter(), true, true);
		filteredTree.getViewer()
				.setContentProvider(new SystemObjectTreeContentProvider());
		filteredTree.getViewer().setUseHashlookup(true);
		filteredTree.getViewer().setInput(systemObjects.toArray());
		filteredTree.setInitialText(Messages.SystemObjectAuswahl_DefaultFilter);
		filteredTree.getViewer().setComparator(new ViewerComparator());
		if (!result.isEmpty()) {
			// Bewusst VOR addSelectionChangedListener!
			filteredTree.getViewer().setSelection(
					new StructuredSelection(result.toArray()), true);
		}
		// Ermögliche Übernahme des per Doppelklick selektierten Objektes und
		// automatisches Schließen des Wizards (Forderung von Herrn Bräuner
		// PrProt Simulation, Punkt 8)
		filteredTree.getViewer()
				.addDoubleClickListener(new FilteredTreeDoubleClickListener());
		filteredTree.getViewer().addSelectionChangedListener(
				new FilteredTreeSelectionChangedListener());
		filteredTree.getViewer()
				.setLabelProvider(new SystemObjectTypeLabelProvider());
		setControl(filteredTree);
		validatePage();
	}

	public List<SystemObject> getAuswahl() {
		return Collections.unmodifiableList(result);
	}

	private void validatePage() {
		setPageComplete(!result.isEmpty());
	}

	@Override
	public boolean performCancel() {
		result.clear();
		validatePage();
		return true;
	}

	@Override
	public boolean performFinish() {
		return true;
	}
}
