/*
 * Rahmenwerk-Plug-in "BitCtrl-Bibliotheken"
 *
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.bitctrl.modell.util.cache.ui.internal;

import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.runtime.IAdapterFactory;

import de.bsvrz.buv.rw.basislib.Rahmenwerk;
import de.bsvrz.buv.rw.basislib.dav.DavVerbindungsEvent;
import de.bsvrz.buv.rw.basislib.dav.DavVerbindungsListener;
import de.bsvrz.buv.rw.bitctrl.internal.RahmenwerkService;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.util.cache.AbstractCache;
import de.bsvrz.sys.funclib.bitctrl.modell.util.cache.AbstractDavVerbindungsCache;
import de.bsvrz.sys.funclib.bitctrl.modell.util.cache.AnzeigeVerfahrenCache;
import de.bsvrz.sys.funclib.bitctrl.modell.util.cache.AnzeigenCache;
import de.bsvrz.sys.funclib.bitctrl.modell.util.cache.AqCache;
import de.bsvrz.sys.funclib.bitctrl.modell.util.cache.MifCache;
import de.bsvrz.sys.funclib.bitctrl.modell.util.cache.ModellObjektCache;
import de.bsvrz.sys.funclib.bitctrl.modell.util.cache.MqCache;
import de.bsvrz.sys.funclib.bitctrl.modell.util.cache.NetzCache;
import de.bsvrz.sys.funclib.bitctrl.modell.util.cache.NetzCacheExtended;
import de.bsvrz.sys.funclib.bitctrl.modell.util.cache.RdsCache;
import de.bsvrz.sys.funclib.bitctrl.modell.util.cache.RdsMeldungenCache;
import de.bsvrz.sys.funclib.bitctrl.modell.util.cache.StrassenTeilSegmenteCache;
import de.bsvrz.sys.funclib.bitctrl.modell.util.cache.StreckenAbschnittCache;
import de.bsvrz.sys.funclib.bitctrl.modell.util.cache.TmcCache;
import de.bsvrz.sys.funclib.bitctrl.modell.util.cache.TmcLocationCodeCache;

/**
 * AdapterFactory für alle möglichen Arten von UI Caches.
 *
 * XXX: In sehr seltenen Fällen kommt es zu einer Art Deadlock, der aber meist
 * nicht als solcher in zu erkennen ist (es gibt kein blockierendes Lock).
 * Jedoch wird ein und der selbe Cache 2 Mal initialisiert und beide bleiben
 * ewig stehen, bzw. versenden keine "Fertig" Meldung. Ein vermutlicher Fix wäre
 * das synchronisieren der Verwendung der {@link CacheFactory#caches} Map - aber
 * die Frage ist, wie kann man das Testen?
 *
 * @author BitCtrl Systems GmbH 2013, Christian Hösel
 *
 *
 *
 */
public class CacheAdapterFactory implements IAdapterFactory {

	private abstract static class CacheFactory<T, C extends AbstractCache> {
		protected Map<T, C> caches = new HashMap<>();

		public abstract C getCache(T base);

		public void reset() {
			for (final AbstractCache cache : caches.values()) {
				cache.setState(AbstractCache.PROP_STATE_CANCELED);
			}
		}
	}

	private final Map<Class<?>, CacheFactory<?, ?>> factories;

	public CacheAdapterFactory() {

		final Rahmenwerk rahmenwerk = CacheServiceImpl.getService()
				.getRahmenWerk();
		rahmenwerk.addDavVerbindungsListener(new DavVerbindungsListener() {

			@Override
			public void verbindungHergestellt(final DavVerbindungsEvent event) {
				// keine Operation erforderlich
			}

			@Override
			public boolean verbindungHalten(final DavVerbindungsEvent event) {
				return false;
			}

			@Override
			public void verbindungGetrennt(final DavVerbindungsEvent event) {
				for (final CacheFactory<?, ?> factory : factories.values()) {
					factory.reset();
				}

			}
		});

		factories = new HashMap<>();
		factories.put(AbstractDavVerbindungsCache.class,
				new CacheFactory<Class<AbstractDavVerbindungsCache>, AbstractDavVerbindungsCache>() {

					@Override
					public AbstractDavVerbindungsCache getCache(
							final Class<AbstractDavVerbindungsCache> base) {
						return new EclipseDavVerbindungsCache();
					}
				});
		factories.put(ModellObjektCache.class,
				new CacheFactory<String, ModellObjektCache>() {
					@Override
					public ModellObjektCache getCache(final String pid) {
						ModellObjektCache moCache = caches.get(pid);
						if (moCache == null) {
							final AbstractDavVerbindungsCache verbindungsCache = CacheAdapterFactory.this
									.getCache(null,
											AbstractDavVerbindungsCache.class);
							moCache = new ModellObjektCache(pid,
									verbindungsCache);
							caches.put(pid, moCache);
						} else {
							cacheReinitRequested(moCache);
						}
						return moCache;
					}
				});
		factories.put(NetzCache.class, new CacheFactory<String, NetzCache>() {
			@Override
			public NetzCache getCache(final String netz) {
				NetzCache netzCache = caches.get(netz);
				if (netzCache == null) {
					final AbstractDavVerbindungsCache verbindungsCache = CacheAdapterFactory.this
							.getCache(null, AbstractDavVerbindungsCache.class);
					final ModellObjektCache moc = CacheAdapterFactory.this
							.getCache(netz, ModellObjektCache.class);
					netzCache = new EclipseNetzCache(moc, verbindungsCache);
					caches.put(netz, netzCache);
				} else {
					cacheReinitRequested(netzCache);
				}
				return netzCache;
			}
		});
		factories.put(TmcCache.class, new CacheFactory<String, TmcCache>() {
			@Override
			public TmcCache getCache(final String netz) {
				TmcCache tmcCache = caches.get(netz);
				if (tmcCache == null) {
					final NetzCache netzCache = CacheAdapterFactory.this
							.getCache(netz, NetzCache.class);
					final AbstractDavVerbindungsCache verbindungsCache = CacheAdapterFactory.this
							.getCache(null, AbstractDavVerbindungsCache.class);
					tmcCache = new EclipseTmcCache(netzCache, verbindungsCache);
					caches.put(netz, tmcCache);
				} else {
					cacheReinitRequested(tmcCache);
				}
				return tmcCache;
			}
		});
		factories.put(NetzCacheExtended.class,
				new CacheFactory<String, NetzCacheExtended>() {
					@Override
					public NetzCacheExtended getCache(final String netz) {
						NetzCacheExtended netzCacheExtended = caches.get(netz);
						if (netzCacheExtended == null) {
							final TmcCache tmcCache = CacheAdapterFactory.this
									.getCache(netz, TmcCache.class);
							netzCacheExtended = new EclipseNetzCacheExtended(
									tmcCache);
							caches.put(netz, netzCacheExtended);
						} else {
							cacheReinitRequested(netzCacheExtended);
						}
						return netzCacheExtended;
					}
				});
		factories.put(MqCache.class, new CacheFactory<String, MqCache>() {
			@Override
			public MqCache getCache(final String netz) {
				MqCache mqCache = caches.get(netz);
				if (mqCache == null) {
					final AbstractDavVerbindungsCache verbindungsCache = CacheAdapterFactory.this
							.getCache(null, AbstractDavVerbindungsCache.class);
					final NetzCache netzCache = CacheAdapterFactory.this
							.getCache(netz, NetzCache.class);
					mqCache = new EclipseMqCache(netzCache, verbindungsCache);
					caches.put(netz, mqCache);
				} else {
					cacheReinitRequested(mqCache);
				}
				return mqCache;
			}
		});
		factories.put(AqCache.class, new CacheFactory<String, AqCache>() {
			@Override
			public AqCache getCache(final String netz) {
				AqCache aqCache = caches.get(netz);
				if (aqCache == null) {
					final AbstractDavVerbindungsCache verbindungsCache = CacheAdapterFactory.this
							.getCache(null, AbstractDavVerbindungsCache.class);
					aqCache = new EclipseAqCache(verbindungsCache);
					caches.put(netz, aqCache);
				} else {
					cacheReinitRequested(aqCache);
				}
				return aqCache;
			}
		});
		factories.put(StrassenTeilSegmenteCache.class,
				new CacheFactory<String, StrassenTeilSegmenteCache>() {
					@Override
					public StrassenTeilSegmenteCache getCache(
							final String netz) {
						StrassenTeilSegmenteCache segmenteCache = caches
								.get(netz);
						if (segmenteCache == null) {
							final AbstractDavVerbindungsCache verbindungsCache = CacheAdapterFactory.this
									.getCache(null,
											AbstractDavVerbindungsCache.class);
							final NetzCacheExtended netzCacheExtended = CacheAdapterFactory.this
									.getCache(netz, NetzCacheExtended.class);
							segmenteCache = new EclipseStrassenTeilSegmenteCache(
									verbindungsCache, netzCacheExtended);
							caches.put(netz, segmenteCache);
						} else {
							cacheReinitRequested(segmenteCache);
						}
						return segmenteCache;
					}
				});
		factories.put(StreckenAbschnittCache.class,
				new CacheFactory<String, StreckenAbschnittCache>() {
					@Override
					public StreckenAbschnittCache getCache(final String netz) {
						StreckenAbschnittCache streckenAbschnittCache = caches
								.get(netz);
						if (streckenAbschnittCache == null) {
							final StrassenTeilSegmenteCache strassenTeilSegmenteCache = CacheAdapterFactory.this
									.getCache(netz,
											StrassenTeilSegmenteCache.class);
							streckenAbschnittCache = new EclipseStreckenAbschnittCache(
									strassenTeilSegmenteCache);
							caches.put(netz, streckenAbschnittCache);
						} else {
							cacheReinitRequested(streckenAbschnittCache);
						}
						return streckenAbschnittCache;
					}
				});
		factories.put(MifCache.class, new CacheFactory<String, MifCache>() {
			@Override
			public MifCache getCache(final String base) {
				MifCache mifCache = caches.get(base);
				if (mifCache == null) {
					final AbstractDavVerbindungsCache verbindungsCache = CacheAdapterFactory.this
							.getCache(null, AbstractDavVerbindungsCache.class);
					mifCache = new EclipseMifCache(verbindungsCache);
					caches.put(base, mifCache);
				} else {
					cacheReinitRequested(mifCache);
				}
				return mifCache;
			}
		});
		factories.put(AnzeigeVerfahrenCache.class,
				new CacheFactory<String, AnzeigeVerfahrenCache>() {
					@Override
					public AnzeigeVerfahrenCache getCache(final String base) {
						AnzeigeVerfahrenCache azvCache = caches.get(base);
						if (null == azvCache) {
							final AbstractDavVerbindungsCache verbindungsCache = CacheAdapterFactory.this
									.getCache(null,
											AbstractDavVerbindungsCache.class);
							azvCache = new EclipseAnzeigeVerfahrenCache(
									verbindungsCache);
							caches.put(base, azvCache);
						} else {
							cacheReinitRequested(azvCache);
						}
						return azvCache;
					}
				});
		factories.put(RdsCache.class, new CacheFactory<String, RdsCache>() {
			@Override
			public RdsCache getCache(final String base) {
				RdsCache rdsCache = caches.get(base);
				if (rdsCache == null) {
					final AbstractDavVerbindungsCache verbindungsCache = CacheAdapterFactory.this
							.getCache(null, AbstractDavVerbindungsCache.class);
					rdsCache = new EclipseRdsCache(verbindungsCache);
					caches.put(base, rdsCache);
				} else {
					cacheReinitRequested(rdsCache);
				}
				return rdsCache;
			}
		});
		factories.put(TmcLocationCodeCache.class,
				new CacheFactory<String, TmcLocationCodeCache>() {
					@Override
					public TmcLocationCodeCache getCache(final String base) {
						TmcLocationCodeCache tmcLocationCodeCache = caches
								.get(base);
						if (tmcLocationCodeCache == null) {
							final AbstractDavVerbindungsCache verbindungsCache = CacheAdapterFactory.this
									.getCache(null,
											AbstractDavVerbindungsCache.class);
							tmcLocationCodeCache = new EclipseTmcLocationCodeCache(
									verbindungsCache);
							caches.put(base, tmcLocationCodeCache);
						} else {
							cacheReinitRequested(tmcLocationCodeCache);
						}
						return tmcLocationCodeCache;
					}
				});
		factories.put(RdsMeldungenCache.class,
				new CacheFactory<String, RdsMeldungenCache>() {
					@Override
					public RdsMeldungenCache getCache(final String lms) {
						RdsMeldungenCache rdsMeldungenCache = caches.get(lms);
						if (rdsMeldungenCache == null) {
							final AbstractDavVerbindungsCache verbindungsCache = CacheAdapterFactory.this
									.getCache(null,
											AbstractDavVerbindungsCache.class);
							final ModellObjektCache moc = CacheAdapterFactory.this
									.getCache(lms, ModellObjektCache.class);
							final RdsCache rdsCache = CacheAdapterFactory.this
									.getCache(RdsCache.class, RdsCache.class);
							final TmcLocationCodeCache tmcLocationCodeCache = CacheAdapterFactory.this
									.getCache(TmcLocationCodeCache.class,
											TmcLocationCodeCache.class);
							rdsMeldungenCache = new EclipseRdsMeldungenCache(
									moc, verbindungsCache, rdsCache,
									tmcLocationCodeCache);
							caches.put(lms, rdsMeldungenCache);
						} else {
							cacheReinitRequested(rdsMeldungenCache);
						}
						return rdsMeldungenCache;
					}
				});

		factories.put(AnzeigenCache.class,
				new CacheFactory<String, AnzeigenCache>() {
					@Override
					public AnzeigenCache getCache(final String base) {
						AnzeigenCache anzeigenCache = caches.get(base);
						if (anzeigenCache == null) {
							final AbstractDavVerbindungsCache verbindungsCache = CacheAdapterFactory.this
									.getCache(null,
											AbstractDavVerbindungsCache.class);
							anzeigenCache = new EclipseAnzeigenCache(
									verbindungsCache);
							caches.put(base, anzeigenCache);
						} else {
							cacheReinitRequested(anzeigenCache);
						}
						return anzeigenCache;
					}
				});
	}

	protected void cacheReinitRequested(final AbstractCache cache) {

		if (cache != null
				&& AbstractCache.PROP_STATE_CANCELED.equals(cache.getState())) {
			final ObjektFactory objektFactory = RahmenwerkService.getService()
					.getObjektFactory();
			if ((objektFactory != null) && objektFactory.isVerbunden()) {
				cache.redoInit();
			}
		}

	}

	// @Override
	@Override
	public synchronized AbstractCache getAdapter(final Object adaptableObject,
			final Class adapterType) {
		if (adaptableObject instanceof final String s) {
			return getCache(s, adapterType);
		}
		if (adaptableObject instanceof final Class c) {
			return getCache(c, adapterType);
		}
		return null;
	}

	private synchronized <K, T2 extends AbstractCache> T2 getCache(final K netz,
			final Class<T2> adapterType) {
		final CacheFactory factory = factories.get(adapterType);
		if (null != factory) {
			return (T2) factory.getCache(netz);
		}
		return null;
	}

	@Override
	public Class[] getAdapterList() {
		return new Class[] { ModellObjektCache.class, NetzCache.class,
				NetzCacheExtended.class, TmcCache.class, MqCache.class,
				StreckenAbschnittCache.class, MifCache.class,
				StrassenTeilSegmenteCache.class, RdsCache.class,
				TmcLocationCodeCache.class, RdsMeldungenCache.class,
				AbstractDavVerbindungsCache.class, AnzeigeVerfahrenCache.class,
				AnzeigenCache.class };
	}
}
