/*
 * Rahmenwerk-Plug-in "BitCtrl-Bibliotheken"
 *
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.bitctrl.modell.util.cache.ui.internal;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.IOException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.osgi.util.NLS;

import de.bsvrz.buv.rw.basislib.einstellungen.SpeicherKey;
import de.bsvrz.buv.rw.bitctrl.eclipse.modell.java.JavaModellEinstellungen;
import de.bsvrz.sys.funclib.bitctrl.modell.util.cache.AbstractCache;
import de.bsvrz.sys.funclib.bitctrl.modell.util.cache.AbstractCache.CacheMonitor;

public class EclipseCacheMonitor
		implements CacheMonitor, PropertyChangeListener {
	private static final long DEFAULT_TIME = 10000; // 10 seconds

	private static final String DEFAULT_NAME = "default";

	public static class RunTimes {
		private Map<String, Long> times = new HashMap<>();

		public Map<String, Long> getTimes() {
			return times;
		}

		public void setTimes(final Map<String, Long> times) {
			this.times = times;
		}
	}

	private RunTimes runTimes;
	private final Map<String, Long> startTimes = new HashMap<>();

	private final class AllCachesProgressJob extends Job {
		private AllCachesProgressJob() {
			super(Messages.EclipseCacheMonitor_JobHeadText);
			IProgressMonitor.class.getName();
		}

		@Override
		protected IStatus run(final IProgressMonitor monitor) {
			synchronized (EclipseCacheMonitor.this) {
				try {
					int maxSize = 0;
					for (final Entry<AbstractCache, Double> entry : cacheProgress
							.entrySet()) {
						Long time = runTimes.getTimes()
								.get(entry.getKey().getClass().getName());
						if (null == time || 0 == time) {
							time = DEFAULT_TIME;
						}
						maxSize += time;
					}
					monitor.beginTask("", maxSize);
					monitor.worked(initCaches.size());
					int currentSize = 0;
					while (!waitingCaches.isEmpty() && !monitor.isCanceled()) {
						try {
							String message;
							if (cancelCaches.isEmpty()) {
								message = Messages.EclipseCacheMonitor_JobSubTextOk;
							} else {
								message = Messages.EclipseCacheMonitor_JobSubTextError;
							}
							final Object[] sizes = new Object[] {
									waitingCaches.size(), initCaches.size(),
									cancelCaches.size() };
							monitor.subTask(NLS.bind(message, sizes));
							EclipseCacheMonitor.this.wait();
							int newSize = 0;
							for (final Entry<AbstractCache, Double> entry : cacheProgress
									.entrySet()) {
								Long time = runTimes.getTimes().get(
										entry.getKey().getClass().getName());
								if (null == time) {
									time = DEFAULT_TIME;
								}
								newSize += entry.getValue() * time;
							}
							monitor.worked(newSize - currentSize);
							currentSize = newSize;
						} catch (final InterruptedException e) {
							e.printStackTrace();
							waitingCaches.clear();
							initCaches.clear();
							cacheProgress.clear();
							currentJob = null;
							monitor.setCanceled(true);
							return Status.CANCEL_STATUS;
						}
					}
					if (waitingCaches.isEmpty()) {
						initCaches.clear();
						cacheProgress.clear();
					}
					currentJob = null;
					monitor.done();
					return Status.OK_STATUS;
				} finally {
					EclipseCacheMonitor.this.notifyAll();
					final JavaModellEinstellungen<RunTimes> jme = new JavaModellEinstellungen<>(
							RunTimes.class);
					try {
						jme.setModellEinstellungen(SpeicherKey.allgemeinLokal(),
								DEFAULT_NAME, runTimes);
					} catch (final IOException e) {
						// TODO Auto-generated catch block
						e.printStackTrace();
					}
				}
			}
		}
	}

	private Job currentJob;
	private final Set<AbstractCache> waitingCaches = new HashSet<>();
	private final Set<AbstractCache> initCaches = new HashSet<>();
	private final Set<AbstractCache> cancelCaches = new HashSet<>();
	private final Map<AbstractCache, Double> cacheProgress = new HashMap<>();

	public EclipseCacheMonitor() {
		final JavaModellEinstellungen<RunTimes> jme = new JavaModellEinstellungen<>(
				RunTimes.class);
		runTimes = jme.getModellEinstellungen(SpeicherKey.allgemeinLokal(),
				DEFAULT_NAME);
		if (null == runTimes) {
			runTimes = new RunTimes();
		}
	}

	// @Override
	@Override
	public synchronized void monitorCache(final AbstractCache cache) {
		if (null != currentJob) {
			currentJob.cancel();
			currentJob = null;
			notifyAll();
			try {
				wait();
			} catch (final InterruptedException e) {
				e.printStackTrace();
			}
		}
		initCaches.remove(cache);
		cancelCaches.remove(cache);
		waitingCaches.add(cache);
		cacheProgress.put(cache, 0.0);
		cache.addPropertyChangeListener(AbstractCache.PROP_STATE, this);
		cache.addPropertyChangeListener(AbstractCache.PROP_PROGRESS, this);
		currentJob = new AllCachesProgressJob();
		currentJob.schedule();
	}

	// @Override
	@Override
	public synchronized void propertyChange(final PropertyChangeEvent evt) {
		final AbstractCache cache = (AbstractCache) evt.getSource();
		if (AbstractCache.PROP_STATE.equals(evt.getPropertyName())) {
			final String newValue = (String) evt.getNewValue();
			if (AbstractCache.PROP_STATE_INITIALIZED.equals(newValue)) {
				cache.removePropertyChangeListener(AbstractCache.PROP_STATE,
						this);
				cache.removePropertyChangeListener(AbstractCache.PROP_PROGRESS,
						this);
				cacheProgress.put(cache, 1.0);
				if (waitingCaches.contains(cache)) {
					waitingCaches.remove(cache);
					initCaches.add(cache);
					notifyAll();
				}
				final String className = cache.getClass().getName();
				final long time = System.currentTimeMillis()
						- startTimes.get(className);
				runTimes.getTimes().put(className, time);
			} else if (AbstractCache.PROP_STATE_CANCELED.equals(newValue)) {
				cache.removePropertyChangeListener(AbstractCache.PROP_STATE,
						this);
				cache.removePropertyChangeListener(AbstractCache.PROP_PROGRESS,
						this);
				waitingCaches.remove(cache);
				cancelCaches.add(cache);
				cacheProgress.put(cache, 1.0);
				notifyAll();
			} else if (AbstractCache.PROP_STATE_STARTED.equals(newValue)) {
				startTimes.put(cache.getClass().getName(),
						System.currentTimeMillis());
			}
		} else if (AbstractCache.PROP_PROGRESS.equals(evt.getPropertyName())) {
			cacheProgress.put(cache,
					Math.max(0.0, Math.min(1.0, (Double) evt.getNewValue())));
			notifyAll();
		}
	}
}
