/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2011 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.sys.funclib.bitctrl.modell.util.bmvew;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NavigableSet;
import java.util.concurrent.ConcurrentSkipListSet;

import javax.swing.event.EventListenerList;

import com.bitctrl.Constants;
import com.bitctrl.util.Interval;
import com.bitctrl.util.logging.LoggerTools;

import de.bsvrz.dav.daf.main.archive.ArchiveDataKind;
import de.bsvrz.sys.funclib.bitctrl.archiv.ArchivException;
import de.bsvrz.sys.funclib.bitctrl.modell.DatensatzUpdateEvent;
import de.bsvrz.sys.funclib.bitctrl.modell.DatensatzUpdateListener;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.SystemObjekt;
import de.bsvrz.sys.funclib.bitctrl.modell.bitctrlbmv.objekte.BcBetriebsMeldungsVerwaltung;
import de.bsvrz.sys.funclib.bitctrl.modell.bitctrlbmv.parameter.PdBcBetriebsmeldungDarstellung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmvewbetriebglobal.attribute.AtlUrlasser;
import de.bsvrz.sys.funclib.bitctrl.modell.tmvewbetriebglobal.attribute.AttMeldungsKlasse;
import de.bsvrz.sys.funclib.bitctrl.modell.tmvewbetriebglobal.attribute.AttMeldungsStatus;
import de.bsvrz.sys.funclib.bitctrl.modell.tmvewbetriebglobal.attribute.AttMeldungsTyp;
import de.bsvrz.sys.funclib.bitctrl.modell.tmvewbetriebglobal.objekte.BetriebsMeldungsVerwaltung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmvewbetriebglobal.objekte.InformationsKanal;
import de.bsvrz.sys.funclib.bitctrl.modell.tmvewbetriebglobal.onlinedaten.OdBetriebsMeldung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmvewbetriebglobal.onlinedaten.OdBetriebsMeldung.Daten;
import de.bsvrz.sys.funclib.bitctrl.modell.util.KappichModellUtil;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.operatingMessage.MessageCauser;
import de.bsvrz.sys.funclib.operatingMessage.MessageGrade;
import de.bsvrz.sys.funclib.operatingMessage.MessageSender;
import de.bsvrz.sys.funclib.operatingMessage.MessageState;
import de.bsvrz.sys.funclib.operatingMessage.MessageType;

/**
 * Hilfsklasse für den Umgang mit der Betriebsmeldungsverwaltung. Die Klasse
 * führt eine Liste der letzten Meldungen. Andere Klassen können sich über
 * Änderungen an dieser Liste informieren lassen.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann, Christian Hösel, Albrecht
 *         Uhlmann
 * @version $Id: Betriebsmeldungsverwaltung.java 28042 2010-12-21 10:29:43Z
 *          uhlmann $
 * @see de.bsvrz.sys.funclib.operatingMessage.MessageSender
 */
public final class Betriebsmeldungsverwaltung {

	/**
	 * PID der BitCtrl-Betriebsmeldungsverwaltung mit erweiterten Parametern.
	 *
	 * @deprecated die erweitereten Parameter der Betriebsmeldungsveraltung werden
	 *             jetzt als Benutzerparameter verwaltet
	 */
	@Deprecated
	public static final String PID_BITCTRL_BMV = "bitctrl.bmv";

	/**
	 * {@link Comparator} zum Vergleichen der Betriebsmeldungen nach Zeit.
	 *
	 * @author BitCtrl Systems GmbH, Ch. Hösel
	 * @version $Id: Betriebsmeldungsverwaltung.java 58515 2015-11-11 13:52:11Z
	 *          hoesel $
	 */
	private final class BetriebsmeldungZeitComparator implements Comparator<OdBetriebsMeldung.Daten> {
		@Override
		public int compare(final Daten o1, final Daten o2) {
			return o1.dGetZeitstempel().compareTo(o2.dGetZeitstempel());
		}
	}

	/**
	 * Liest die letzen Betriebsmeldungen aus dem Archiv und cacht diese; meldet
	 * sich auf neue Betriebsmeldungen als Empfänger an.
	 */
	private class Meldungsempfaenger implements DatensatzUpdateListener {

		@Override
		public void datensatzAktualisiert(final DatensatzUpdateEvent event) {
			final List<OdBetriebsMeldung.Daten> neu = new ArrayList<>();

			final OdBetriebsMeldung.Daten datum = (Daten) event.getDatum();

			if ((datum != null) && datum.dContainsDaten()) {
				if (event.getObjekt() == betriebsMeldungsVerwaltung) {

					synchronized (meldungsliste) {
						meldungsliste.add(datum);
						neu.add(datum);
					}
					final List<OdBetriebsMeldung.Daten> entfernt = cleanUpMeldungen(null);
					fireMeldungslisteChanged(false, neu, entfernt);
				} else if (event.getObjekt() instanceof InformationsKanal) {
					final InformationsKanal kanal = (InformationsKanal) event.getObjekt();

					if (informationskanaeleMeldungenMap.containsKey(kanal)) {
						informationskanaeleMeldungenMap.get(kanal).add(datum);
					} else {
						final NavigableSet<OdBetriebsMeldung.Daten> liste = new ConcurrentSkipListSet<>(
								new BetriebsmeldungZeitComparator());
						liste.add(datum);
						informationskanaeleMeldungenMap.put(kanal, liste);
					}
					neu.add(datum);
					final List<OdBetriebsMeldung.Daten> entfernt = cleanUpMeldungen(kanal);
					fireMeldungslisteInformationskanalChanged(false, kanal, neu, entfernt);
				}
			}
		}
	}

	/** Der Logger der Klasse. */
	private final Debug log;

	/** Die Liste der angemeldeten Listener. */
	private final EventListenerList listeners = new EventListenerList();

	/** Die Liste der angemeldeten Listener. */
	private final EventListenerList archivListeners = new EventListenerList();

	private final HashMap<InformationsKanal, EventListenerList> informationskanalListeners = new HashMap<>();
	private final HashMap<InformationsKanal, EventListenerList> archivInformationskanalListeners = new HashMap<>();

	/** Die Liste der gecachten Meldungen. */
	private final NavigableSet<OdBetriebsMeldung.Daten> meldungsliste = new ConcurrentSkipListSet<>(
			new BetriebsmeldungZeitComparator());

	private final HashMap<InformationsKanal, NavigableSet<OdBetriebsMeldung.Daten>> informationskanaeleMeldungenMap = new HashMap<>();

	/** Liste er Befehle die beim Meldungsempfang verarbeitet werden. */
	private final List<BetriebsmeldungCommand> befehlsliste;

	/** Empfänger der Betriebsmeldungen. */
	private Meldungsempfaenger empfaenger;

	/** Die Darstellungsparameter für Meldungen. */
	// private PdBcBetriebsmeldungDarstellung.Daten darstellungsparameter;

	/** Die ObjektFactory. */
	private final ObjektFactory factory;

	/** Das BetriebsMeldungsVerwaltung-Objekt im DAV. */
	private final BetriebsMeldungsVerwaltung betriebsMeldungsVerwaltung;

	/**
	 * Maximale Anzahl von Tagen, f&uuml;r die Meldungen intern vorgehalten werden.
	 */
	private long maxTage = 5;

	/**
	 * Gibt die maximale Anzahl von Tagen zur&uuml;ck, f&uuml;r die Meldungen intern
	 * vorgehalten werden.
	 *
	 * @return Tage
	 */
	public long getMaxTage() {
		return maxTage;
	}

	/**
	 * Setzt die maximale Anzahl von Tagen, f&uuml;r die Meldungen intern
	 * vorgehalten werden.
	 *
	 * @param maxTage Tage
	 */
	public void setMaxTage(final long maxTage) {
		this.maxTage = maxTage;
	}

	/**
	 * Gibt die maximale Anzahl von Meldungen zur&uuml;ck, die intern vorgehalten
	 * werden.
	 *
	 * @return Anzahl
	 */
	public int getMaxAnzahl() {
		return maxAnzahl;
	}

	/**
	 * Setzt die maximale Anzahl Meldungen, die intern vorgehalten werden.
	 *
	 * @param maxAnzahl Anzahl
	 */
	public void setMaxAnzahl(final int maxAnzahl) {
		this.maxAnzahl = maxAnzahl;
	}

	/** Maximale von Meldungen, die intern vorgehalten werden. */
	private int maxAnzahl = 100;

	/**
	 * Speichert pro {@link ObjektFactory} eine Instanz der Klasse
	 * {@link Betriebsmeldungsverwaltung}.
	 */
	private static final Map<ObjektFactory, Betriebsmeldungsverwaltung> betriebsmeldungsVerwaltungen = new HashMap<>();

	/**
	 * Liefert zu einer {@link ObjektFactory} immer dieselbe
	 * {@link Betriebsmeldungsverwaltung}.
	 *
	 * @param factory die Factory
	 * @return die BMV zur Factory
	 */
	public static Betriebsmeldungsverwaltung getInstanz(final ObjektFactory factory) {
		final Betriebsmeldungsverwaltung bmv;
		synchronized (betriebsmeldungsVerwaltungen) {
			if (betriebsmeldungsVerwaltungen.get(factory) == null) {
				betriebsmeldungsVerwaltungen.put(factory, new Betriebsmeldungsverwaltung(factory, null));
			}
			bmv = betriebsmeldungsVerwaltungen.get(factory);
		}
		return bmv;
	}

	/**
	 * Konstruktor mit BetriebsMeldungsVerwaltung.
	 *
	 * @param factory    ObjektFaktory
	 * @param verwaltung BetriebsMeldungsVerwaltung
	 */
	private Betriebsmeldungsverwaltung(final ObjektFactory factory, final BetriebsMeldungsVerwaltung verwaltung) {
		if (factory == null) {
			throw new IllegalArgumentException("Das Argument 'factory' darf nicht 'null' sein!");
		}

		this.factory = factory;
		log = Debug.getLogger();
		befehlsliste = new ArrayList<>();

		if (verwaltung == null) {
			betriebsMeldungsVerwaltung = KappichModellUtil.getAOE(factory);
		} else {
			betriebsMeldungsVerwaltung = verwaltung;
		}

		log.info("Betriebsmeldungsverwaltung konstruiert, " + "aber noch nicht angemeldet.");
	}

	/**
	 * Führt falls noch nicht geschehen die Empfängeranmeldung für Betriebsmeldungen
	 * und das Auslesen der letzten Meldungen aus dem Archiv aus. Die Operation wird
	 * in einem eigenen Thread ausgeführt, um eventuelle Listeneranmeldungen nicht
	 * zu blockieren.
	 */
	@SuppressWarnings("unchecked")
	private void anmeldenMeldungsempfaenger() {
		if (empfaenger != null) {
			// Anmeldung bereits durchgeführt.
			return;
		}

		empfaenger = new Meldungsempfaenger();

		final Thread bmvInitialisierer = new Thread("BMV initialisieren") {

			@Override
			public void run() {

				final long now = factory.getDav().getTime() + (30 * Constants.MILLIS_PER_SECOND);

				final OdBetriebsMeldung datensatzBetriebsMeldung = betriebsMeldungsVerwaltung.getOdBetriebsMeldung();
				if ((datensatzBetriebsMeldung != null) && factory.isVerbunden()) {

					// Als Empfänger für Betriebsmeldungen anmelden
					datensatzBetriebsMeldung.addUpdateListener(OdBetriebsMeldung.Aspekte.Information, empfaenger);

					log.info("Initiale Archivabfrage. maxTage=" + getMaxTage() + ", maxAnzahl=" + getMaxAnzahl());
					try {
						final Iterator<Daten> archivIterator = datensatzBetriebsMeldung.getArchivdatenIterator(
								OdBetriebsMeldung.Aspekte.Information,
								new Interval(now - (getMaxTage() * Constants.MILLIS_PER_DAY), now), false,
								ArchiveDataKind.ONLINE);
						final List<OdBetriebsMeldung.Daten> archiveMsgs = new ArrayList<>();
						while (archivIterator.hasNext()) {
							final Daten d = archivIterator.next();
							archiveMsgs.add(d);
						}

						meldungsliste.addAll(archiveMsgs);
						final List<OdBetriebsMeldung.Daten> entfernt = cleanUpMeldungen(null);
						archiveMsgs.removeAll(entfernt);
						fireMeldungslisteChanged(true, archiveMsgs, entfernt);

						log.info("Initiale Archivabfrage " + " erfolgreich beendet.");
					} catch (final ArchivException e) {
						log.error("Initiale Archivabfrage " + " fehlerhaft beendet.", e);
					}
				}

				final List<InformationsKanal> informationskanaele = (List<InformationsKanal>) factory
						.bestimmeModellobjekte(InformationsKanal.PID);
				for (final InformationsKanal i : informationskanaele) {

					i.getOdBetriebsMeldung().addUpdateListener(OdBetriebsMeldung.Aspekte.Information, empfaenger);

					try {

						final Iterator<Daten> archivIterator = i.getOdBetriebsMeldung().getArchivdatenIterator(
								OdBetriebsMeldung.Aspekte.Information,
								new Interval(now - (getMaxTage() * Constants.MILLIS_PER_DAY), now), false,
								ArchiveDataKind.ONLINE);
						NavigableSet<Daten> list = informationskanaeleMeldungenMap.get(i);
						if (list == null) {
							list = new ConcurrentSkipListSet<>(new BetriebsmeldungZeitComparator());
							informationskanaeleMeldungenMap.put(i, list);
						}
						final List<OdBetriebsMeldung.Daten> archiveMsgs = new ArrayList<>();
						while (archivIterator.hasNext()) {
							final Daten d = archivIterator.next();
							archiveMsgs.add(d);
						}
						list.addAll(archiveMsgs);
						final List<OdBetriebsMeldung.Daten> entfernt = cleanUpMeldungen(i);
						archiveMsgs.removeAll(entfernt);
						fireMeldungslisteInformationskanalChanged(true, i, archiveMsgs, entfernt);
						log.info("Initiale Archivabfrage " + i.getName() + " erfolgreich beendet.");
					} catch (final ArchivException e) {
						log.error("Initiale Archivabfrage " + i.getName() + " fehlerhaft beendet.", e);
					}
				}

				log.info("Betriebsmeldungsverwaltung bereit.");
			}
		};
		bmvInitialisierer.setDaemon(true);
		bmvInitialisierer.start();
	}

	/**
	 * Entfernt nicht mehr gültige Meldungen aus der Liste.
	 *
	 * @return die Liste der entfernten Meldungen.
	 */
	private List<OdBetriebsMeldung.Daten> cleanUpMeldungen(final InformationsKanal informationskanal) {
		final List<OdBetriebsMeldung.Daten> entfernt = new ArrayList<>();

		if (informationskanal == null) {
			// Entferne Meldungen, die zu alt sind.
			final long maxZeitstempel = System.currentTimeMillis() - (maxTage * Constants.MILLIS_PER_DAY);

			while ((meldungsliste.size() > maxAnzahl) || (!meldungsliste.isEmpty()
					&& (meldungsliste.first().dGetZeitstempel().getTime() < maxZeitstempel))) {
				entfernt.add(meldungsliste.pollFirst());
			}
		} else {

			final NavigableSet<Daten> list = informationskanaeleMeldungenMap.get(informationskanal);
			// Entferne Meldungen, die zu alt sind.
			final long maxZeitstempel = System.currentTimeMillis() - (maxTage * Constants.MILLIS_PER_DAY);

			while ((list.size() > maxAnzahl)
					|| (!list.isEmpty() && (list.first().dGetZeitstempel().getTime() < maxZeitstempel))) {
				entfernt.add(list.pollFirst());
			}
		}

		return entfernt;
	}

	/**
	 * Registriert einen Listener für Betriebsmeldungen. Die Anmeldung für den
	 * Empfang der Betriebsmeldungen und das Auslesen der letzten Meldungen aus dem
	 * Archiv erfolgt einmalig beim ersten Aufruf der Methode.
	 *
	 * @param l ein Listener.
	 */
	public void addBetriebsmeldungListener(final BetriebsmeldungListener l) {
		anmeldenMeldungsempfaenger();
		listeners.add(BetriebsmeldungListener.class, l);
	}

	/**
	 * Registriert einen Listener für Betriebsmeldungen. Die Anmeldung für den
	 * Empfang der Betriebsmeldungen und das Auslesen der letzten Meldungen aus dem
	 * Archiv erfolgt einmalig beim ersten Aufruf der Methode.
	 *
	 * @param l ein Listener.
	 */
	public void addBetriebsmeldungListenerMitArchiv(final BetriebsmeldungListener l) {
		anmeldenMeldungsempfaenger();
		archivListeners.add(BetriebsmeldungListener.class, l);
	}

	public void addInformationskanalListenerMitArchiv(final InformationsKanal informationskanal,
			final BetriebsmeldungListener l) {

		EventListenerList listenerList = archivInformationskanalListeners.get(informationskanal);
		anmeldenMeldungsempfaenger();
		if (listenerList == null) {
			listenerList = new EventListenerList();
			archivInformationskanalListeners.put(informationskanal, listenerList);
		}
		listenerList.add(BetriebsmeldungListener.class, l);
	}

	public void addInformationskanalListener(final InformationsKanal informationskanal,
			final BetriebsmeldungListener l) {

		EventListenerList listenerList = informationskanalListeners.get(informationskanal);
		anmeldenMeldungsempfaenger();
		if (listenerList == null) {
			listenerList = new EventListenerList();
			informationskanalListeners.put(informationskanal, listenerList);
		}
		listenerList.add(BetriebsmeldungListener.class, l);
	}

	public void removeInformationskanalListener(final InformationsKanal informationskanal,
			final BetriebsmeldungListener l) {

		EventListenerList listenerList = informationskanalListeners.get(informationskanal);
		if (listenerList != null) {
			listenerList.remove(BetriebsmeldungListener.class, l);
		}

		listenerList = archivInformationskanalListeners.get(informationskanal);
		if (listenerList != null) {
			listenerList.remove(BetriebsmeldungListener.class, l);
		}
	}

	/**
	 * Deregistriert einen Listener für Betriebsmeldungen.
	 *
	 * @param l ein Listener.
	 */
	public void removeBetriebsmeldungListener(final BetriebsmeldungListener l) {
		listeners.remove(BetriebsmeldungListener.class, l);
		archivListeners.remove(BetriebsmeldungListener.class, l);
	}

	/**
	 * Informiert die angemeldeten Listener über die Änderung der Meldungsliste.
	 *
	 * @param neu      die Liste der neu hinzugekommenen Meldungen.
	 * @param entfernt die Liste der entfernten Meldungen.
	 */
	private synchronized void fireMeldungslisteChanged(final boolean archiv, final List<OdBetriebsMeldung.Daten> neu,
			final List<OdBetriebsMeldung.Daten> entfernt) {
		final BetriebsmeldungEvent e = new BetriebsmeldungEvent(this, neu, entfernt);

		for (final BetriebsmeldungListener l : archivListeners.getListeners(BetriebsmeldungListener.class)) {
			l.meldungslisteChanged(e);
		}

		if (!archiv) {
			for (final BetriebsmeldungListener l : listeners.getListeners(BetriebsmeldungListener.class)) {
				l.meldungslisteChanged(e);
			}
		}
	}

	private void fireMeldungslisteInformationskanalChanged(final boolean archiv, final InformationsKanal kanal,
			final List<Daten> neu, final List<Daten> entfernt) {
		final BetriebsmeldungEvent e = new BetriebsmeldungEvent(kanal, neu, entfernt);

		EventListenerList list = archivInformationskanalListeners.get(kanal);
		if (list != null) {
			for (final BetriebsmeldungListener l : list.getListeners(BetriebsmeldungListener.class)) {
				l.meldungslisteChanged(e);
			}
		}

		if (!archiv) {
			list = informationskanalListeners.get(kanal);
			if (list != null) {
				for (final BetriebsmeldungListener l : list.getListeners(BetriebsmeldungListener.class)) {
					l.meldungslisteChanged(e);
				}
			}
		}

	}

	/**
	 * Fügt einen Befehl der Befehlsliste hinzu.
	 *
	 * @param befehl ein Befehl.
	 */
	public void addBefehl(final BetriebsmeldungCommand befehl) {
		befehlsliste.add(befehl);
	}

	/**
	 * Entfernt einen Befehl aus der Befehlsliste.
	 *
	 * @param befehl ein Befehl.
	 */
	public void removeBefehl(final BetriebsmeldungCommand befehl) {
		befehlsliste.remove(befehl);
	}

	/**
	 * Fragt, ob die Betriebsmeldungsverwaltung gestartet wurde. Dies ist nur der
	 * Fall, wenn der Datensatz nach der Anmeldung <em>Keine Daten</em> oder
	 * <em>Daten</em> liefert.
	 *
	 * @return {@code true}, wenn Meldungen empfangen werden können.
	 */
	// public boolean isBereit() {
	// final Status status = datensatzBetriebsMeldung.abrufenDatum(
	// OdBetriebsMeldung.Aspekte.Information).getDatenStatus();
	// return status == Status.DATEN || status == Status.KEINE_DATEN;
	// }

	/**
	 * Gibt die gecachter Meldungen zurück. Die Anmeldung für den Empfang der
	 * Betriebsmeldungen und das Auslesen der letzten Meldungen aus dem Archiv
	 * erfolgt einmalig beim ersten Aufruf der Methode.
	 *
	 *
	 * @return eine unveränderliche Liste als Kopie der aktuellen Meldungen.
	 */
	public List<OdBetriebsMeldung.Daten> getMeldungsliste() {
		List<OdBetriebsMeldung.Daten> bliste = new ArrayList<>();
		anmeldenMeldungsempfaenger();
		bliste = Collections.unmodifiableList(new ArrayList<>(meldungsliste));
		return bliste;
	}

	public List<OdBetriebsMeldung.Daten> getMeldungsliste(final InformationsKanal kanal) {
		List<OdBetriebsMeldung.Daten> bliste = new ArrayList<>();

		anmeldenMeldungsempfaenger();
		final NavigableSet<Daten> list = informationskanaeleMeldungenMap.get(kanal);
		if (list == null) {
			return new ArrayList<>();
		}
		bliste = new ArrayList<>(Collections.unmodifiableSet(list));
		return bliste;
	}

	/**
	 * Sendet eine Betriebsmeldung.
	 *
	 * @param typ    der Meldungstyp.
	 * @param mtz    der Meldungstypzusatz.
	 * @param klasse die Meldungsklasse.
	 * @param text   der Meldungstext.
	 */
	// public void sende(final AttMeldungsTyp typ, final MeldungsTypZusatz mtz,
	// final AttMeldungsKlasse klasse, final String text) {
	// getSender().sendMessage(
	// getMessageType(typ),
	// getMessageTypeAddOn(mtz.compile()),
	// getMessageGrade(klasse),
	// null,
	// new MessageCauser(factory..getBenutzer()
	// .getSystemObject(), "", ""), text);
	// }

	/**
	 * Sendet eine Betriebsmeldung.
	 *
	 * @param typ    der Meldungstyp.
	 * @param mtz    der Meldungstypzusatz.
	 * @param klasse die Meldungsklasse.
	 * @param status der Meldungsstatus.
	 * @param text   der Meldungstext.
	 */
	public void sende(final AttMeldungsTyp typ, final MeldungsTypZusatz mtz, final AttMeldungsKlasse klasse,
			final AttMeldungsStatus status, final String text) {
		getSender().sendMessage("", getMessageType(typ), getMessageTypeAddOn(mtz.compile()), getMessageGrade(klasse),
				getMessageState(status), text);
	}

	/**
	 * Sendet eine Betriebsmeldung.
	 *
	 * @param typ      der Meldungstyp.
	 * @param mtz      der Meldungstypzusatz.
	 * @param klasse   die Meldungsklasse.
	 * @param status   der Meldungsstatus.
	 * @param referenz ein Systemobjekt auf das sich die Meldung bezieht.
	 * @param text     der Meldungstext.
	 * @param urlasser die Urlasserinformation.
	 */
	public void sende(final AttMeldungsTyp typ, final MeldungsTypZusatz mtz, final AttMeldungsKlasse klasse,
			final AttMeldungsStatus status, final SystemObjekt referenz, final String text,
			final AtlUrlasser urlasser) {
		getSender().sendMessage("", getMessageType(typ), getMessageTypeAddOn(mtz.compile()), getMessageGrade(klasse),
				referenz != null ? referenz.getSystemObject() : null, getMessageState(status),
				new MessageCauser(urlasser.getBenutzerReferenz().getSystemObject(), urlasser.getUrsache(),
						urlasser.getVeranlasser()),
				text);
	}

	/**
	 * Sendet eine Betriebsmeldung.
	 * <p>
	 * <em>Hinweis:</em> Diese Methode ist nur für das erneute Senden (Quittieren,
	 * Kommentieren oder Wiederholen) einer empfangenen Meldung gedacht.
	 *
	 * @param meldung eine Meldung.
	 */
	public void sende(final OdBetriebsMeldung.Daten meldung) {
		getSender().sendMessage(meldung.getID(), getMessageType(meldung.getMeldungsTyp()),
				getMessageTypeAddOn(meldung.getMeldungsTypZusatz()), getMessageGrade(meldung.getMeldungsKlasse()),
				((SystemObjekt) meldung.getReferenz().get(0)).getSystemObject(), getMessageState(meldung.getStatus()),
				new MessageCauser(meldung.getUrlasser().getBenutzerReferenz().getSystemObject(),
						meldung.getUrlasser().getUrsache(), meldung.getUrlasser().getVeranlasser()),
				meldung.getMeldungsText());
	}

	/**
	 * Konvertiert von einem Modellwert in einen Datenverteilerwert.
	 *
	 * @param typ ein Meldungstyp im Modell.
	 * @return der Meldungstyp im Datenverteiler.
	 */
	public MessageType getMessageType(final AttMeldungsTyp typ) {
		if (AttMeldungsTyp.ZUSTAND_0_SYSTEM.equals(typ)) {
			return MessageType.SYSTEM_DOMAIN;
		} else if (AttMeldungsTyp.ZUSTAND_1_FACH.equals(typ)) {
			return MessageType.APPLICATION_DOMAIN;
		} else {
			log.warning("Unbekannter Meldungstyp: " + typ);
			return MessageType.APPLICATION_DOMAIN;
		}
	}

	/**
	 * Konvertiert von einem Modellwert in einen Datenverteilerwert.
	 *
	 * @param klasse eine Meldungsklasse im Modell.
	 * @return die Meldungsklasse im Datenverteiler.
	 */
	public MessageGrade getMessageGrade(final AttMeldungsKlasse klasse) {

		if (AttMeldungsKlasse.ZUSTAND_0_FATAL.equals(klasse)) {
			return MessageGrade.FATAL;
		} else if (AttMeldungsKlasse.ZUSTAND_1_FEHLER.equals(klasse)) {
			return MessageGrade.ERROR;
		} else if (AttMeldungsKlasse.ZUSTAND_2_WARNUNG.equals(klasse)) {
			return MessageGrade.WARNING;
		} else if (AttMeldungsKlasse.ZUSTAND_3_INFORMATION.equals(klasse)) {
			return MessageGrade.INFORMATION;
		} else {
			log.warning("Unbekannte Meldungsklasse: " + klasse);
			return MessageGrade.INFORMATION;
		}
	}

	/**
	 * Konvertiert von einem Modellwert in einen Datenverteilerwert.
	 *
	 * @param status ein Meldungsstatus im Modell.
	 * @return der Meldungsstatus im Datenverteiler.
	 */
	private MessageState getMessageState(final AttMeldungsStatus status) {

		if (AttMeldungsStatus.ZUSTAND_0_MELDUNG.equals(status)) {
			return MessageState.MESSAGE;
		} else if (AttMeldungsStatus.ZUSTAND_2_NEUE_MELDUNG.equals(status)) {
			return MessageState.NEW_MESSAGE;
		} else if (AttMeldungsStatus.ZUSTAND_3_WIEDERHOLUNGSMELDUNG.equals(status)) {
			return MessageState.REPEAT_MESSAGE;
		} else if (AttMeldungsStatus.ZUSTAND_4_AENDERUNGSMELDUNG.equals(status)) {
			return MessageState.CHANGE_MESSAGE;
		} else if (AttMeldungsStatus.ZUSTAND_1_GUTMELDUNG.equals(status)) {
			return MessageState.GOOD_MESSAGE;
		} else {
			log.warning("Unbekannter Meldungsstatus: " + status);
			return MessageState.MESSAGE;
		}
	}

	/**
	 * Generiert falls nötig einen Standardmeldungszusatz.
	 *
	 * @param meldungsTypZusatz ein Zusatz, der {@code null} oder ein Leerstring
	 *                          sein kann.
	 * @return ein gültiger Meldungstypzusatz.
	 */
	private String getMessageTypeAddOn(final String meldungsTypZusatz) {
		if ((meldungsTypZusatz == null) || (meldungsTypZusatz.length() == 0)) {
			return LoggerTools.getCallPosition(new Throwable());
		}
		return meldungsTypZusatz;
	}

	/**
	 * Gibt den Sender von Betriebsmeldungen zurück. Die Methode ist identisch mit
	 * dem Aufruf von {@link MessageSender#getInstance()}. Diese Methode dient
	 * lediglich dazu, dass in Klassen nicht zwei Betriebsmeldungsverwaltungen
	 * verwendet werden müssen.
	 *
	 * @return der Sender.
	 */
	public MessageSender getSender() {
		return MessageSender.getInstance();
	}

	/**
	 * Gibt die aktuellen Darstellungsparameter für Betriebsmeldungen zurück.
	 *
	 * @return die Darstellungsparameter.
	 * @deprecated Soll nicht mehr verwendet werden. Stattdessen werden die
	 *             Darstellungen für Betriebsmeldungen im Betriebsmeldungsplugin als
	 *             Benutzerparameter verwaltet.
	 */
	@Deprecated
	public PdBcBetriebsmeldungDarstellung.Daten getDarstellungsparameter() {
		final BcBetriebsMeldungsVerwaltung bvBmv = (BcBetriebsMeldungsVerwaltung) factory
				.getModellobjekt(PID_BITCTRL_BMV);
		if (bvBmv != null) {
			final PdBcBetriebsmeldungDarstellung param = bvBmv.getPdBcBetriebsmeldungDarstellung();
			final PdBcBetriebsmeldungDarstellung.Daten dp = param.getDatum();
			if ((null != dp) && dp.dContainsDaten()) {
				setMaxTage(dp.getMaxHistory().getValue());
				setMaxAnzahl(dp.getMaxAnzahl().getValue());
			}
			return dp;
		}
		return null;
	}

}
