/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2011 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.sys.funclib.bitctrl.modell.util.bmvew;

import java.text.ParseException;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;

import de.bsvrz.sys.funclib.bitctrl.modell.SystemObjekt;

/**
 * Hilfsklasse zum Generieren und parsen von Meldungstypzusätzen.
 *
 * @author BitCtrl Systems GmbH, Falko
 * @version $Id$
 */
public class MeldungsTypZusatz {

	public static final String SYM_PARAMETER = "::";

	public static final String SYM_ASSIGN = ":=";

	public static final String SYM_DELIMITER = ";;";

	private final Map<String, String> parameter = new HashMap<>();
	private final String id;

	public static MeldungsTypZusatz parse(final String meldungsTypZusatz) throws ParseException {
		final MeldungsTypZusatz mtz;
		String[] rest = meldungsTypZusatz.split(SYM_PARAMETER);

		if (rest.length == 1) {
			mtz = new MeldungsTypZusatz(rest[0]);
		} else if (rest.length == 2) {
			mtz = new MeldungsTypZusatz(rest[0]);

			rest = rest[1].split(SYM_DELIMITER);
			for (final String param : rest) {
				final String[] s = param.split(SYM_ASSIGN);
				if (s.length == 1) {
					mtz.set(s[0], "");
				} else if (s.length == 2) {
					mtz.set(s[0], s[1]);
				} else {
					throw new ParseException("Das Trennzeichen \"" + SYM_ASSIGN
							+ "\" kommt mehr als einmal im Parameter \"" + s[0] + "\" vor.", 0);
				}
			}
		} else {
			throw new ParseException("Das Trennzeichen \"" + SYM_PARAMETER + "\" kommt mehr als einmal vor.", 0);
		}

		return mtz;
	}

	public MeldungsTypZusatz(final String id) {
		this.id = id;
	}

	public String compile() {
		StringBuilder s = new StringBuilder(Objects.toString(id, ""));

		if (!parameter.isEmpty()) {
			s.append(SYM_PARAMETER);
			final Iterator<Entry<String, String>> iterator = parameter.entrySet().iterator();
			while (iterator.hasNext()) {
				final Entry<String, String> e = iterator.next();
				s.append(e.getKey()).append(SYM_ASSIGN).append(e.getValue());
				if (iterator.hasNext()) {
					s.append(SYM_DELIMITER);
				}
			}
		}

		return s.toString();
	}

	/**
	 * @return the id
	 */
	public String getId() {
		return id;
	}

	/**
	 * Gibt eine unveränderliche Kopie der Paramter zurück.
	 *
	 * @return die Kopie der Parameter.
	 */
	public Map<String, String> getParameter() {
		return Collections.unmodifiableMap(parameter);
	}

	/**
	 * prüft ob der Schlüssel als Parameter zur Verfügung steht.
	 *
	 * @param name der Schlüssel, der geprüft werden soll
	 * @return der Zustand
	 */
	public boolean containsKey(final String name) {
		return parameter.containsKey(name);
	}

	/**
	 * liefert den zu dem übergebenen Schlüssel definierten String.
	 *
	 * @param name der Schlüssel
	 * @return der definierte String oder null
	 */
	public String getString(final String name) {
		return parameter.get(name);
	}

	/**
	 * setzt den String der über den angegebenen Schlüssel referenziert wird.
	 *
	 * @param name  der Schlüssel
	 * @param value der String
	 */
	public void set(final String name, final String value) {
		parameter.put(name, value);
	}

	/**
	 * Legt einen Leerstring ab, wenn {@code value == null}.
	 *
	 * @param name  der Schlüssel
	 * @param value der Wert
	 */
	public void set(final String name, final SystemObjekt value) {
		if (value != null) {
			parameter.put(name, value.getPid());
		} else {
			parameter.put(name, "");
		}
	}

	/**
	 * liefert den Wert zum übergebenen Schlüssels als Integer.
	 *
	 * @param name der Schlüssel
	 * @return der Wert
	 */
	public int getInteger(final String name) {
		return Integer.parseInt(parameter.get(name));
	}

	/**
	 * setzt einen Integerwert für den übergebenen Schlüssel. Der Wert wird intern
	 * als String verwaltet.
	 *
	 * @param name  der Schlüssel
	 * @param value der Wert
	 */
	public void set(final String name, final int value) {
		parameter.put(name, String.valueOf(value));
	}

	/**
	 * liefert den Wert zum übergebenen Schlüssels als Long.
	 *
	 * @param name der Schlüssel
	 * @return der Wert
	 */
	public long getLong(final String name) {
		return Long.parseLong(parameter.get(name));
	}

	/**
	 * setzt einen Longwert für den übergebenen Schlüssel. Der Wert wird intern als
	 * String verwaltet.
	 *
	 * @param name  der Schlüssel
	 * @param value der Wert
	 */
	public void set(final String name, final long value) {
		parameter.put(name, String.valueOf(value));
	}

	/**
	 * liefert den Wert zum übergebenen Schlüssels als double.
	 *
	 * @param name der Schlüssel
	 * @return der Wert
	 */
	public double getDouble(final String name) {
		return Double.parseDouble(parameter.get(name));
	}

	/**
	 * setzt einen Doublewert für den übergebenen Schlüssel. Der Wert wird intern
	 * als String verwaltet.
	 *
	 * @param name  der Schlüssel
	 * @param value der Wert
	 */
	public void set(final String name, final double value) {
		parameter.put(name, String.valueOf(value));
	}

	/**
	 * liefert den Wert zum übergebenen Schlüssels als boolean.
	 *
	 * @param name der Schlüssel
	 * @return der Wert
	 */
	public boolean getBoolean(final String name) {
		return Boolean.parseBoolean(parameter.get(name));
	}

	/**
	 * setzt einen Boolean-Wert für den übergebenen Schlüssel. Der Wert wird intern
	 * als String verwaltet.
	 *
	 * @param name  der Schlüssel
	 * @param value der Wert
	 */
	public void set(final String name, final boolean value) {
		parameter.put(name, String.valueOf(value));
	}

	/**
	 * Der Meldungstypzusatz ist gleich, wenn Id und Parameter übereinstimmen. Die
	 * Reihenfolge der Parameter spielt keine Rolle.
	 *
	 * {@inheritDoc}
	 */
	@Override
	public boolean equals(final Object obj) {
		if (obj == this) {
			return true;
		} else if (obj instanceof MeldungsTypZusatz) {
			final MeldungsTypZusatz o = (MeldungsTypZusatz) obj;
			return Objects.equals(id, o.id) && Objects.equals(parameter, o.parameter);
		}
		return false;
	}

	/**
	 * Gibt den Hash-Wert der Id zurück.
	 *
	 * {@inheritDoc}
	 *
	 * @see #getId()
	 */
	@Override
	public int hashCode() {
		return Objects.hashCode(id);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString() {
		String s;

		s = getClass().getName() + "[";
		s += "id=" + id;
		s += ", paramater=" + parameter;
		s += "]";

		return s;
	}

}
