/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2011 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.sys.funclib.bitctrl.modell.util.cache;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.SortedSet;
import java.util.TreeSet;

import com.bitctrl.util.monitor.IMonitor;
import com.bitctrl.util.monitor.SubMonitor;

import de.bsvrz.sys.funclib.bitctrl.geolib.WGS84Koordinate;
import de.bsvrz.sys.funclib.bitctrl.geolib.WGS84Polygon;
import de.bsvrz.sys.funclib.bitctrl.geolib.WGS84Punkt;
import de.bsvrz.sys.funclib.bitctrl.modell.SystemObjekt;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.attribute.AttLaenge;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.konfigurationsdaten.KdBestehtAusLinienObjekten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.konfigurationsdaten.KdLinienKoordinaten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.konfigurationsdaten.KdPunktKoordinaten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.konfigurationsdaten.KdPunktLiegtAufLinienObjekt;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.MessQuerschnittAllgemein;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.StrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.StrassenTeilSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.util.DavCacheLader;
import de.bsvrz.sys.funclib.bitctrl.modell.util.geolib.WGS84Util;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Cache fuer Anzeigeverfahren.
 * 
 * @author BitCtrl Systems GmbH, thierfelder
 * 
 *         XXX Prüfen, warum der Serializable implemntiert hatte
 * 
 */
public class AnzeigeVerfahrenCache extends AbstractCache {

	/**
	 * Mappt Strassensegmente auf eine nach Offset sortierte Liste der auf ihnen
	 * gelegenen MQs.
	 */
	private Map<StrassenSegment, SortedSet<MqaMitPosition>> segmentToMqList;

	/**
	 * Mappt die Strassensegmente auf deren Polygone.
	 */
	private Map<StrassenSegment, WGS84Polygon> segmentToPolygon;

	/**
	 * Mappt Messquerschnitte auf Strassensegmente.
	 */
	private Map<MessQuerschnittAllgemein, StrassenSegment> mqToSegment;

	/**
	 * Mappt Messquerschnitte auf deren Position auf ihrem Strassensegment.
	 */
	private Map<MessQuerschnittAllgemein, Double> mqToOffset;

	private boolean initialisiert = false;

	/**
	 * DAV-Verbindung.
	 */
	private AbstractDavVerbindungsCache davVerbindungsCache;

	private AnzeigeVerfahrenCache() {
		//
	}

	/**
	 * Konstruktor.
	 * 
	 * @param davVerbCache Verbindung zum Datenverteiler.
	 */
	public AnzeigeVerfahrenCache(final AbstractDavVerbindungsCache davVerbCache) {
		super("AZV-Cache", davVerbCache);
	}

	/**
	 * Vor-initialisierung.
	 * 
	 * @param davVerbCache Verbindung zum Datenverteiler.
	 */
	public void preInit(final AbstractDavVerbindungsCache davVerbCache) {
		davVerbindungsCache = davVerbCache;
	}

	@Override
	protected boolean doInit(final IMonitor monitor) {
		if (!initialisiert) {
			segmentToMqList = new HashMap<>();
			segmentToPolygon = new HashMap<>();
			mqToSegment = new HashMap<>();
			mqToOffset = new HashMap<>();

			initialisiert = true;
			try {
				DavCacheLader.lade(davVerbindungsCache.getClientDavInterface(), MessQuerschnittAllgemein.PID,
						KdPunktLiegtAufLinienObjekt.PID);
				DavCacheLader.lade(davVerbindungsCache.getClientDavInterface(), MessQuerschnittAllgemein.PID,
						KdPunktKoordinaten.PID);
				DavCacheLader.lade(davVerbindungsCache.getClientDavInterface(), StrassenSegment.PID,
						KdBestehtAusLinienObjekten.PID);
				DavCacheLader.lade(davVerbindungsCache.getClientDavInterface(), StrassenTeilSegment.PID,
						KdLinienKoordinaten.PID);

				final List<? extends SystemObjekt> mqaList = davVerbindungsCache.getObjektFactory()
						.bestimmeModellobjekte(MessQuerschnittAllgemein.PID);

				final SubMonitor subMonitor = SubMonitor.convert(monitor);
				subMonitor.beginTask("Initialisiere AZV-Cache ...", mqaList.size());

				/**
				 * Ordne allen Strassensegmenten eine in Fahrtrichtung geordnete Liste aller auf
				 * ihm befindlichen MQs zu.
				 */
				for (final SystemObjekt mqs : mqaList) {
					subMonitor.worked(1);
					final MessQuerschnittAllgemein mq = (MessQuerschnittAllgemein) mqs;
					final KdPunktLiegtAufLinienObjekt kdPunktLiegtAufLinienObjekt = mq.getKdPunktLiegtAufLinienObjekt();
					if (kdPunktLiegtAufLinienObjekt != null) {
						final de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.konfigurationsdaten.KdPunktLiegtAufLinienObjekt.Daten data = kdPunktLiegtAufLinienObjekt
								.getDatum();
						if ((data != null) && (data.getLinienReferenz() != null) && (data.getOffset() != null)) {
							final StrassenSegment segment = (StrassenSegment) data.getLinienReferenz();
							final AttLaenge laenge = data.getOffset();
							if (laenge != null) {
								final double offset = laenge.doubleValue();
								mqToSegment.put(mq, segment);
								mqToOffset.put(mq, offset);

								SortedSet<MqaMitPosition> mqList = segmentToMqList.get(segment);
								if (mqList == null) {
									mqList = new TreeSet<>();
									segmentToMqList.put(segment, mqList);

									final List<WGS84Punkt> punkte = new ArrayList<>();
									for (final WGS84Koordinate punkt : WGS84Util.konvertiere(segment)) {
										punkte.add(new WGS84Punkt(punkt));
									}
									final WGS84Polygon polygon = new WGS84Polygon(punkte);

									// /**
									// * Debug: Start.
									// */
									// Double x1 = null;
									// Double y1 = null;
									// Double x2 = null;
									// Double y2 = null;
									// int i = 0;
									// for (WGS84Punkt p : polygon
									// .getKoordinaten()) {
									// i++;
									// if (x1 == null) {
									// x1 = p.getBreite();
									// y1 = p.getLaenge();
									// } else {
									// x2 = p.getBreite();
									// y2 = p.getLaenge();
									//
									// double len = Math.sqrt((x2 - x1)
									// * (x2 - x1) + (y2 - y1)
									// * (y2 - y1));
									// if (Double.isNaN(len)) {
									// String s = "!!!!!NaN!!!!!: "
									// + len + ", MQ: " + mq
									// + ", Segm: " + segment;
									// Debug.getLogger().error(s);
									// System.out.println(s);
									// System.err.println(s);
									// } else if (len > 0.01) {
									// DecimalFormat decFormat = new
									// DecimalFormat(
									// "###.######");
									// String xy = "";
									// int j = 0;
									// for (WGS84Punkt p2 : polygon
									// .getKoordinaten()) {
									// j++;
									// xy += " ("
									// + decFormat
									// .format(p2
									// .getBreite())
									// + ", "
									// + decFormat
									// .format(p2
									// .getLaenge())
									// + ")"
									// + (j == i ? "* ("
									// + len + ")"
									// : "")
									// + "\n";
									// }
									//
									// String s = "!!!!!ZACKE!!!!!: "
									// + len + ", MQ: " + mq
									// + ", Segm: " + segment
									// + "\nKoordinaten:\n"
									// + xy;
									// Debug.getLogger().error(s);
									// System.out.println(s);
									// System.err.println(s);
									// }
									// x1 = x2;
									// y1 = y2;
									// }
									// }
									// /**
									// * Debug: Ende.
									// */

									segmentToPolygon.put(segment, polygon);
								}
								mqList.add(new MqaMitPosition(mq, offset));
							}
						}
					}
				}

				return true;
			} catch (final Exception ex) {
				ex.printStackTrace();
				Debug.getLogger().error("Initialisierung fehlgeschlagen.", ex);
			}
		} else {
			return true;
		}
		return false;
	}

	/**
	 * Erfragt Map mit Strassensegmente auf eine nach Offset sortierte Liste der auf
	 * ihnen gelegenen MQs.
	 * 
	 * @return Map mit Strassensegmente auf eine nach Offset sortierte Liste der auf
	 *         ihnen gelegenen MQs.
	 */
	public final Map<StrassenSegment, SortedSet<MqaMitPosition>> getSegmentToMqList() {
		if (!isInitialisiert()) {
			throw new IllegalStateException("Noch nicht initialisiert");
		}
		return segmentToMqList;
	}

	/**
	 * Erfragt Map mit Strassensegmente auf deren Polygone.
	 * 
	 * @return Map mit Strassensegmente auf deren Polygone.
	 */
	public final Map<StrassenSegment, WGS84Polygon> getSegmentToPolygon() {
		if (!isInitialisiert()) {
			throw new IllegalStateException("Noch nicht initialisiert");
		}
		return segmentToPolygon;
	}

	/**
	 * Erfragt Messquerschnitte auf Strassensegmente.
	 * 
	 * @return Messquerschnitte auf Strassensegmente.
	 */
	public final Map<MessQuerschnittAllgemein, StrassenSegment> getMqToSegment() {
		if (!isInitialisiert()) {
			throw new IllegalStateException("Noch nicht initialisiert");
		}
		return mqToSegment;
	}

	/**
	 * Erfragt Map Messquerschnitte auf deren Position auf ihrem Strassensegment.
	 * 
	 * @return Map Messquerschnitte auf deren Position auf ihrem Strassensegment.
	 */
	public final Map<MessQuerschnittAllgemein, Double> getMqToOffset() {
		if (!isInitialisiert()) {
			throw new IllegalStateException("Noch nicht initialisiert");
		}
		return mqToOffset;
	}

	/**
	 * Allgemeiner Messquerschnitt mit der Position, an der er sich auf seinem
	 * Segment befindet.
	 * 
	 * @author BitCtrl Systems GmbH, thierfelder
	 * @version $Id$
	 */
	public static class MqaMitPosition implements Comparable<MqaMitPosition> {

		/**
		 * Allgemeiner MQ.
		 */
		public final MessQuerschnittAllgemein mqa;

		/**
		 * Die Position (Offset), an der der MQ sich auf seinem Segment befindet.
		 */
		public final double offset;

		/**
		 * Standardkonstruktor.
		 * 
		 * @param mqa    Allgemeiner MQ.
		 * @param offset die Position (Offset), an der der MQ sich auf seinem Segment
		 *               befindet.
		 */
		protected MqaMitPosition(final MessQuerschnittAllgemein mqa, final double offset) {
			this.mqa = mqa;
			this.offset = offset;
		}

		@Override
		public int compareTo(final MqaMitPosition that) {
			return new Double(offset).compareTo(that.offset);
		}

		@Override
		public int hashCode() {
			return Objects.hash(offset);
		}

		@Override
		public boolean equals(final Object obj) {
			if (this == obj) {
				return true;
			}
			if (obj == null) {
				return false;
			}
			if (getClass() != obj.getClass()) {
				return false;
			}
			final MqaMitPosition other = (MqaMitPosition) obj;
			if (Double.doubleToLongBits(offset) != Double.doubleToLongBits(other.offset)) {
				return false;
			}
			return true;
		}

		@Override
		public String toString() {
			return mqa + " (" + offset + ")";
		}

	}

}
