/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2011 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.bitctrl.modell.util.cache;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import com.bitctrl.util.monitor.IMonitor;
import com.bitctrl.util.monitor.SubMonitor;

import de.bsvrz.sys.funclib.bitctrl.modell.SystemObjekt;
import de.bsvrz.sys.funclib.bitctrl.modell.att.Feld;
import de.bsvrz.sys.funclib.bitctrl.modell.metamodellglobal.objekte.KonfigurationsObjekt;
import de.bsvrz.sys.funclib.bitctrl.modell.tmanzeigenglobal.konfigurationsdaten.KdAnzeige;
import de.bsvrz.sys.funclib.bitctrl.modell.tmanzeigenglobal.konfigurationsdaten.KdAnzeigeQuerschnitt;
import de.bsvrz.sys.funclib.bitctrl.modell.tmanzeigenglobal.objekte.Anzeige;
import de.bsvrz.sys.funclib.bitctrl.modell.tmanzeigenglobal.objekte.AnzeigeQuerschnitt;
import de.bsvrz.sys.funclib.bitctrl.modell.tmanzeigenglobal.objekte.AnzeigeTyp;
import de.bsvrz.sys.funclib.bitctrl.modell.tmanzeigenglobal.objekte.Dirigent;
import de.bsvrz.sys.funclib.bitctrl.modell.tmanzeigenglobal.objekte.WvzInhalt;
import de.bsvrz.sys.funclib.bitctrl.modell.tmanzeigenglobal.objekte.WzgInhaltGrafik;
import de.bsvrz.sys.funclib.bitctrl.modell.tmanzeigenglobalzwischenschicht.konfigurationsdaten.KdAnzeigeAktor;
import de.bsvrz.sys.funclib.bitctrl.modell.tmanzeigenglobalzwischenschicht.konfigurationsdaten.KdWvzInhaltUmsetzung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmanzeigenglobalzwischenschicht.objekte.AnzeigeAktor;
import de.bsvrz.sys.funclib.bitctrl.modell.tmanzeigenglobalzwischenschicht.objekte.AnzeigeQuerschnittAktor;
import de.bsvrz.sys.funclib.bitctrl.modell.tmanzeigenglobalzwischenschicht.objekte.WvzInhaltUmsetzung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmanzeigenglobalzwischenschichttls.objekte.AnzeigeAktorTls;
import de.bsvrz.sys.funclib.bitctrl.modell.tmanzeigenglobalzwischenschichttls.objekte.AnzeigeQuerschnittAktorTls;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkextlsglobal.objekte.De;
import de.bsvrz.sys.funclib.bitctrl.modell.util.DavCacheLader;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Cache f&uuml;r Anzeigen.
 *
 * @author BitCtrl Systems GmbH, Gieseler
 * 
 *         XXX Prüfen, warum der Serializable implemntiert hatte
 */
public class AnzeigenCache extends AbstractCache {

	/**
	 * Mappt Anzeigetypen auf Anzeigeinhalte
	 */
	private Map<AnzeigeTyp, Collection<WvzInhalt>> anzeigeTyp2AnzeigeInhalt;

	/**
	 * Mappt Anzeigen auf Anzeigeaktoren.
	 */
	private Map<Anzeige, AnzeigeAktor> anzeige2anzeigeaktor;

	/**
	 * Mappt Anzeigeaktoren auf Anzeigen.
	 */
	private Map<AnzeigeAktor, Anzeige> anzeigeaktor2anzeige;

	/**
	 * Mappt AnzeigeaktorenTls auf DEs.
	 */
	private Map<AnzeigeAktorTls, List<De>> anzeigeaktor2de;

	/**
	 * Mappt DEs auf AnzeigeaktorenTls.
	 */
	private Map<De, AnzeigeAktorTls> de2anzeigeAktor;

	/**
	 * Mappt Anzeigequerschnitte auf Anzeigequerschnittaktoren.
	 */
	private Map<AnzeigeQuerschnitt, AnzeigeQuerschnittAktor> anzeigequerschnitt2anzeigequerschnittaktor;

	/**
	 * Mappt Anzeigequerschnittaktoren auf Anzeigequerschnitte.
	 */
	private Map<AnzeigeQuerschnittAktor, AnzeigeQuerschnitt> anzeigequerschnittaktor2anzeigequerschnitt;

	/**
	 * Mappt AnzeigequerschnittaktorenTls auf DEs.
	 */
	private Map<AnzeigeQuerschnittAktorTls, De> anzeigequerschnittaktor2de;

	/**
	 * Mappt DEs auf AnzeigequerschnittaktorenTls.
	 */
	private Map<De, AnzeigeQuerschnittAktorTls> de2anzeigequerschnittAktor;

	/**
	 * Mappt Anzeigen auf Anzeigequerschnitte.
	 */
	private Map<Anzeige, AnzeigeQuerschnitt> anzeige2aq;

	/**
	 * Mappt {@link WvzInhalt}e auf {@link WvzInhaltUmsetzung}en.
	 */
	private Map<WvzInhalt, WvzInhaltUmsetzung> wvzInhalt2WvzInhaltUmsetzung;

	/**
	 * Mappt {@link WzgInhaltGrafik}en auf {@link WvzInhalt}e.
	 */
	private Map<WzgInhaltGrafik, WvzInhalt> grafik2wvzInhaltUmsetzung;

	/**
	 * Flag, ob initialisiert.
	 */
	private boolean initialisiert = false;

	/**
	 * DAV-Verbindung.
	 */
	private AbstractDavVerbindungsCache davVerbindungsCache;

	/**
	 * Liste der AQs.
	 */
	private List<AnzeigeQuerschnitt> aqList;

	/**
	 * Liste der Dirigenten.
	 */
	private List<Dirigent> dirigentList;

	/**
	 * Mappt {@link Anzeige}n auf {@link AnzeigeTyp}en.
	 */
	private HashMap<Anzeige, AnzeigeTyp> anzeige2anzeigeTyp;

	/**
	 * Mappt {@link AnzeigeTyp}en auf {@link Anzeige}n.
	 */
	private HashMap<AnzeigeTyp, List<Anzeige>> anzeigeTyp2Anzeigen;

	/**
	 * Mappt {@link WvzInhalt}e auf {@link AnzeigeTyp}en.
	 */
	private HashMap<WvzInhalt, List<AnzeigeTyp>> wvzInhalt2AnzeigeTyp;

	/**
	 * Privater Konstruktor.
	 */
	private AnzeigenCache() {
		//
	}

	/**
	 * Konstruktor.
	 *
	 * @param davVerbCache Verbindung zum Datenverteiler.
	 */
	public AnzeigenCache(final AbstractDavVerbindungsCache davVerbCache) {
		super("Anzeigen-Cache", davVerbCache);
	}

	/**
	 * Vor-initialisierung.
	 *
	 * @param davVerbCache Verbindung zum Datenverteiler.
	 */
	public void preInit(final AbstractDavVerbindungsCache davVerbCache) {
		davVerbindungsCache = davVerbCache;
	}

	@SuppressWarnings("unchecked")
	@Override
	protected boolean doInit(final IMonitor monitor) {
		if (!initialisiert) {
			anzeige2anzeigeaktor = new HashMap<>();
			anzeigeaktor2de = new HashMap<>();
			de2anzeigeAktor = new HashMap<>();
			anzeige2aq = new HashMap<>();
			anzeigeaktor2anzeige = new HashMap<>();
			anzeige2anzeigeTyp = new HashMap<>();
			anzeigeTyp2Anzeigen = new HashMap<>();
			anzeigeTyp2AnzeigeInhalt = new HashMap<>();
			dirigentList = new ArrayList<>();

			try {
				DavCacheLader.lade(davVerbindungsCache.getClientDavInterface(), AnzeigeQuerschnitt.PID,
						KdAnzeigeQuerschnitt.PID);
				DavCacheLader.lade(davVerbindungsCache.getClientDavInterface(), Anzeige.PID, KdAnzeige.PID);
				DavCacheLader.lade(davVerbindungsCache.getClientDavInterface(), AnzeigeAktor.PID, KdAnzeigeAktor.PID);

				dirigentList.addAll(
						(List<Dirigent>) (davVerbindungsCache.getObjektFactory().bestimmeModellobjekte(Dirigent.PID)));

				aqList = new ArrayList<>();
				aqList.addAll((List<AnzeigeQuerschnitt>) (davVerbindungsCache.getObjektFactory()
						.bestimmeModellobjekte(AnzeigeQuerschnitt.PID)));

				final SubMonitor subMonitor = SubMonitor.convert(monitor);
				subMonitor.beginTask("Initialisiere Anzeigen-Cache ...", aqList.size());

				/**
				 * Ordne allen AQs die Anzeigen zu
				 */
				for (final SystemObjekt aqs : aqList) {
					subMonitor.worked(1);
					final AnzeigeQuerschnitt aq = (AnzeigeQuerschnitt) aqs;
					for (final Anzeige anz : aq.getAnzeigen()) {
						anzeige2aq.put(anz, aq);
					}
				}

				final List<? extends SystemObjekt> aktorenList = davVerbindungsCache.getObjektFactory()
						.bestimmeModellobjekte(AnzeigeAktor.PID);

				for (final SystemObjekt aktors : aktorenList) {
					final AnzeigeAktor aktor = (AnzeigeAktor) aktors;
					final Anzeige anzeigeReferenz = aktor.getKdAnzeigeAktor().getDatum().getAnzeigeReferenz();
					if (anzeigeReferenz != null) {
						anzeige2anzeigeaktor.put(anzeigeReferenz, aktor);
						anzeigeaktor2anzeige.put(aktor, anzeigeReferenz);
					}

					if (aktors instanceof AnzeigeAktorTls) {
						final AnzeigeAktorTls aktorTls = (AnzeigeAktorTls) aktors;
						final Feld<KonfigurationsObjekt> zugehoerigeDEs = aktorTls.getKdAnzeigeAktorTls().getDatum()
								.getZugehoerigeDEs();
						final List<De> deListe = new ArrayList<>();
						for (final KonfigurationsObjekt deo : zugehoerigeDEs) {
							deListe.add((De) deo);
							de2anzeigeAktor.put((De) deo, aktorTls);
						}
						anzeigeaktor2de.put(aktorTls, deListe);
					}
				}

				initialisiereAnzeige2AnzeigeTyp(monitor);
				initialisiereWvzInhalt2AnzeigeTyp(monitor);
				initialisiereWvzInhalt2WvzInhaltUmsetzung(monitor);
				initialisiereAnzeigeTyp2AnzeigeInhalt(monitor);
				initialisiereWzgInhaltGrafik2WvzInhaltUmsetzung(monitor);
				initialisiereAnzeigequerschnitt2AnzeigequerschnittAktor(monitor);

				initialisiert = true;
				return true;
			} catch (final Exception ex) {
				ex.printStackTrace();
				Debug.getLogger().error("Initialisierung fehlgeschlagen.", ex);
			}
		} else {
			return true;
		}
		return false;
	}

	/**
	 * Auflösung der Rückreferenzen von {@link AnzeigeTyp}en nach {@link Anzeige}.
	 *
	 * @param monitor der Monitor zur Visualisierung des Fortschritts
	 */
	private void initialisiereAnzeigeTyp2AnzeigeInhalt(final IMonitor monitor) {
		final SubMonitor subMonitor = SubMonitor.convert(monitor);
		subMonitor.beginTask("Auflösen der Rückreferenzen zwischen Anzeigen und Anzeigetypen ...",
				anzeige2anzeigeTyp.size());
		for (final AnzeigeTyp anzeigeTyp : anzeige2anzeigeTyp.values()) {
			anzeigeTyp2AnzeigeInhalt.put(anzeigeTyp, anzeigeTyp.getWvzInhalt());
			subMonitor.worked(1);
		}
		subMonitor.done();
	}

	/**
	 * Auflösen der Rückreferenz zwischen {@link AnzeigeTyp} und {@link WvzInhalt}.
	 *
	 * @param monitor der Monitor zur Visualisierung des Fortschritts
	 */
	private void initialisiereWvzInhalt2AnzeigeTyp(final IMonitor monitor) {
		wvzInhalt2AnzeigeTyp = new HashMap<>();
		@SuppressWarnings("unchecked")
		final List<AnzeigeTyp> anzeigenTypList = (List<AnzeigeTyp>) davVerbindungsCache.getObjektFactory()
				.bestimmeModellobjekte(AnzeigeTyp.PID);
		final SubMonitor subMonitor = SubMonitor.convert(monitor);
		subMonitor.beginTask("Initialisiere Beziehungen zwischen WvzInhalten und Anzeigetypen ...",
				anzeigenTypList.size());
		for (final AnzeigeTyp typ : anzeigenTypList) {
			for (final WvzInhalt inhalt : typ.getWvzInhalt()) {
				if (!wvzInhalt2AnzeigeTyp.containsKey(inhalt)) {
					wvzInhalt2AnzeigeTyp.put(inhalt, new ArrayList<AnzeigeTyp>());
				}
				wvzInhalt2AnzeigeTyp.get(inhalt).add(typ);
			}
			monitor.worked(1);
		}
		monitor.done();

	}

	/**
	 * Auflösen der Beziehungen zwischen {@link Anzeige}n und {@link AnzeigeTyp} n.
	 *
	 * @param monitor der Monitor zur Visualisierung des Fortschritts
	 */
	private void initialisiereAnzeige2AnzeigeTyp(final IMonitor monitor) {

		@SuppressWarnings("unchecked")
		final List<Anzeige> anzeigenList = (List<Anzeige>) davVerbindungsCache.getObjektFactory()
				.bestimmeModellobjekte(Anzeige.PID);
		final SubMonitor subMonitor = SubMonitor.convert(monitor);
		subMonitor.beginTask("Initialisiere Beziehungen zwischen Anzeigen und Anzeigetypen ...", anzeigenList.size());
		for (final Anzeige anzeige : anzeigenList) {
			final AnzeigeTyp typ = anzeige.getKdAnzeige().getDatum().getAnzeigeTyp();
			anzeige2anzeigeTyp.put(anzeige, typ);
			if (!anzeigeTyp2Anzeigen.containsKey(typ)) {
				anzeigeTyp2Anzeigen.put(typ, new ArrayList<Anzeige>());
			}
			anzeigeTyp2Anzeigen.get(typ).add(anzeige);
			subMonitor.worked(1);
		}
		subMonitor.done();
	}

	/**
	 * Auflösen der Beziehungen zwischen {@link WvzInhalt} und
	 * {@link WvzInhaltUmsetzung}.
	 *
	 * @param monitor der Monitor zur Visualisierung des Fortschritts
	 */
	private void initialisiereWvzInhalt2WvzInhaltUmsetzung(final IMonitor monitor) {
		wvzInhalt2WvzInhaltUmsetzung = new HashMap<>();
		@SuppressWarnings("unchecked")
		final List<WvzInhaltUmsetzung> umsetzungsListe = (List<WvzInhaltUmsetzung>) davVerbindungsCache
				.getObjektFactory().bestimmeModellobjekte(WvzInhaltUmsetzung.PID);
		final SubMonitor subMonitor = SubMonitor.convert(monitor);
		subMonitor.beginTask("Initialisiere Beziehungen zwischen Wvz-Inhalt-Umsetzungen und Wvz-Inhalten ...",
				umsetzungsListe.size());
		davVerbindungsCache.getObjektFactory().getDaten(umsetzungsListe, KdWvzInhaltUmsetzung.class);
		for (final WvzInhaltUmsetzung umsetzung : umsetzungsListe) {
			final KdWvzInhaltUmsetzung.Daten datum = umsetzung.getKdWvzInhaltUmsetzung().getDatum();
			final WvzInhalt inhalt = datum.getWvzInhaltReferenz();
			wvzInhalt2WvzInhaltUmsetzung.put(inhalt, umsetzung);
			subMonitor.worked(1);
		}
		subMonitor.done();
	}

	/**
	 * Auflösen der Beziehungen zwischen {@link WzgInhaltGrafik} und
	 * {@link WvzInhalt}.
	 *
	 * @param monitor Progressmonitor.
	 */
	private void initialisiereWzgInhaltGrafik2WvzInhaltUmsetzung(final IMonitor monitor) {
		grafik2wvzInhaltUmsetzung = new LinkedHashMap<>();
		@SuppressWarnings("unchecked")
		final List<WvzInhalt> inhaltsListe = (List<WvzInhalt>) davVerbindungsCache.getObjektFactory()
				.bestimmeModellobjekte(WvzInhalt.PID);
		final SubMonitor subMonitor = SubMonitor.convert(monitor);
		subMonitor.beginTask("Initialisiere Beziehungen zwischen Grafiken und WVZ-Inhalten ...", inhaltsListe.size());
		for (final WvzInhalt inhalt : inhaltsListe) {
			for (final WzgInhaltGrafik grafik : inhalt.getKdWvzInhalt().getDatum().getGrafikDarstellungen()) {
				grafik2wvzInhaltUmsetzung.put(grafik, inhalt);
			}
			subMonitor.worked(1);
		}
		subMonitor.done();
	}

	/**
	 * Auflösen der Beziehungen zwischen {@link AnzeigeQuerschnittAktor} und
	 * {@link AnzeigeQuerschnitt}.
	 *
	 * @param monitor Progressmonitor.
	 */
	private void initialisiereAnzeigequerschnitt2AnzeigequerschnittAktor(final IMonitor monitor) {
		anzeigequerschnitt2anzeigequerschnittaktor = new LinkedHashMap<>();
		anzeigequerschnittaktor2anzeigequerschnitt = new HashMap<>();
		anzeigequerschnittaktor2de = new HashMap<>();
		de2anzeigequerschnittAktor = new HashMap<>();

		@SuppressWarnings("unchecked")
		final List<AnzeigeQuerschnittAktor> aktoren = (List<AnzeigeQuerschnittAktor>) davVerbindungsCache
				.getObjektFactory().bestimmeModellobjekte(AnzeigeQuerschnittAktor.PID);
		final SubMonitor subMonitor = SubMonitor.convert(monitor);
		subMonitor.beginTask("Initialisiere Beziehungen zwischen Anzeigequerschnitt und Anzeigequerschnittaktoren ...",
				aktoren.size());

		for (final AnzeigeQuerschnittAktor aktor : aktoren) {
			final AnzeigeQuerschnitt anzeigeQuerschnittReferenz = aktor.getKdAnzeigeQuerschnittAktor().getDatum()
					.getAnzeigeQuerschnittReferenz();

			if (anzeigeQuerschnittReferenz != null) {
				anzeigequerschnitt2anzeigequerschnittaktor.put(anzeigeQuerschnittReferenz, aktor);
				anzeigequerschnittaktor2anzeigequerschnitt.put(aktor, anzeigeQuerschnittReferenz);

				if (aktor instanceof AnzeigeQuerschnittAktorTls) {
					final AnzeigeQuerschnittAktorTls aktorTls = (AnzeigeQuerschnittAktorTls) aktor;
					final KonfigurationsObjekt zugehoerigeDE = aktorTls.getKdAnzeigeQuerschnittAktorTls().getDatum()
							.getZugehoerigeDE();
					de2anzeigequerschnittAktor.put((De) zugehoerigeDE, aktorTls);
					anzeigequerschnittaktor2de.put(aktorTls, (De) zugehoerigeDE);
				}

				subMonitor.worked(1);
			}
		}

		subMonitor.done();
	}

	/**
	 * Liefert die {@link WvzInhaltUmsetzung}, in der der übergebene
	 * {@link WvzInhalt} referenziert wird.
	 *
	 * @param wvzInhalt der Inhalt
	 * @return die {@link WvzInhaltUmsetzung}.
	 */
	public final WvzInhaltUmsetzung getWvzInhaltUmsetzung(final WvzInhalt wvzInhalt) {
		if (!isInitialisiert()) {
			throw new IllegalStateException("Noch nicht initialisiert");
		}
		return wvzInhalt2WvzInhaltUmsetzung.get(wvzInhalt);
	}

	/**
	 * Pr&uuml;ft, ob ein Anzeigequerschnitt einem Dirigenten zugeordnet ist.
	 *
	 * @param anzeigeQuerschnitt AnzeigeQuerschnitt
	 * @return true, wenn der Anzeigequerschnitt einem Dirigenten zugeordnet ist,
	 *         sonst false
	 */
	public boolean gehoertZuDirigent(final AnzeigeQuerschnitt anzeigeQuerschnitt) {
		for (final Dirigent d : getDirigenten()) {
			if (d.getAnzeigeQuerschnitte().contains(anzeigeQuerschnitt)) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Gibt den Dirigenten zu einen Anzeigequerschnitt zur&uuml;ck.
	 *
	 * @param anzeigeQuerschnitt AnzeigeQuerschnitt
	 * @return Dirigent oder <code>null</code>, wenn der Anzeigequerschnitt keinem
	 *         Dirigenten zugeordnet ist
	 */
	public Dirigent getDirigent(final AnzeigeQuerschnitt anzeigeQuerschnitt) {
		for (final Dirigent d : getDirigenten()) {
			if (d.getAnzeigeQuerschnitte().contains(anzeigeQuerschnitt)) {
				return d;
			}
		}

		return null;
	}

	// /**
	// * Erfragt die Menge aller Anzeigen an einem Standort.
	// *
	// * @param standort
	// * ein Standort.
	// * @return die Menge aller Anzeigen an einem Standort.
	// */
	// public final Set<Anzeige> getAnzeigen(final Standort standort) {
	// return standort2Anzeigen.get(standort);
	// }
	//
	// /**
	// * Gibt eine Liste aller Standorte zur&uuml;ck.
	// *
	// * @return {@link Standort} oder {@code null} wenn nicht definiert
	// */
	// public final List<Standort> getStandorte() {
	// if (!isInitialisiert()) {
	// throw new IllegalStateException("Noch nicht initialisiert");
	// }
	//
	// return new ArrayList<Standort>(standortList);
	// }

	/**
	 * Gibt eine Liste aller Dirigenten zur&uuml;ck.
	 *
	 * @return die Menge aller {@link Dirigent}en.
	 */
	public final List<Dirigent> getDirigenten() {
		if (!isInitialisiert()) {
			throw new IllegalStateException("Noch nicht initialisiert");
		}

		return new ArrayList<>(dirigentList);
	}

	/**
	 * Gibt eine Liste aller Anzeigequerschnitte zur&uuml;ck.
	 *
	 * @return {@link AnzeigeQuerschnitt} oder {@code null} wenn nicht definiert
	 */
	public final List<AnzeigeQuerschnitt> getAnzeigeQuerschnitte() {
		if (!isInitialisiert()) {
			throw new IllegalStateException("Noch nicht initialisiert");
		}
		return Collections.unmodifiableList(aqList);
	}

	/**
	 * Gibt eine Liste aller Anzeigen zur&uuml;ck.
	 *
	 * @return {@link AnzeigeAktor} oder {@code null} wenn nicht definiert
	 */
	public final ArrayList<Anzeige> getAnzeigen() {
		if (!isInitialisiert()) {
			throw new IllegalStateException("Noch nicht initialisiert");
		}
		return new ArrayList<>(anzeige2anzeigeaktor.keySet());
	}

	/**
	 * Gibt eine Liste aller Anzeigeaktoren zur&uuml;ck.
	 *
	 * @return {@link AnzeigeAktor} oder {@code null} wenn nicht definiert
	 */
	public final Collection<AnzeigeAktor> getAnzeigeAktoren() {
		if (!isInitialisiert()) {
			throw new IllegalStateException("Noch nicht initialisiert");
		}
		return anzeige2anzeigeaktor.values();
	}

	/**
	 * Gibt die Anzeigen zu einem Anzeigequerschnitt zur&uuml;ck.
	 *
	 * @param anzeigequerschnitt Anzeigequerschnitt
	 *
	 * @return die Menge der {@link Anzeige}n.
	 */
	public final Collection<Anzeige> getAnzeigen(final AnzeigeQuerschnitt anzeigequerschnitt) {
		if (!isInitialisiert()) {
			throw new IllegalStateException("Noch nicht initialisiert");
		}

		return anzeigequerschnitt.getAnzeigen();
	}

	/**
	 * Gibt den Anzeigeaktor zu einer Anzeige zur&uuml;ck.
	 *
	 * @param anzeige Anzeige
	 *
	 * @return {@link AnzeigeAktor} oder {@code null} wenn nicht definiert
	 */
	public final AnzeigeAktor getAnzeigeAktor(final Anzeige anzeige) {
		if (!isInitialisiert()) {
			throw new IllegalStateException("Noch nicht initialisiert");
		}
		return anzeige2anzeigeaktor.get(anzeige);
	}

	/**
	 * Gibt den Anzeigequerschnittaktor zu einem Anzeigequerschnitt zur&uuml;ck.
	 *
	 * @param anzeigequerschnitt Anzeigequerschnitt
	 *
	 * @return {@link AnzeigeAktor} oder {@code null} wenn nicht definiert
	 */
	public final AnzeigeQuerschnittAktor getAnzeigeQuerschnittAktor(final AnzeigeQuerschnitt anzeigequerschnitt) {
		if (!isInitialisiert()) {
			throw new IllegalStateException("Noch nicht initialisiert");
		}
		return anzeigequerschnitt2anzeigequerschnittaktor.get(anzeigequerschnitt);
	}

	/**
	 * Gibt den AnzeigeAktor zu einem DE zur&uuml;ck.
	 *
	 * @param de DE
	 *
	 * @return {@link AnzeigeAktor} oder {@code null} wenn nicht definiert
	 */
	public final AnzeigeAktor getAnzeigeAktor(final De de) {
		if (!isInitialisiert()) {
			throw new IllegalStateException("Noch nicht initialisiert");
		}

		return de2anzeigeAktor.get(de);
	}

	/**
	 * Gibt die Anzeige zu einem DE zur&uuml;ck.
	 *
	 * @param de DE
	 *
	 * @return {@link Anzeige} oder {@code null} wenn nicht definiert
	 */
	public final Anzeige getAnzeige(final De de) {
		if (!isInitialisiert()) {
			throw new IllegalStateException("Noch nicht initialisiert");
		}

		final AnzeigeAktor aktor = getAnzeigeAktor(de);
		if (aktor != null) {
			return anzeigeaktor2anzeige.get(aktor);
		}

		return null;
	}

	/**
	 * Gibt die Anzeigequerschnitt zu einem DE zur&uuml;ck.
	 *
	 * @param de DE
	 *
	 * @return {@link AnzeigeQuerschnitt} oder {@code null} wenn nicht definiert
	 */
	public final AnzeigeQuerschnitt getAnzeigeQuerschnitt(final De de) {
		if (!isInitialisiert()) {
			throw new IllegalStateException("Noch nicht initialisiert");
		}

		final Anzeige anzeige = getAnzeige(de);
		if (anzeige != null) {
			return anzeige2aq.get(anzeige);
		}

		return null;
	}

	/**
	 * Gibt die Anzeigequerschnitt zu einer Anzeige zur&uuml;ck.
	 *
	 * @param anzeige Anzeige
	 *
	 * @return {@link AnzeigeQuerschnitt} oder {@code null} wenn nicht definiert
	 */
	public final AnzeigeQuerschnitt getAnzeigeQuerschnitt(final Anzeige anzeige) {
		if (!isInitialisiert()) {
			throw new IllegalStateException("Noch nicht initialisiert");
		}

		return anzeige2aq.get(anzeige);
	}

	/**
	 * Liefert den {@link AnzeigeTyp} der {@link Anzeige}.
	 *
	 * @param anzeige {@link Anzeige}.
	 * @return {@link AnzeigeTyp} oder <code>null</code> (sollte allerdings nie
	 *         passieren).
	 */
	public final AnzeigeTyp getAnzeigeTyp(final Anzeige anzeige) {
		if (!isInitialisiert()) {
			throw new IllegalStateException("Noch nicht initialisiert");
		}
		return anzeige2anzeigeTyp.get(anzeige);
	}

	/**
	 * Liefert alle {@link Anzeige}n des {@link AnzeigeTyp}s.
	 *
	 * @param typ der Typ for den Anzeigen ermittelt werden sollen
	 * @return die Menge von {@link Anzeige}n des {@link AnzeigeTyp}s.
	 */
	public final Collection<Anzeige> getAnzeigen(final AnzeigeTyp typ) {
		if (!isInitialisiert()) {
			throw new IllegalStateException("Noch nicht initialisiert");
		}
		return Collections.unmodifiableCollection(anzeigeTyp2Anzeigen.get(typ));
	}

	/**
	 * Liefert die {@link AnzeigeTyp}en, zu denen der {@link WvzInhalt} gehört.
	 *
	 * @param inhalt der Inhalt für den Anzeigetypen ermittelt werden sollen
	 * @return die {@link AnzeigeTyp}en.
	 */
	public final Collection<AnzeigeTyp> getAnzeigeTyp(final WvzInhalt inhalt) {
		if (!isInitialisiert()) {
			throw new IllegalStateException("Noch nicht initialisiert");
		}
		return Collections.unmodifiableCollection(wvzInhalt2AnzeigeTyp.get(inhalt));
	}

	/**
	 * Liefert die {@link De}, zum {@link AnzeigeQuerschnittAktor}.
	 *
	 * @param aktor AnzeigeQuerschnittAktor
	 * @return der zugeordnete De.
	 */
	public final De getDe(final AnzeigeQuerschnittAktor aktor) {
		if (!isInitialisiert()) {
			throw new IllegalStateException("Noch nicht initialisiert");
		}

		return anzeigequerschnittaktor2de.get(aktor);
	}

	/**
	 * Liefert die {@link De}, zum {@link AnzeigeAktor}.
	 *
	 * @param aktor AnzeigeAktor
	 * @return die zugeordneten De.
	 */
	public final List<De> getDe(final AnzeigeAktor aktor) {
		if (!isInitialisiert()) {
			throw new IllegalStateException("Noch nicht initialisiert");
		}

		return anzeigeaktor2de.get(aktor);
	}

	/**
	 * Erfragt die WVZ-Inhalte eines Anzeigetyps.
	 *
	 * @param anzeigeTyp ein Anzeigetyp.
	 * @return die WVZ-Inhalte eines Anzeigetyps.
	 */
	public Collection<WvzInhalt> getWvzInhalte(final AnzeigeTyp anzeigeTyp) {
		if (!isInitialisiert()) {
			throw new IllegalStateException("Noch nicht initialisiert");
		}
		return anzeigeTyp2AnzeigeInhalt.get(anzeigeTyp);
	}

	/**
	 * Erfragt denjenigen (einzigen!) WvzInhalt, der auf die übergebene Grafik
	 * verweist.
	 *
	 * @param grafik die Grafik
	 * @return der WvzInhalt oder null, wenn es keinen gibt.
	 */
	public WvzInhalt getContainingWvzInhalt(final WzgInhaltGrafik grafik) {
		if (!isInitialisiert()) {
			throw new IllegalStateException("Noch nicht initialisiert");
		}
		return grafik2wvzInhaltUmsetzung.get(grafik);
	}
}
