/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2011 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.sys.funclib.bitctrl.modell.util.cache;

import java.io.Serializable;
import java.util.ArrayList;

import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.konfigurationsdaten.KdFlaechenKoordinaten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.konfigurationsdaten.KdKomplexKoordinaten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.objekte.Flaeche;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.objekte.FlaecheXY;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.objekte.Komplex;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.objekte.KomplexXY;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.objekte.Linie;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.objekte.Punkt;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Cache fuer ein DAV-Objekt vom Typ <code>typ.komplexXY</code>.
 *
 * @author BitCtrl Systems GmbH, thierfelder
 * @version $Id$
 */
public class KomplexXYCacheObject implements Serializable {

	private static final long serialVersionUID = -2480300327145657916L;

	/**
	 * Die PID des gespeicherten Komplexes.
	 */
	private final String pid;

	/**
	 * Liste der Punktlisten des Komplexes.
	 */
	private final ArrayList<ArrayList<CachePunkt>> polygonList = new ArrayList<>();

	@SuppressWarnings("unused")
	private KomplexXYCacheObject() {
		pid = null;
	}

	/**
	 * Konstruktor.
	 *
	 * @param komplexXY ein DAV-Objekt vom Typ <code>typ.komplexXY</code>.
	 */
	public KomplexXYCacheObject(final KomplexXY komplexXY) {
		pid = komplexXY.getPid();
		final KdKomplexKoordinaten.Daten koordinaten = komplexXY.getKdKomplexKoordinaten().getDatum();
		for (final Komplex komplex : koordinaten.getKomplexReferenz()) {
			Debug.getLogger().error("TODO#1:Komplex: " + komplex.getPid());
			// TODO
		}
		for (final Linie linie : koordinaten.getLinienReferenz()) {
			Debug.getLogger().error("TODO#2:Linie: " + linie.getPid());
			// TODO
		}
		for (final Punkt punkt : koordinaten.getPunktReferenz()) {
			Debug.getLogger().error("TODO#3:Punkt: " + punkt.getPid());
			// TODO
		}
		for (final Flaeche flaeche : koordinaten.getFlaechenReferenz()) {
			if (flaeche instanceof FlaecheXY) {
				final FlaecheXY flaecheXY = (FlaecheXY) flaeche;
				final KdFlaechenKoordinaten.Daten flaechenKoordinaten = flaecheXY.getKdFlaechenKoordinaten().getDatum();

				if (flaechenKoordinaten.getX().size() != flaechenKoordinaten.getY().size()) {
					Debug.getLogger().warning("Ungleiche Anzahl X- und Y-Koordinaten in " + komplexXY + " (X: "
							+ flaechenKoordinaten.getX().size() + ", Y: " + flaechenKoordinaten.getY().size() + ")");
				}

				final ArrayList<CachePunkt> punktListe = new ArrayList<>();
				for (int i = 0; i < Math.min(flaechenKoordinaten.getX().size(),
						flaechenKoordinaten.getY().size()); i++) {
					punktListe.add(new CachePunkt(flaechenKoordinaten.getX().get(i).doubleValue(),
							flaechenKoordinaten.getY().get(i).doubleValue()));
				}
				polygonList.add(punktListe);
			} else {
				Debug.getLogger().error("TODO#4.1:Flaeche vom Typ " + flaeche.getTyp());
				// TODO
			}
		}
	}

	/**
	 * Erfragt die PID des gespeicherten Komplexes.
	 *
	 * @return die PID des gespeicherten Komplexes.
	 */
	public final String getPid() {
		return pid;
	}

	/**
	 * Erfragt die Anzahl der Polygone in diesem Komplex.
	 *
	 * @return die Anzahl der Polygone in diesem Komplex.
	 */
	public final int size() {
		return polygonList.size();
	}

	/**
	 * Erfragt die Anzahl der Punkte in einem Polygon.
	 *
	 * @param polygonIndex der Index des Polygons.
	 * @return die Anzahl der Punkte in einem Polygon.
	 */
	public final int size(final int polygonIndex) {
		return polygonList.get(polygonIndex).size();
	}

	/**
	 * @param polygonIndex der Index innerhalb des Polygonliste
	 * @param punktIndex   der Index innerhalb des ausgewählten Polygons
	 * @return der ermittelte Wert
	 */
	public final double getX(final int polygonIndex, final int punktIndex) {
		return polygonList.get(polygonIndex).get(punktIndex).x;
	}

	/**
	 * @param polygonIndex der Index innerhalb des Polygonliste
	 * @param punktIndex   der Index innerhalb des ausgewählten Polygons
	 * @return der ermittelte Wert
	 */
	public final double getY(final int polygonIndex, final int punktIndex) {
		return polygonList.get(polygonIndex).get(punktIndex).y;
	}

	private static class CachePunkt implements Serializable {

		/**
		 *
		 */
		private static final long serialVersionUID = 2670756026449248870L;

		final Double x;

		final Double y;

		CachePunkt(final double x, final double y) {
			this.x = x;
			this.y = y;
		}

	}

}
