/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2011 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.bitctrl.modell.util.cache;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.bitctrl.util.monitor.IMonitor;
import com.bitctrl.util.monitor.SubMonitor;

import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.konfigurationsdaten.KdPunktLiegtAufLinienObjekt;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.konfigurationsdaten.KdPunktLiegtAufLinienObjekt.Daten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.FahrStreifen;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.MessQuerschnitt;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.MessQuerschnittAllgemein;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.StrassenSegment;
import de.bsvrz.sys.funclib.debug.Debug;

public class MqCache extends AbstractCache {
	// MessQuerschnitt <-> Straßensegment
	private Map<MessQuerschnittAllgemein, StrassenSegment> mq2ss;
	private Map<StrassenSegment, Set<MessQuerschnittAllgemein>> ss2mq;

	// MessQuerschnitt <-> FahrStreifen
	private Map<FahrStreifen, MessQuerschnitt> fs2mq;

	private AbstractDavVerbindungsCache verbindungsCache;
	private NetzCache netzCache;

	public MqCache(final NetzCache netzCache, final AbstractDavVerbindungsCache verbindungsCache) {
		super(netzCache, verbindungsCache);
	}

	protected void preInit(final NetzCache netzCache, final AbstractDavVerbindungsCache verbindungsCache) {
		this.netzCache = netzCache;
		this.verbindungsCache = verbindungsCache;
	}

	@Override
	protected synchronized boolean doInit(final IMonitor monitor) {
		final long start = System.currentTimeMillis();
		setName(getNetzCache().getName());
		final SubMonitor subMonitor = SubMonitor.convert(monitor, "Initialisiere Messquerschnitt-Cache ...", 1000);
		// MessQuerschnitt <-> Straßensegment
		mq2ss = new HashMap<>();
		ss2mq = new HashMap<>();

		fs2mq = new HashMap<>();
		final Set<StrassenSegment> strassenSegmente = getNetzCache().getStrassenSegmente();
		subMonitor.newChild(100);
		final ObjektFactory objektFactory = verbindungsCache.getObjektFactory();
		final List<MessQuerschnittAllgemein> modellObjekte = (List<MessQuerschnittAllgemein>) objektFactory
				.bestimmeModellobjekte(MessQuerschnittAllgemein.PID);

		final SubMonitor subMonitor2 = subMonitor.newChild(900);
		subMonitor2.beginTask("", modellObjekte.size());

		final List<Daten> daten = objektFactory.getDaten(modellObjekte, KdPunktLiegtAufLinienObjekt.class);

		for (int i = 0; i < modellObjekte.size(); i++) {
			final MessQuerschnittAllgemein mqa = modellObjekte.get(i);
			final KdPunktLiegtAufLinienObjekt.Daten datum = daten.get(i);
			subMonitor2.setTaskName(mqa.getName());
			try {
				if (null == datum) {
					continue;
				}

				final StrassenSegment mqSegment = (StrassenSegment) datum.getLinienReferenz();
				if ((null == mqSegment) || !strassenSegmente.contains(mqSegment)) {
					continue;
				}
				mq2ss.put(mqa, mqSegment);
				Set<MessQuerschnittAllgemein> mqs = ss2mq.get(mqSegment);
				if (null == mqs) {
					mqs = new HashSet<>();
					ss2mq.put(mqSegment, mqs);
				}
				mqs.add(mqa);

				if (mqa instanceof MessQuerschnitt) {
					final MessQuerschnitt mq = (MessQuerschnitt) mqa;
					final Collection<FahrStreifen> fahrStreifenCol = mq.getFahrStreifen();
					for (final FahrStreifen fahrStreifen : fahrStreifenCol) {
						if (fs2mq.containsKey(fahrStreifen)) {
							Debug.getLogger().warning("Fehler in der Cache initialisierung!\n"
									+ "Ein FahrStreifen gehöhrt nur zu einem MessQuerschnitt!", fahrStreifen);
						}
						fs2mq.put(fahrStreifen, mq);
					}
				}
			} finally {
				subMonitor2.worked(1);
			}
		}
		// TODO Messquerschnitte der Straßenteilsegmente sortieren
		Debug.getLogger().info("Benötigte Zeit zum Initialisieren des Messquerschnitt-Caches: "
				+ (System.currentTimeMillis() - start) + " ms");
		return true;
	}

	protected final NetzCache getNetzCache() {
		return netzCache;
	}

	public StrassenSegment getStrassenSegment(final MessQuerschnittAllgemein mq) {
		ensureInit();
		return mq2ss.get(mq);
	}

	public Set<MessQuerschnittAllgemein> getMessquerschnitte(final StrassenSegment ss) {
		ensureInit();
		final Set<MessQuerschnittAllgemein> mqs = ss2mq.get(ss);
		return null == mqs ? Collections.emptySet() : Collections.unmodifiableSet(mqs);
	}

	public Set<MessQuerschnittAllgemein> getMessquerschnitte() {
		ensureInit();
		return Collections.unmodifiableSet(mq2ss.keySet());
	}

	public MessQuerschnitt getMessquerschnitt(final FahrStreifen fs) {
		ensureInit();
		return fs2mq.get(fs);
	}

	public Collection<FahrStreifen> getFahrstreifen(final MessQuerschnitt mq) {
		return mq.getFahrStreifen();
	}
}
