/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2011 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.bitctrl.modell.util.cache;

import java.beans.PropertyChangeListener;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.bitctrl.util.monitor.IMonitor;
import com.bitctrl.util.monitor.SubMonitor;

import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.SystemObjekt;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.attribute.AttRdsTMCRichtung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcglobal.attribute.AttTmcRichtung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.konfigurationsdaten.KdAeusseresStrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.konfigurationsdaten.KdGesamtStrasse;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.konfigurationsdaten.KdStrasse.Daten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.konfigurationsdaten.KdStrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.AeusseresStrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.GesamtStrasse;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.InneresStrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.Netz;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.NetzBestandTeil;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.Strasse;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.StrassenKnoten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.StrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.VerkehrsModellNetz;
import de.bsvrz.sys.funclib.debug.Debug;

public class NetzCache extends AbstractCache {

	private static final Debug log = Debug.getLogger();

	private VerkehrsModellNetz verkehrsModellNetz;

	private ModellObjektCache moc;

	private AbstractDavVerbindungsCache verbindungsCache;

	private List<AeusseresStrassenSegment> aess;

	// StraßenKnoten <-> InneresStraßenSegment
	private Map<StrassenKnoten, Set<InneresStrassenSegment>> sk2iss;
	private Map<InneresStrassenSegment, StrassenKnoten> iss2sk;

	// Straße <-> StraßenSegment
	private Map<Strasse, List<StrassenSegment>> s2ss;
	private Map<StrassenSegment, Strasse> ss2s;

	private Set<GesamtStrasse> gesamtstrassen;

	/**
	 * Map, welche zu einem Straßennamen die dazugehörige Straße speichert.
	 */
	private Map<String, Strasse> strassenNamenMap;

	/**
	 * Map, welche die Straßennummer zu einer Straße speichert.
	 */
	private HashMap<Strasse, String> strassenNummernMap;

	/**
	 * Map, welche den Namen des Von-Knotens zu einer Straße in positiver Richtung
	 * speichert. Bei negativer Richtung ist dann der Von-Knoten der Nach-Knoten.
	 */
	private Map<Strasse, String> vonStrassenKnotenMap;

	/**
	 * Map, welche den Namen des Nach-Knotens zu einer Straße in positiver Richtung
	 * speichert. Bei negativer Richtung ist dann der Nach-Knoten der Von-Knoten.
	 */
	private Map<Strasse, String> nachStrassenKnotenMap;

	/**
	 * Initialisert den Netzcache, in dem alle notwendige Daten aus der
	 * Konfiguration gelesen und zwischengespeichert werden. Insbesondere werden
	 * notwendige Rückwärtreferenzen aufgelöst, die in der Konfiguration so nicht
	 * gesichert werden.
	 *
	 * Nach Aufruf der Methode ist der Cache <strong>nicht</strong> initialisert.
	 * Stattdessen wird ein Job angestoßen der diese Aufgabe übernimmt.
	 *
	 * @param moc              der Cache, von welchem dieser Cache abhängig ist
	 * @param verbindungsCache der verwendete Verbindungscache
	 *
	 * @see AbstractCache#isInitialisiert()
	 * @see #addPropertyChangeListener(PropertyChangeListener)
	 * @see #PROP_STATE
	 */
	public NetzCache(final ModellObjektCache moc, final AbstractDavVerbindungsCache verbindungsCache) {
		super(moc, verbindungsCache);
	}

	protected void preInit(final ModellObjektCache moc, final AbstractDavVerbindungsCache verbindungsCache) {
		this.moc = moc;
		this.verbindungsCache = verbindungsCache;
	}

	@Override
	protected boolean doInit(final IMonitor monitor) {
		final long start = System.currentTimeMillis();
		verkehrsModellNetz = (VerkehrsModellNetz) moc.getModellObjekt();

		setName(verkehrsModellNetz.getName());
		final SubMonitor subMonitor = SubMonitor.convert(monitor);
		subMonitor.beginTask("Initialisiere " + getName(), 2000);
		sk2iss = new HashMap<>();
		iss2sk = new HashMap<>();

		s2ss = new HashMap<>();
		ss2s = new HashMap<>();
		gesamtstrassen = new HashSet<>();

		// ss2sts = new HashMap<StrassenSegment, List<StrassenTeilSegment>>();
		// sts2ss = new HashMap<StrassenTeilSegment, List<StrassenSegment>>();

		aess = new ArrayList<>();

		strassenNamenMap = new HashMap<>();
		strassenNummernMap = new HashMap<>();
		vonStrassenKnotenMap = new HashMap<>();
		nachStrassenKnotenMap = new HashMap<>();

		// (1) Ermittel Netzbestandteile
		subMonitor.subTask("Lade Netzbestandteile");
		ermittelAeussereStrassenSegmente(getNetz(), new ArrayList<Netz>());
		final Collection<SystemObject> aessSO = resolveSystemObjects(aess);
		subMonitor.worked(50);
		if (subMonitor.isCanceled()) {
			subMonitor.done();
			return false;
		}

		// (2) Lade Konfigurationsdaten "AeussereStrassenSegment"
		subMonitor.subTask("Lade StrassenSegment-Daten der Netzbestandteile");
		final DataModel dataModel = verbindungsCache.getObjektFactory().getDav().getDataModel();
		if (isBenutzeBlockAbruf()) {
			dataModel.getConfigurationData(aessSO, dataModel.getAttributeGroup(KdStrassenSegment.PID));
		} else {
			for (final SystemObject so : aessSO) {
				so.getConfigurationData(dataModel.getAttributeGroup(KdStrassenSegment.PID));
			}
		}
		subMonitor.worked(250);
		if (subMonitor.isCanceled()) {
			subMonitor.done();
			return false;
		}

		// (3) Lade Daten
		subMonitor.subTask("Lade AeussereStrassenSegment-Daten der Netzbestandteile");
		if (isBenutzeBlockAbruf()) {
			dataModel.getConfigurationData(aessSO, dataModel.getAttributeGroup(KdAeusseresStrassenSegment.PID));
		} else {
			for (final SystemObject so : aessSO) {
				so.getConfigurationData(dataModel.getAttributeGroup(KdAeusseresStrassenSegment.PID));
			}

		}

		subMonitor.worked(250);
		if (subMonitor.isCanceled()) {
			subMonitor.done();
			return false;
		}

		// (4) Analysieren
		final SubMonitor subMonitor2 = subMonitor.newChild(1450);
		final int anzahlSegmente = aess.size();
		subMonitor2.beginTask("Analysiere Netzbestandteile", anzahlSegmente);
		if (anzahlSegmente > 0) {
			int i = 1;
			for (final AeusseresStrassenSegment segment : aess) {

				subMonitor2.subTask(new StringBuffer("Analysiere Netzbestandteil ").append(i).append(" von ")
						.append(anzahlSegmente).append(": ").append(segment.getName()).toString());

				cacheAeusseresStrassenSegment(segment);
				subMonitor2.worked(1);
				if (subMonitor2.isCanceled()) {
					subMonitor2.done();
					return false;
				}
				i++;
			}
		}

		final ObjektFactory objektFactory = verbindungsCache.getObjektFactory();
		@SuppressWarnings("unchecked")
		final List<GesamtStrasse> alleGesamtStrassen = (List<GesamtStrasse>) objektFactory
				.bestimmeModellobjekte(GesamtStrasse.PID);
		objektFactory.getDaten(alleGesamtStrassen, KdGesamtStrasse.class);

		final Set<Strasse> alleStrassenDesNetzes = s2ss.keySet();

		for (final GesamtStrasse g : alleGesamtStrassen) {
			final KdGesamtStrasse.Daten datum = g.getKdGesamtStrasse().getDatum();
			if (datum != null) {
				boolean enthalten = datum.getStrasseFR1().stream().anyMatch(s -> alleStrassenDesNetzes.contains(s));
				if (!enthalten) {
					enthalten = datum.getStrasseFR2().stream().anyMatch(s -> alleStrassenDesNetzes.contains(s));
				}

				if (enthalten) {
					gesamtstrassen.add(g);
				}
			}
		}

		// TODO Straßensegmente der Straßen sortieren
		// TODO Straßenteilsegmente der Straßensegmente sortieren

		Debug.getLogger().info("Benötigte Zeit zum Initialisieren des BitCtrl Netz-Caches: "
				+ (System.currentTimeMillis() - start) + " ms");

		return true;
	}

	/**
	 * Ermittelt rekursiv sämtliche äußeren Straßensegmente eines Netzes sowie deren
	 * Teilnetze.
	 *
	 * @param netz          das Netz
	 * @param besuchteNetze die bereits besuchte Netze, verhindert Endlosschleifen
	 */
	private void ermittelAeussereStrassenSegmente(final Netz netz, final ArrayList<Netz> besuchteNetze) {

		if (besuchteNetze.contains(netz)) {
			return;
		}
		besuchteNetze.add(netz);

		for (final NetzBestandTeil netzBestandTeil : netz.getNetzBestandTeile()) {
			if ((netzBestandTeil instanceof AeusseresStrassenSegment) && !aess.contains(netzBestandTeil)) {
				aess.add((AeusseresStrassenSegment) netzBestandTeil);
			} else if (netzBestandTeil instanceof Netz) {
				ermittelAeussereStrassenSegmente((Netz) netzBestandTeil, besuchteNetze);
			}
		}
	}

	/**
	 * Cacht die Daten des Start- und Endknoten eines äußeren Straßensegments.
	 *
	 * @param segment das äußere Straßensegment
	 */
	protected void cacheAeusseresStrassenSegment(final AeusseresStrassenSegment segment) {
		// cacheStrassenTeilSegmente(aess);
		cacheStrasse(segment);

		StrassenKnoten sk = segment.getKdAeusseresStrassenSegment().getDatum().getVonKnoten();
		if (sk != null) {
			cacheInneresStrasseSegment(sk);
		}

		sk = segment.getKdAeusseresStrassenSegment().getDatum().getNachKnoten();
		if (sk != null) {
			cacheInneresStrasseSegment(sk);
		}
	}

	/**
	 * Erzeugt die Referenzen StraßenKnoten &lt;-&gt; InneresStraßenSegment. Cacht
	 * auch die Straße und die Straßenteilsegmente der inneren Straßensegmente.
	 *
	 * @param sk der zu cachende Straßenknoten
	 */
	protected void cacheInneresStrasseSegment(final StrassenKnoten sk) {
		if (sk2iss.containsKey(sk)) {
			// Straßenknoten wurde bereits gecacht
			return;
		}

		final Set<InneresStrassenSegment> mengeISS = new HashSet<>();
		for (final InneresStrassenSegment iss : sk.getInnereStrassenSegmente()) {
			iss2sk.put(iss, sk);
			mengeISS.add(iss);

			cacheStrasse(iss);
			// cacheStrassenTeilSegmente(iss);
		}
		sk2iss.put(sk, mengeISS);
	}

	/**
	 * Erzeugt die Referenzen StraßenSegment &lt;-&gt; Straßen.
	 *
	 * @param ss das Straßensegment
	 */
	protected void cacheStrasse(final StrassenSegment ss) {
		if (ss2s.containsKey(ss)) {
			// Straßensegment wurde bereits gecacht
			return;
		}

		Strasse s = null;
		s = ss.getKdStrassenSegment().getDatum().getGehoertZuStrasse();
		if (s == null) {
			// XXX: Diese Konvention (".00000.") gilt nur für BaWü!!
			if (!ss.getPid().contains(".00000.")) {
				log.warning("Strassensegment ohne Straßenzuordnung: " + ss);
			}
		} else {
			ss2s.put(ss, s);

			if (!s2ss.containsKey(s)) {
				s2ss.put(s, new ArrayList<StrassenSegment>());
			}
			s2ss.get(s).add(ss);

			// Parse den Straßennamen und fülle die entsprechenden Maps
			if ((s.getKdStrasse() == null) || (s.getKdStrasse().getDatum() == null)) {
				return;
			}
			final Daten datum = s.getKdStrasse().getDatum();
			final String name = s.getName();
			if (strassenNamenMap.get(name) == null) {
				strassenNamenMap.put(name, s);
			}
			final String nummer = datum.getTyp().toString().substring(0, 1) + datum.getNummer().getValue();
			if (strassenNummernMap.get(s) == null) {
				strassenNummernMap.put(s, nummer);
			}
			if (name.contains(" von ") && name.contains(" nach ")) {
				if (vonStrassenKnotenMap.get(s) == null) {
					final String vonStrassenKnoten = name.substring(name.indexOf(" von ") + 5, name.indexOf(" nach "));
					vonStrassenKnotenMap.put(s, vonStrassenKnoten);
				}
				if (nachStrassenKnotenMap.get(s) == null) {
					final String nachStrassenKnoten = name.substring(name.indexOf(" nach ") + 6);
					nachStrassenKnotenMap.put(s, nachStrassenKnoten);
				}
			}

		}
	}

	/**
	 * Gibt das Netz zurück, welches diesem Netzcache zugrunde liegt.
	 *
	 * @return das Netz auf dem die Karte basiert.
	 */
	public VerkehrsModellNetz getNetz() {
		return verkehrsModellNetz;
	}

	/**
	 * Gibt die {@link GesamtStrasse}s im Netz zurück.
	 *
	 * @return die vorhandenen {@link GesamtStrasse}n.
	 */
	public Set<GesamtStrasse> getGesamtStrassen() {
		ensureInit();
		return Collections.unmodifiableSet(gesamtstrassen);
	}

	/**
	 * Liefert die {@link Strasse}n einer {@link GesamtStrasse} in der gefragten
	 * Richtung. Es werden nur {@link Strasse}n ermittelt, die auch im Netz sind.
	 * 
	 */
	public Set<Strasse> getStrassen(final GesamtStrasse gesamtStraße, final AttTmcRichtung richtung) {

		final Set<Strasse> strassen = getStrassen();
		final KdGesamtStrasse.Daten datum = gesamtStraße.getKdGesamtStrasse().getDatum();
		if (datum != null) {
			if (AttTmcRichtung.ZUSTAND_1_POSITIV.equals(richtung)) {
				return datum.getStrasseFR1().stream().filter(s -> strassen.contains(s)).collect(Collectors.toSet());
			} else if (AttTmcRichtung.ZUSTAND_1N_NEGATIV.equals(richtung)) {
				return datum.getStrasseFR2().stream().filter(s -> strassen.contains(s)).collect(Collectors.toSet());
			}
		}
		return Collections.unmodifiableSet(Collections.emptySet());
	}

	/**
	 * Gibt die Straßen im Netz zurück.
	 *
	 * @return die vorhandenen Strassen.
	 */
	public Set<Strasse> getStrassen() {
		ensureInit();
		return Collections.unmodifiableSet(s2ss.keySet());
	}

	/**
	 * Liefert die äußeren Straßensegmente des Netzes zurück.
	 *
	 * @return die äußeren Straßensegmente des Netzes
	 */
	public List<AeusseresStrassenSegment> getAeussereStrassenSegmente() {
		ensureInit();
		return Collections.unmodifiableList(aess);
	}

	/**
	 * Gibt alle Straßensegmente im Netz zurück.
	 *
	 * @return die Straßensegmente im Netz.
	 */
	public Set<StrassenSegment> getStrassenSegmente() {
		ensureInit();
		return Collections.unmodifiableSet(ss2s.keySet());
	}

	/**
	 * Gibt die Menge aller Strassenknoten zurück, die sich im Netz befinden.
	 *
	 * @return die vorhandenen Strassenknoten.
	 */
	public Set<StrassenKnoten> getStrassenKnoten() {
		ensureInit();
		return Collections.unmodifiableSet(sk2iss.keySet());
	}

	/**
	 * Gibt zu einem inneren Straßensegment den dazugehörigen Straßenknoten zurück.
	 *
	 * @param iss ein inneres Straßensegment.
	 * @return der dazugehörige Straßenknoten oder <code>null</code>, wenn das
	 *         innere Straßensegment unbekannt ist.
	 */
	public StrassenKnoten getStrassenKnoten(final InneresStrassenSegment iss) {
		ensureInit();
		return iss2sk.get(iss);
	}

	/**
	 * Gibt die inneren Straßensegmente eines Straßenknotens zurück.
	 *
	 * @param sk ein Straßenknoten.
	 * @return die inneren Straßensegmente oder eine leere Liste, wenn der
	 *         Straßenknoten keine Straßensegmente hat oder unbekannt ist.
	 */
	public Set<InneresStrassenSegment> getInneresStrassenSegmente(final StrassenKnoten sk) {
		ensureInit();
		return sk2iss.get(sk);
	}

	/**
	 * Gibt die Straßensegmente einer Straße zurück.
	 *
	 * @param strasse eine Straße.
	 * @return die Straßensegmente der Strasse.
	 */
	public List<StrassenSegment> getStrassenSegmente(final Strasse strasse) {
		ensureInit();
		final List<StrassenSegment> list = s2ss.get(strasse);
		if (list != null) {
			return Collections.unmodifiableList(list);
		}
		return Collections.unmodifiableList(Collections.emptyList());
	}

	/**
	 * Gibt die Straße zu einem Straßensegment zurück.
	 *
	 * @param ss ein Straßensegment.
	 * @return die Straße auf dem das Straßensegment liegt.
	 */
	public Strasse getStrasse(final StrassenSegment ss) {
		ensureInit();
		return ss2s.get(ss);
	}

	/**
	 * Liefert die zu einem gegebenen Straßennamen gehörige Straße zurück.
	 *
	 * @param strassenName der Straßenname, darf nicht <code>null</code> sein
	 *
	 * @return die zu einem gegebenen Straßennamen gehörige Straße, kann
	 *         <code>null</code> sein
	 */
	public Strasse getStrasse(final String strassenName) {
		ensureInit();
		assert strassenName != null;
		return strassenNamenMap.get(strassenName);
	}

	/**
	 * Liefert die Straßennummer einer gegebenen Straße zurück.
	 *
	 * @param strasse die Straße, darf nicht <code>null</code> sein
	 *
	 * @return die Straßennummer einer gegebenen Straße
	 */
	public String getStrassenNummer(final Strasse strasse) {
		ensureInit();
		assert strasse != null;
		return strassenNummernMap.get(strasse);
	}

	/**
	 * Liefert den Straßennamen zu einer gegebenen Straßennummer und Richtung
	 * zurück. Sollte kein Straßenname für die gegebene Straßennummer existieren,
	 * wird die Straßennummer zurückgegeben.
	 *
	 * @param strasse  die Straße, darf nicht <code>null</code> sein
	 * @param richtung die Richtung, darf nicht <code>null</code> sein
	 *
	 * @return Straßennamen zu einer gegebenen Straßennummer und Richtungr, niemals
	 *         <code>null</code>
	 */
	public String getStrassenName(final Strasse strasse, final AttRdsTMCRichtung richtung) {
		ensureInit();
		assert strasse != null;
		assert richtung != null;
		if (richtung.getValue() == AttRdsTMCRichtung.ZUSTAND_0_POSITIV.getValue()) {
			return strasse.getName();
		}
		final String nummer = getStrassenNummer(strasse);
		final String von = getVonStrassenKnoten(strasse, richtung);
		final String nach = getNachStrassenKnoten(strasse, richtung);
		if ((nummer != null) && !"".equals(nummer) && (von != null) && !"".equals(von) && (nach != null)
				&& !"".equals(nach)) {
			return new StringBuffer(nummer).append(" von ").append(von).append(" nach ").append(nach).toString();
		}
		return strasse.getName();
	}

	/**
	 * Liefert den Straßennamen zu einer gegebenen Straße und Richtung zurück.
	 *
	 * @param strasse  die Straße, darf nicht <code>null</code> sein
	 * @param richtung die Richtung, darf auch <code>null</code> sein
	 *
	 * @return Straßennamen zu einer gegebenen Straßennummer und Richtung, niemals
	 *         <code>null</code>
	 */
	public String getStrassenName(final Strasse strasse, final AttTmcRichtung richtung) {
		ensureInit();
		assert strasse != null;
		if (AttTmcRichtung.ZUSTAND_1N_NEGATIV.equals(richtung)) {
			return getStrassenName(strasse, AttRdsTMCRichtung.ZUSTAND_1_NEGATIV);
		}
		if (AttTmcRichtung.ZUSTAND_1_POSITIV.equals(richtung)) {
			return getStrassenName(strasse, AttRdsTMCRichtung.ZUSTAND_0_POSITIV);
		}
		return strasse.getName();
	}

	/**
	 * Liefert den Namen des Von-Straßenknotens zu einer gegebenen Straße zurück.
	 *
	 * @param strasse  die Straße, darf nicht <code>null</code> sein
	 * @param richtung die Straßenrichtung, darf nicht <code>null</code> sein
	 *
	 * @return Von-Straßenknoten zu einer gegebenen Straßennummer, kann
	 *         <code>null</code> sein
	 */
	public String getVonStrassenKnoten(final Strasse strasse, final AttRdsTMCRichtung richtung) {
		ensureInit();
		assert strasse != null;
		assert richtung != null;
		final String vonStrassenKnoten = (richtung.getValue() == AttRdsTMCRichtung.ZUSTAND_0_POSITIV.getValue())
				? vonStrassenKnotenMap.get(strasse)
				: nachStrassenKnotenMap.get(strasse);
		if ((vonStrassenKnoten != null) && !"".equals(vonStrassenKnoten)) {
			return vonStrassenKnoten;
		}
		return null;
	}

	/**
	 * Liefert den Namen des Von-Straßenknotens zu einer gegebenen Straße zurück.
	 *
	 * @param strasse  die Straße, darf nicht <code>null</code> sein
	 * @param richtung die Straßenrichtung, darf nicht <code>null</code> sein
	 *
	 * @return Von-Straßenknoten zu einer gegebenen Straßennummer, kann
	 *         <code>null</code> sein
	 */
	public String getVonStrassenKnoten(final Strasse strasse, final AttTmcRichtung richtung) {
		return getVonStrassenKnoten(strasse,
				richtung == AttTmcRichtung.ZUSTAND_1_POSITIV ? AttRdsTMCRichtung.ZUSTAND_0_POSITIV
						: AttRdsTMCRichtung.ZUSTAND_1_NEGATIV);
	}

	/**
	 * Liefert den Namen des Nach-Straßenknotens zu einer gegebenen Straße zurück.
	 *
	 * @param strasse  die Straße, darf nicht <code>null</code> sein
	 * @param richtung die Straßenrichtung, darf nicht <code>null</code> sein
	 *
	 * @return Nach-Straßenknoten zu einer gegebenen Straßennummer, kann
	 *         <code>null</code> sein
	 */
	public String getNachStrassenKnoten(final Strasse strasse, final AttRdsTMCRichtung richtung) {
		ensureInit();
		assert strasse != null;
		assert richtung != null;
		final String nachStrassenKnoten = (richtung.getValue() == AttRdsTMCRichtung.ZUSTAND_0_POSITIV.getValue())
				? nachStrassenKnotenMap.get(strasse)
				: vonStrassenKnotenMap.get(strasse);
		if ((nachStrassenKnoten != null) && !"".equals(nachStrassenKnoten)) {
			return nachStrassenKnoten;
		}
		return null;
	}

	/**
	 * Liefert den Namen des Nach-Straßenknotens zu einer gegebenen Straße zurück.
	 *
	 * @param strasse  die Straße, darf nicht <code>null</code> sein
	 * @param richtung die Straßenrichtung, darf nicht <code>null</code> sein
	 *
	 * @return Nach-Straßenknoten zu einer gegebenen Straßennummer, kann
	 *         <code>null</code> sein
	 */
	public String getNachStrassenKnoten(final Strasse strasse, final AttTmcRichtung richtung) {
		return getNachStrassenKnoten(strasse,
				richtung == AttTmcRichtung.ZUSTAND_1_POSITIV ? AttRdsTMCRichtung.ZUSTAND_0_POSITIV
						: AttRdsTMCRichtung.ZUSTAND_1_NEGATIV);
	}

	private Collection<SystemObject> resolveSystemObjects(final Collection<? extends SystemObjekt> objekte) {
		final ArrayList<SystemObject> objects = new ArrayList<>(objekte.size());
		for (final SystemObjekt objekt : objekte) {
			objects.add(objekt.getSystemObject());
		}
		return objects;
	}
}
