/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2011 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.bitctrl.modell.util.cache;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.bitctrl.util.monitor.IMonitor;
import com.bitctrl.util.monitor.SubMonitor;

import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.SystemObjekt;
import de.bsvrz.sys.funclib.bitctrl.modell.att.Zahl;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.attribute.AttRdsEreignisKategorie;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.konfigurationsdaten.KdRdsLandesKennung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.konfigurationsdaten.KdRdsQuantitaet;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.objekte.RdsLandesKennung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.objekte.RdsQuantitaet;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmsttmccodes.konfigurationsdaten.KdRdsTmcCode;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmsttmccodes.objekte.RdsTmcCode;
import de.bsvrz.sys.funclib.bitctrl.modell.util.rdstmc.RdsLandesKennungWrapper;
import de.bsvrz.sys.funclib.bitctrl.modell.util.rdstmc.RdsQuantitaetWrapper;
import de.bsvrz.sys.funclib.bitctrl.modell.util.rdstmc.RdsTmcCodeWrapper;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * <p>
 * Diese Hilfsklasse cacht sämtliche für dieses Plug-in benötigten Informationen
 * bzgl. TMC-Location-Codes.
 * </p>
 * 
 * @author BitCtrl Systems GmbH, Andreas Meissner
 * @version $Id: TmcLocationCodeCache.java 22422 2010-03-26 14:55:24Z meissner $
 */
public class RdsCache extends AbstractCache {

	/**
	 * Map, welche zu einem Ereignis-Code den dazugehörigen RdsTmcCodeWrapper
	 * speichert.
	 */
	private Map<Zahl<Integer>, RdsTmcCodeWrapper> rdsTmcCodeMap;

	/**
	 * Map, welche zu einer Ereigniskategorie die dazugehörige Liste von
	 * Ereignis-Codes speichert.
	 */
	private Map<AttRdsEreignisKategorie, List<Zahl<Integer>>> kategorieEreignisCodeMap;

	/**
	 * Map, welche zu einer RdsQuantitaet den dazugehörigen RdsQuantitaetWrapper
	 * speichert.
	 */
	private Map<RdsQuantitaet, RdsQuantitaetWrapper> quantitaetWrapperMap;

	/**
	 * Map, welche zu einer Kennung die dazugehörige RdsQuantitaet speichert.
	 */
	private Map<String, RdsQuantitaet> quantitaetenKennungMap;

	/**
	 * Map, welche zu einer RdsLandesKennung den dazugehörigen
	 * RdsLandesKennungWrapper speichert.
	 */
	private Map<RdsLandesKennung, RdsLandesKennungWrapper> landesKennungWrapperMap;

	/**
	 * Alphabetisch sortierte Liste aller RdsQuantitaetWrapper.
	 */
	private List<RdsQuantitaetWrapper> rdsQuantitaetWrapper;

	/**
	 * Alphabetisch sortierte Liste aller RdsLandesKennungWrapper.
	 */
	private List<RdsLandesKennungWrapper> rdsLandesKennungWrapper;

	/**
	 * Map, welche zu einem Landescode die dazugehörige RdsLandesKennung speichert.
	 */
	private Map<String, RdsLandesKennung> code2LandesKennungMap;

	/**
	 * Der Verbindungscache.
	 */
	private AbstractDavVerbindungsCache verbindungsCache;

	/**
	 * Der Konstruktor.
	 * 
	 * @param verbindungsCache der Verbindungscache
	 */
	public RdsCache(final AbstractDavVerbindungsCache verbindungsCache) {
		super(verbindungsCache);
	}

	/**
	 * Vorinitialisierung.
	 * 
	 * @param vCache Verbindungscache
	 */
	protected void preInit(final AbstractDavVerbindungsCache vCache) {
		verbindungsCache = vCache;
	}

	@Override
	protected boolean doInit(final IMonitor monitor) {
		if (!verbindungsCache.getObjektFactory().isVerbunden()) {
			return false;
		}
		final SubMonitor subMonitor = SubMonitor.convert(monitor);
		subMonitor.beginTask("Initialisiere RDS-Cache", 30000);

		boolean ergebnis = false;
		if (ermittelRdsLandesKennungen(subMonitor.newChild(10000)) && ermittelRdsTmcCodes(subMonitor.newChild(10000))) {
			ergebnis = ermittelRdsEreignisQuantitaeten(subMonitor.newChild(10000));
		}

		if ((landesKennungWrapperMap == null) || landesKennungWrapperMap.isEmpty() || (rdsTmcCodeMap == null)
				|| rdsTmcCodeMap.isEmpty() || (quantitaetenKennungMap == null) || quantitaetenKennungMap.isEmpty()
				|| (quantitaetWrapperMap == null) || quantitaetWrapperMap.isEmpty()
				|| (kategorieEreignisCodeMap == null) || kategorieEreignisCodeMap.isEmpty()) {

			Debug.getLogger().error("Der RDS-Cache konnte nicht vollständig geladen werden. "
					+ "Bitte überprüfen Sie, ob die aktuellen Versionen der dazugehörigen Konfigurationsbereiche 'kb.tmKExLMStGlobal' (Ver.13) und 'kb.tmKExLMStTMCCodes' (Ver.7) vorliegen.");
		}
		return ergebnis;
	}

	/**
	 * Ermittelt die {@link RdsTmcCode}.
	 * 
	 * @param monitor der Monitor
	 * 
	 * @return <code>true</code>, wenn die Method erfolgreich ausgeführt wurde,
	 *         ansonsten <code>false</code>
	 */
	@SuppressWarnings("unchecked")
	private boolean ermittelRdsTmcCodes(final IMonitor monitor) {

		rdsTmcCodeMap = new HashMap<>();
		kategorieEreignisCodeMap = new HashMap<>();

		final int totalWork = 120000;
		final int work = totalWork / 3;
		final String taskName = "Lade RDS-TMC-Code-Mapping";
		final SubMonitor subMonitor = SubMonitor.convert(monitor, totalWork);
		subMonitor.setTaskName(taskName);
		subMonitor.beginTask(taskName, totalWork);

		final ObjektFactory objektFactory = verbindungsCache.getObjektFactory();
		final DataModel dataModel = objektFactory.getDav().getDataModel();

		// (1) Lade
		subMonitor.subTask("Lade Mappings");
		final Collection<? extends RdsTmcCode> rdsTmcCodes = (Collection<? extends RdsTmcCode>) objektFactory
				.bestimmeModellobjekte(RdsTmcCode.PID);
		final Collection<SystemObject> rdsTmcCodesSO = resolveSystemObjects(rdsTmcCodes);
		subMonitor.worked(work);
		if (subMonitor.isCanceled()) {
			subMonitor.done();
			return false;
		}

		// (2) Lade Daten
		subMonitor.subTask("Lade Daten der Mappings");
		if (isBenutzeBlockAbruf()) {
			dataModel.getConfigurationData(rdsTmcCodesSO, dataModel.getAttributeGroup(KdRdsTmcCode.PID));
		} else {
			for (final SystemObject so : rdsTmcCodesSO) {
				so.getConfigurationData(dataModel.getAttributeGroup(KdRdsTmcCode.PID));
			}
		}
		subMonitor.worked(work);
		if (subMonitor.isCanceled()) {
			subMonitor.done();
			return false;
		}

		// (3) Analysiere
		final int anzahlRdsTmcCodes = rdsTmcCodes.size();
		if (anzahlRdsTmcCodes == 0) {
			return true;
		}

		final String taskName2 = "Analysiere Mapping";
		subMonitor.subTask(taskName2);
		final SubMonitor subMonitor2 = subMonitor.newChild(work);
		subMonitor2.beginTask(taskName2, rdsTmcCodes.size());

		int i = 0;
		for (final RdsTmcCode rdsTmcCode : rdsTmcCodes) {

			try {

				subMonitor2.subTask(new StringBuffer("Analysiere RDS-TMC-Code-Mapping ").append(i).append(" von ")
						.append(anzahlRdsTmcCodes).toString());

				final RdsTmcCodeWrapper wrapper = new RdsTmcCodeWrapper(rdsTmcCode);
				rdsTmcCodeMap.put(wrapper.getCode(), wrapper);
				List<Zahl<Integer>> ereignisCodes = kategorieEreignisCodeMap.get(wrapper.getKategorie());
				if (ereignisCodes == null) {
					ereignisCodes = new ArrayList<>();
					kategorieEreignisCodeMap.put(wrapper.getKategorie(), ereignisCodes);
				}
				ereignisCodes.add(wrapper.getCode());

			} catch (final RuntimeException e) {
				// tut nix
			} finally {
				subMonitor2.worked(1);
				if (subMonitor2.isCanceled()) {
					subMonitor2.done();
					return false;
				}
				i++;
			}
		}
		return true;
	}

	/**
	 * Ermittelt die {@link RdsQuantitaet}.
	 * 
	 * @param monitor der Monitor
	 * 
	 * @return <code>true</code>, wenn die Method erfolgreich ausgeführt wurde,
	 *         ansonsten <code>false</code>
	 */
	@SuppressWarnings("unchecked")
	private boolean ermittelRdsEreignisQuantitaeten(final IMonitor monitor) {

		quantitaetWrapperMap = new HashMap<>();
		quantitaetenKennungMap = new HashMap<>();

		final int totalWork = 120000;
		final int work = totalWork / 3;
		final String taskName = "Lade RDS-Ereignis-Quantitäten";
		final SubMonitor subMonitor = SubMonitor.convert(monitor, totalWork);
		subMonitor.setTaskName(taskName);
		subMonitor.beginTask(taskName, totalWork);

		final ObjektFactory objektFactory = verbindungsCache.getObjektFactory();
		final DataModel dataModel = objektFactory.getDav().getDataModel();

		// (1) Lade
		subMonitor.subTask("Lade Quantitäten");
		final Collection<? extends RdsQuantitaet> rdsQuantitaeten = (Collection<? extends RdsQuantitaet>) objektFactory
				.bestimmeModellobjekte(RdsQuantitaet.PID);
		final Collection<SystemObject> rdsQuantitaetenSO = resolveSystemObjects(rdsQuantitaeten);
		subMonitor.worked(work);
		if (subMonitor.isCanceled()) {
			subMonitor.done();
			return false;
		}

		// (2) Lade Daten
		subMonitor.subTask("Lade Daten der Mappings");
		if (isBenutzeBlockAbruf()) {
			dataModel.getConfigurationData(rdsQuantitaetenSO, dataModel.getAttributeGroup(KdRdsQuantitaet.PID));
		} else {
			for (final SystemObject so : rdsQuantitaetenSO) {
				so.getConfigurationData(dataModel.getAttributeGroup(KdRdsQuantitaet.PID));
			}
		}
		subMonitor.worked(work);
		if (subMonitor.isCanceled()) {
			subMonitor.done();
			return false;
		}

		// (3) Analysiere
		final int anzahlRdsQuantitaeten = rdsQuantitaeten.size();
		if (anzahlRdsQuantitaeten == 0) {
			return true;
		}

		final String taskName2 = "Analysiere RDS-Ereignis-Quantitäten";
		subMonitor.subTask(taskName2);
		final SubMonitor subMonitor2 = subMonitor.newChild(work);
		subMonitor2.beginTask(taskName2, rdsQuantitaeten.size());

		int i = 0;
		for (final RdsQuantitaet rdsQuantitaet : rdsQuantitaeten) {

			try {

				subMonitor2.subTask(new StringBuffer("Analysiere RDS-Ereignis-Quantität ").append(i).append(" von ")
						.append(anzahlRdsQuantitaeten).toString());

				final RdsQuantitaetWrapper wrapper = new RdsQuantitaetWrapper(rdsQuantitaet);
				quantitaetWrapperMap.put(rdsQuantitaet, wrapper);
				quantitaetenKennungMap.put(wrapper.getKennung(), rdsQuantitaet);

			} catch (final RuntimeException e) {
				// tut nix
			} finally {
				subMonitor2.worked(1);
				if (subMonitor2.isCanceled()) {
					subMonitor2.done();
					return false;
				}
				i++;
			}
		}
		return true;
	}

	/**
	 * Ermittelt die {@link RdsLandesKennung}.
	 * 
	 * @param monitor der Monitor
	 * 
	 * @return <code>true</code>, wenn die Method erfolgreich ausgeführt wurde,
	 *         ansonsten <code>false</code>
	 */
	@SuppressWarnings("unchecked")
	private boolean ermittelRdsLandesKennungen(final IMonitor monitor) {

		code2LandesKennungMap = new HashMap<>();
		landesKennungWrapperMap = new HashMap<>();

		final int totalWork = 120000;
		final int work = totalWork / 3;
		final String taskName = "Lade RDS-Landeskennungen";
		final SubMonitor subMonitor = SubMonitor.convert(monitor, totalWork);
		subMonitor.setTaskName(taskName);
		subMonitor.beginTask(taskName, totalWork);

		final ObjektFactory objektFactory = verbindungsCache.getObjektFactory();
		final DataModel dataModel = objektFactory.getDav().getDataModel();

		// (1) Lade
		subMonitor.subTask("Lade RDS-Landeskennungen");
		final Collection<? extends RdsLandesKennung> rdsLandesKennungen = (Collection<? extends RdsLandesKennung>) objektFactory
				.bestimmeModellobjekte(RdsLandesKennung.PID);
		final Collection<SystemObject> rdsLandesKennungenSO = resolveSystemObjects(rdsLandesKennungen);
		subMonitor.worked(work);
		if (subMonitor.isCanceled()) {
			subMonitor.done();
			return false;
		}

		// (2) Lade Daten
		subMonitor.subTask("Lade Daten der RDS-Landeskennungen");
		if (isBenutzeBlockAbruf()) {
			dataModel.getConfigurationData(rdsLandesKennungenSO, dataModel.getAttributeGroup(KdRdsLandesKennung.PID));
		} else {
			for (final SystemObject so : rdsLandesKennungenSO) {
				so.getConfigurationData(dataModel.getAttributeGroup(KdRdsLandesKennung.PID));
			}
		}
		subMonitor.worked(work);
		if (subMonitor.isCanceled()) {
			subMonitor.done();
			return false;
		}

		// (3) Analysiere
		final int anzahlRdsLandesKennungen = rdsLandesKennungen.size();
		if (anzahlRdsLandesKennungen == 0) {
			return true;
		}

		final String taskName2 = "Analysiere Landeskennungen";
		subMonitor.subTask(taskName2);
		final SubMonitor subMonitor2 = subMonitor.newChild(work);
		subMonitor2.beginTask(taskName2, rdsLandesKennungen.size());

		int i = 0;
		for (final RdsLandesKennung rdsLandesKennung : rdsLandesKennungen) {

			try {

				subMonitor2.subTask(new StringBuffer("Analysiere RDS-Landeskennung ").append(i).append(" von ")
						.append(anzahlRdsLandesKennungen).toString());

				final RdsLandesKennungWrapper wrapper = new RdsLandesKennungWrapper(rdsLandesKennung);
				landesKennungWrapperMap.put(rdsLandesKennung, wrapper);
				code2LandesKennungMap.put(wrapper.getLandesCode(), rdsLandesKennung);

			} catch (final RuntimeException e) {
				// tut nix
			} finally {
				subMonitor2.worked(1);
				if (subMonitor2.isCanceled()) {
					subMonitor2.done();
					return false;
				}
				i++;
			}
		}
		return true;
	}

	/**
	 * Liefert die zu einem gegeben Ereignis-Code dazugehörigen RdsTmcCodeWrapper
	 * zurück.
	 * 
	 * @param ereignisCode der Ereignis-Code, darf nicht <code>null</code> sein
	 * 
	 * @return die zu einem gegeben Ereignis-Code dazugehörigen RdsTmcCodeWrapper,
	 *         kann <code>null</code> sein
	 */
	public RdsTmcCodeWrapper getRdsTmcCodeWrapper(final Zahl<Integer> ereignisCode) {
		ensureInit();
		assert ereignisCode != null;
		if (rdsTmcCodeMap != null) {
			return rdsTmcCodeMap.get(ereignisCode);
		}
		return null;
	}

	/**
	 * Liefert die zu einer gegeben Ereigniskategorie dazugehörige Liste von
	 * Ereignis-Codes zurück.
	 * 
	 * @param ereignisKategorie der Ereignis-Code, darf nicht <code>null</code> sein
	 * 
	 * @return die zu einer gegeben Ereigniskategorie dazugehörige Liste von
	 *         Ereignis-Codes, niemals <code>null</code>
	 */
	public List<Zahl<Integer>> getCodes(final AttRdsEreignisKategorie ereignisKategorie) {
		ensureInit();
		assert ereignisKategorie != null;
		if (kategorieEreignisCodeMap != null) {
			return kategorieEreignisCodeMap.get(ereignisKategorie);
		}
		return new ArrayList<>();
	}

	/**
	 * Liefert den zu einer gegebenen RDS-Quantitaet dazugehörigen
	 * RdsQuantitaetWrapper zurück.
	 * 
	 * @param rdsQuantitaet die RDS-Quantitaet
	 * 
	 * @return den zu einer gegebenen RDS-Quantitaet dazugehörigen
	 *         RdsQuantitaetWrapper
	 */
	public RdsQuantitaetWrapper getRdsQuantitaetWrapper(final RdsQuantitaet rdsQuantitaet) {
		ensureInit();
		if ((rdsQuantitaet != null) && (quantitaetWrapperMap != null)) {
			return quantitaetWrapperMap.get(rdsQuantitaet);
		}
		return null;
	}

	/**
	 * Liefert die zu einer gegebenen Kennung dazugehörige {@link RdsQuantitaet}
	 * zurück.
	 * 
	 * @param kennung die Kennung, darf nicht <code>null</code> sein
	 * 
	 * @return die zu einer gegebenen Kennung dazugehörige {@link RdsQuantitaet}
	 */
	public RdsQuantitaet getRdsQuantitaet(final String kennung) {
		ensureInit();
		assert kennung != null;
		if (quantitaetenKennungMap != null) {
			return quantitaetenKennungMap.get(kennung);
		}
		return null;
	}

	/**
	 * Liefert das zu einer gegebenen Kennung dazugehörige
	 * {@link RdsQuantitaetWrapper}-Objekt zurück.
	 * 
	 * @param kennung die Kennung, darf nicht <code>null</code> sein
	 * 
	 * @return das zu einer gegebenen Kennung dazugehörige
	 *         {@link RdsQuantitaetWrapper}-Objekt
	 */
	public RdsQuantitaetWrapper getRdsQuantitaetWrapper(final String kennung) {
		final RdsQuantitaet rdsQuantitaet = getRdsQuantitaet(kennung);
		if (rdsQuantitaet != null) {
			return getRdsQuantitaetWrapper(rdsQuantitaet);
		}
		return null;
	}

	/**
	 * Liefert eine sortierte Liste aller im System definierter RDS-Quantitäten
	 * zurück.
	 * 
	 * @return eine sortierte Liste aller im System definierter RDS-Quantitäten,
	 *         niemals <code>null</code>
	 */
	public List<RdsQuantitaetWrapper> getRdsQuantitaetWrapper() {
		ensureInit();
		return getRdsQuantitaetWrapperInternal();
	}

	/**
	 * Liefert eine sortierte Liste aller im System definierter RDS-Quantitäten
	 * zurück.
	 * 
	 * <p>
	 * Achtung: Methode zur internen Benutzung (ohne vorangestellten Methodenaufruf
	 * {@link #ensureInit()}.
	 * </p>
	 * 
	 * @return eine sortierte Liste aller im System definierter RDS-Quantitäten,
	 *         niemals <code>null</code>
	 */
	@SuppressWarnings("unchecked")
	private List<RdsQuantitaetWrapper> getRdsQuantitaetWrapperInternal() {
		if (rdsQuantitaetWrapper == null) {
			Collection<RdsQuantitaetWrapper> wrapper = null;
			if (quantitaetWrapperMap != null) {
				wrapper = quantitaetWrapperMap.values();
			}
			if (wrapper != null) {
				rdsQuantitaetWrapper = new ArrayList<>(wrapper);
				Collections.sort(rdsQuantitaetWrapper, new Comparator() {
					@Override
					public int compare(final Object o1, final Object o2) {
						if ((o1 != null) && (o2 != null)) {
							return o1.toString().compareTo(o2.toString());
						}
						return 0;
					}
				});
			}
		}
		if (rdsQuantitaetWrapper != null) {
			return rdsQuantitaetWrapper;
		}
		return new ArrayList<>();
	}

	/**
	 * Liefert den zu einer gegebenen RDS-Landeskennung den dazugehörigen
	 * RdsLandesKennungWrapper zurück.
	 * 
	 * @param rdsLandesKennung die RDS-Landeskennung, darf nicht <code>null</code>
	 *                         sein
	 * 
	 * @return den zu einer gegebenen RDS-Landeskennung dazugehörigen
	 *         RdsLandesKennungWrapper
	 */
	public RdsLandesKennungWrapper getRdsLandesKennungWrapper(final RdsLandesKennung rdsLandesKennung) {
		ensureInit();
		assert rdsLandesKennung != null;
		if (landesKennungWrapperMap != null) {
			return landesKennungWrapperMap.get(rdsLandesKennung);
		}
		return null;
	}

	/**
	 * Liefert eine sortierte Liste aller im System definierter RDS-Landeskennungen
	 * zurück.
	 * 
	 * @return eine sortierte Liste aller im System definierter RDS-Landeskennungen,
	 *         niemals <code>null</code>
	 */
	public List<RdsLandesKennungWrapper> getRdsLandesKennungWrapper() {
		ensureInit();
		return getRdsLandesKennungWrapperInternal();
	}

	/**
	 * Liefert eine sortierte Liste aller im System definierter RDS-Landeskennungen
	 * zurück.
	 * 
	 * <p>
	 * Achtung: Methode zur internen Benutzung (ohne vorangestellten Methodenaufruf
	 * {@link #ensureInit()}.
	 * </p>
	 * 
	 * @return eine sortierte Liste aller im System definierter RDS-Landeskennungen,
	 *         niemals <code>null</code>
	 */
	@SuppressWarnings("unchecked")
	private List<RdsLandesKennungWrapper> getRdsLandesKennungWrapperInternal() {
		if (rdsLandesKennungWrapper == null) {
			Collection<RdsLandesKennungWrapper> wrapper = null;
			if (landesKennungWrapperMap != null) {
				wrapper = landesKennungWrapperMap.values();
			}
			if (wrapper != null) {
				rdsLandesKennungWrapper = new ArrayList<>(wrapper);
				Collections.sort(rdsLandesKennungWrapper, new Comparator() {
					@Override
					public int compare(final Object o1, final Object o2) {
						if ((o1 != null) && (o2 != null)) {
							return o1.toString().compareTo(o2.toString());
						}
						return 0;
					}
				});
			}
		}
		if (rdsLandesKennungWrapper != null) {
			return rdsLandesKennungWrapper;
		}
		return new ArrayList<>();
	}

	/**
	 * Liefert das zu einem gegebenen Landescode gehörige
	 * {@link RdsLandesKennung}-SystemObjekt zurück.
	 * 
	 * @param landesCode der Landescode, darf nicht <code>null</code> sein
	 * 
	 * @return das zu einem gegebenen Landescode gehörige
	 *         {@link RdsLandesKennung}-SystemObjekt
	 */
	public RdsLandesKennung getRdsLandesKennung(final String landesCode) {
		ensureInit();
		assert landesCode != null;
		if (code2LandesKennungMap != null) {
			return code2LandesKennungMap.get(landesCode);
		}
		return null;
	}

	/**
	 * Liefert für eine gegebene Sammlung von {@link SystemObjekt}en, die
	 * dazugehörige Sammlung vom {@link SystemObject}s zurück.
	 * 
	 * @param objekte die Sammlung von {@link SystemObjekt}en
	 * 
	 * @return die dazugehörige Sammlung vom {@link SystemObject}s
	 */
	private Collection<SystemObject> resolveSystemObjects(final Collection<? extends SystemObjekt> objekte) {
		final ArrayList<SystemObject> objects = new ArrayList<>(objekte.size());
		for (final SystemObjekt objekt : objekte) {
			objects.add(objekt.getSystemObject());
		}
		return objects;
	}
}
