/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2011 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.bitctrl.modell.util.cache;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import de.bsvrz.sys.funclib.bitctrl.modell.att.Feld;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.attribute.AtlBetriebsKilometer;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.StrassenTeilSegment;

/**
 * Wrapper, welcher für ein StrassenTeilSegment-SystemObjekt sämtliche Daten
 * vorhält.
 * 
 * @author BitCtrl Systems GmbH, Andreas Meissner
 * 
 * @see StrassenTeilSegment
 */
public class StrassenTeilSegmentWrapper {

	/** Das gewrappte Straßenteilsegment. */
	private final StrassenTeilSegment strassenTeilSegment;

	/** Anzahl der Fahrstreifen. */
	private final byte anzahlFahrstreifen;

	/** Länge des Straßenteilsegments in Meter. */
	private final double laenge;

	/**
	 * Liste aller für dieses Straßenteilsegment definierte Betriebskilometer.
	 */
	private final Map<String, List<AtlBetriebsKilometer>> betriebsKilometerMap;

	/**
	 * Der Konstruktor.
	 * 
	 * @param strassenTeilSegment das zu wrappende Straßenteilsegment, nicht
	 *                            <code>null</code>
	 */
	/* package */ StrassenTeilSegmentWrapper(final StrassenTeilSegment strassenTeilSegment) {

		if (strassenTeilSegment == null) {
			throw new IllegalArgumentException("Übergebene Straßenteilsegment darf nicht null sein.");
		}

		final de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.konfigurationsdaten.KdStrassenTeilSegment.Daten kdStrassenTeilSegment = strassenTeilSegment
				.getKdStrassenTeilSegment().getDatum();
		final de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.konfigurationsdaten.KdBetriebsKilometer.Daten kdBetriebsKilometer = strassenTeilSegment
				.getKdBetriebsKilometer().getDatum();

		this.strassenTeilSegment = strassenTeilSegment;
		laenge = kdStrassenTeilSegment.getLaenge().intValue();
		anzahlFahrstreifen = kdStrassenTeilSegment.getAnzahlFahrStreifen().getValue();

		// ermittel Betriebskilometer
		betriebsKilometerMap = new HashMap<>();
		if (kdBetriebsKilometer != null) {
			final Feld<AtlBetriebsKilometer> betriebsKilometerFeld = kdBetriebsKilometer.getBetriebsKilometer();
			if (betriebsKilometerFeld != null) {
				for (final AtlBetriebsKilometer bk : betriebsKilometerFeld) {
					String blockNummer = bk.getBlockNummer();
					if ((blockNummer == null) || "".equals(blockNummer.trim())) {
						blockNummer = "-1";
					}
					List<AtlBetriebsKilometer> list = betriebsKilometerMap.get(blockNummer);
					if (list == null) {
						list = new ArrayList<>();
						betriebsKilometerMap.put(blockNummer, list);
					}
					list.add(bk);
				}
			}
		}
	}

	/**
	 * Liefert das gewrappte Straßenteilsegment zurück.
	 * 
	 * @return das gewrappte Straßenteilsegment
	 */
	public StrassenTeilSegment getStrassenTeilSegment() {
		return strassenTeilSegment;
	}

	/**
	 * Liefert die Anzahl der Fahrstreifen für dieses Straßenteilsegment zurück.
	 * 
	 * @return die Anzahl der Fahrstreifen für dieses Straßenteilsegment
	 */
	public byte getAnzahlFahrstreifen() {
		return anzahlFahrstreifen;
	}

	/**
	 * Liefert die Länge dieses Straßenteilsegmentes zurück (Einheit: Meter).
	 * 
	 * @return die Länge dieses Straßenteilsegmentes (Einheit: Meter)
	 */
	public double getLaenge() {
		return laenge;
	}

	/**
	 * Liefert die Liste aller für dieses Straßenteilsegment gesetzten
	 * Betriebskilometer zurück.
	 * 
	 * @param blockNummer die BlockNummer
	 * 
	 * @return die Liste aller für dieses Straßenteilsegment gesetzten
	 *         Betriebskilometer, niemals <code>null</code>
	 */
	public List<AtlBetriebsKilometer> getBetriebsKilometer(final String blockNummer) {
		if (blockNummer == null) {
			return new ArrayList<>();
		}
		final List<AtlBetriebsKilometer> ergebnis = betriebsKilometerMap.get(blockNummer);
		if (ergebnis != null) {
			return ergebnis;
		}
		return new ArrayList<>();
	}

	/**
	 * Liefert die Menge der Blocknummern zurück, für welche Betriebskilometer für
	 * dieses Straßenteilsegment definiert sind.
	 * 
	 * @return die Menge der Blocknummern, für welche Betriebskilometer für dieses
	 *         Straßenteilsegment definiert sind, niemals <code>null</code>
	 */
	public List<String> getBlockNummern() {
		if (betriebsKilometerMap.keySet() != null) {
			return new ArrayList<>(betriebsKilometerMap.keySet());
		}
		return new ArrayList<>();
	}
}
