/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2011 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.bitctrl.modell.util.cache;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.bitctrl.util.monitor.IMonitor;
import com.bitctrl.util.monitor.SubMonitor;

import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcglobal.attribute.AttTmcRichtung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.konfigurationsdaten.KdStrasse.Daten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.AeusseresStrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.InneresStrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.Strasse;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.StrassenKnoten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.StrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.VerkehrsModellNetz;

/**
 * Der {@link StreckenAbschnittCache} cached nicht alle Inhalte aus dem
 * {@link NetzCacheExtended} und wurde aus diesem Grund in einen eigenen Cache
 * ausgelagert.
 *
 * @author BitCtrl Systems GmbH, schnepel
 */
public class StreckenAbschnittCache extends AbstractCache {

	private NetzCache netzCache;
	private NetzCacheExtended netzCacheExtended;
	private StrassenTeilSegmenteCache strassenTeilSegmenteCache;

	/**
	 * Map, welche zu jeder Straße (abhängig von der Straßenrichtung) die
	 * existierenden Straßenabschnitte speichert.
	 */
	private Map<AttTmcRichtung, Map<String, List<StreckenAbschnitt>>> streckenAbschnitteMap;

	/**
	 * Der Konstruktor.
	 *
	 * @param parent der StrassenTeilSegmenteCache, von welchem dieser Cache
	 *               abhängig ist
	 */
	public StreckenAbschnittCache(final StrassenTeilSegmenteCache parent) {
		super(parent);
	}

	/**
	 * Initialisiert den Cache.
	 *
	 * @param parent der StrassenTeilSegmenteCache, von welchem dieser Cache
	 *               abhängig ist
	 */
	protected void preInit(final StrassenTeilSegmenteCache parent) {
		strassenTeilSegmenteCache = parent;
		netzCacheExtended = parent.getNetzCacheExtended();
		netzCache = netzCacheExtended.getNetzCache();
	}

	@Override
	protected boolean doInit(final IMonitor monitor) {
		setName(getNetzCacheExtended().getName());

		final SubMonitor subMonitor = SubMonitor.convert(monitor);
		final int work = 40000;
		subMonitor.beginTask("Initialisiere Streckenabschnitte", work);

		if (bestimmeStreckenAbschitte(subMonitor.newChild(30000))) {
			return sortiereStrassenKnoten(subMonitor.newChild(10000));
		}
		return false;
	}

	/**
	 * Liefert die Straßenknoten zurück, welche auf einer gegebenen Straße einen
	 * Startstraßenknoten für eine Engstelle darstellen können.
	 *
	 * @param strasse  die Strasse, darf nicht <code>null</code> sein
	 * @param richtung die Richtung, darf nicht <code>null</code> sein
	 *
	 * @return die Straßenknoten, welche auf einer gegebenen Straße einen
	 *         Startstraßenknoten für eine Engstelle darstellen können, niemals
	 *         <code>null</code>
	 */
	public List<StrassenKnoten> getStartStrassenKnoten(final Strasse strasse, final AttTmcRichtung richtung) {
		ensureInit();
		if (strasse == null) {
			throw new IllegalArgumentException("Übergebene Straße darf nicht null sein.");
		}
		if (richtung == null) {
			throw new IllegalArgumentException("Übergebene Richtung darf nicht null sein.");
		}

		final List<StrassenKnoten> ergebnis = netzCacheExtended.getStrassenKnoten(strasse, richtung);
		if ((ergebnis == null) || ergebnis.isEmpty()) {
			return new ArrayList<>();
		}

		// Möglichen Startstraßenknoten sind alle Straßenknoten,
		// die mindestens ein inneres Straßensegment besitzen oder
		// Nachfolgersegmente haben
		for (final Iterator<StrassenKnoten> iter = ergebnis.iterator(); iter.hasNext();) {
			final StrassenKnoten knoten = iter.next();
			if (!getNachfolgerStrassenKnoten(strasse, richtung, knoten).isEmpty()) {
				continue;
			}
			if (knoten.getInnereStrassenSegmente() != null) {
				for (final InneresStrassenSegment iss : knoten.getInnereStrassenSegmente()) {
					if (richtung.equals(netzCacheExtended.getRichtung(iss))
							&& strasse.equals(netzCacheExtended.getStrasse(iss))) {
						continue;
					}
				}
			}
			iter.remove();
		}
		return ergebnis;
	}

	/**
	 * Liefert die Straßenknoten zurück, welche sich auf der gegebenen Straße nach
	 * dem gegebenen Straßenknoten befinden.
	 *
	 * @param strasse  die Strasse, nicht <code>null</code>
	 * @param richtung die Richtung, nicht <code>null</code>
	 * @param knoten   der Straßenknoten, nicht <code>null</code>
	 *
	 * @return die Straßenknoten, welche sich auf der gegebenen Straße nach dem
	 *         gegebenen Straßenknoten befinden, niemals <code>null</code>
	 */
	public List<StrassenKnoten> getNachfolgerStrassenKnoten(final Strasse strasse, final AttTmcRichtung richtung,
			final StrassenKnoten knoten) {
		ensureInit();
		if (strasse == null) {
			throw new IllegalArgumentException("Übergebene Straße darf nicht null sein.");
		}
		if (knoten == null) {
			throw new IllegalArgumentException("Übergebener Straßenknoten darf nicht null sein.");
		}

		final List<StrassenKnoten> ergebnis = new ArrayList<>();

		final String strassenNummer = netzCacheExtended.getStrassenNummer(strasse);
		if (strassenNummer != null) {
			final Map<String, List<StreckenAbschnitt>> richtungsMap = streckenAbschnitteMap.get(richtung);
			if (richtungsMap != null) {
				final List<StreckenAbschnitt> streckenAbschnitte = richtungsMap.get(strassenNummer);
				if (streckenAbschnitte != null) {
					for (final StreckenAbschnitt streckenAbschnitt : streckenAbschnitte) {
						if (streckenAbschnitt.getStrassenKnoten().contains(knoten)) {
							final List<StrassenKnoten> nachfolgerStrassenKnoten = streckenAbschnitt
									.getNachfolgerStrassenKnoten(knoten);
							for (final StrassenKnoten nachfolgeKnoten : nachfolgerStrassenKnoten) {
								if (!ergebnis.contains(nachfolgeKnoten)) {
									ergebnis.add(nachfolgeKnoten);
								}
							}
						}
					}
				}
			}
		}
		return ergebnis;
	}

	/**
	 * Liefert die Straßenknoten zurück, welche sich auf der gegebenen Straße vor
	 * dem gegebenen Straßenknoten befinden.
	 *
	 * @param strasse  die Strasse, nicht <code>null</code>
	 * @param richtung die Richtung, nicht <code>null</code>
	 * @param knoten   der Straßenknoten, nicht <code>null</code>
	 *
	 * @return die Straßenknoten, welche sich auf der gegebenen Straße nach dem
	 *         gegebenen Straßenknoten befinden, niemals <code>null</code>
	 */
	public List<StrassenKnoten> getVorgaengerStrassenKnoten(final Strasse strasse, final AttTmcRichtung richtung,
			final StrassenKnoten knoten) {
		ensureInit();
		if (strasse == null) {
			throw new IllegalArgumentException("Übergebene Straße darf nicht null sein.");
		}
		if (knoten == null) {
			throw new IllegalArgumentException("Übergebener Straßenknoten darf nicht null sein.");
		}

		final List<StrassenKnoten> ergebnis = new ArrayList<>();

		final String strassenNummer = netzCacheExtended.getStrassenNummer(strasse);
		if (strassenNummer != null) {
			final Map<String, List<StreckenAbschnitt>> richtungsMap = streckenAbschnitteMap.get(richtung);
			if (richtungsMap != null) {
				final List<StreckenAbschnitt> streckenAbschnitte = richtungsMap.get(strassenNummer);
				if (streckenAbschnitte != null) {
					for (final StreckenAbschnitt streckenAbschnitt : streckenAbschnitte) {
						if (streckenAbschnitt.getStrassenKnoten().contains(knoten)) {
							final List<StrassenKnoten> vorgaengerStrassenKnoten = streckenAbschnitt
									.getVorgaengerStrassenKnoten(knoten);
							for (final StrassenKnoten vorgaengerKnoten : vorgaengerStrassenKnoten) {
								if (!ergebnis.contains(vorgaengerKnoten)) {
									ergebnis.add(vorgaengerKnoten);
								}
							}
						}
					}
				}
			}
		}
		return ergebnis;
	}

	/**
	 * Hilfsmethode, welche die sämtliche Streckenabschnitte für alle verfügbaren
	 * Strassen bestimmt. Diese Methode initialisiert die streckenAbschnittsMap.
	 *
	 * @param monitor der Monitor
	 *
	 * @return <code>false</code>, wenn der Monitor gecancelt wurde, ansonsten
	 *         <code>true</code>
	 *
	 * @see #streckenAbschnitteMap
	 */
	private boolean bestimmeStreckenAbschitte(final IMonitor monitor) {

		monitor.setTaskName("Bestimme Straßenabschnitte");

		int i = 0;
		final int totalWork = 1000000;
		final List<Strasse> strassen = netzCacheExtended.getStrassen();
		final int anzahlStrassen = strassen.size() * AttTmcRichtung.getZustaende().size();
		final int work = totalWork / anzahlStrassen;

		streckenAbschnitteMap = new HashMap<>();

		for (final AttTmcRichtung richtung : AttTmcRichtung.getZustaende()) {
			streckenAbschnitteMap.put(richtung, new HashMap<>());

			for (final Strasse strasse : strassen) {
				monitor.subTask("Analysisiere Straße: " + strasse.getName() + " - " + richtung.toString() + " (" + i
						+ "/" + anzahlStrassen + ")");

				final String strassenNummer = netzCacheExtended.getStrassenNummer(strasse);
				if (strassenNummer != null) {
					streckenAbschnitteMap.get(richtung).put(strassenNummer,
							bestimmeStreckenAbschnitte(strasse, richtung));
				}

				monitor.worked(work);
				if (monitor.isCanceled()) {
					monitor.done();
					return false;
				}
				i++;
			}
		}
		return true;
	}

	/**
	 * Hilfsmethode, welche die existierenden Streckenabschnitte für eine gegebene
	 * Strasse bestimmt.
	 *
	 * @param strasse  die Strasse, nicht <code>null</code>
	 * @param richtung die Strassenrichtung (positiv oder negativ)
	 *
	 * @return die Liste der existierenden Streckenabschnitte für die gegebene
	 *         Strasse, niemals <code>null</code>
	 */
	private List<StreckenAbschnitt> bestimmeStreckenAbschnitte(final Strasse strasse, final AttTmcRichtung richtung) {

		final Daten strassenDaten = strasse.getKdStrasse().getDatum();
		final String strassenNummer = netzCacheExtended.getStrassenNummer(strassenDaten);
		if ((strassenDaten == null) || (strassenNummer == null)) {
			return new ArrayList<>();
		}

		final List<StrassenKnoten> knotenListe = netzCacheExtended.getStrassenKnoten(strassenDaten, richtung);
		if (knotenListe.isEmpty()) {
			return new ArrayList<>();
		}

		// (1) bestimme alle mögliche Startknoten
		final List<StrassenKnoten> startKnoten = new ArrayList<>();
		for (final StrassenKnoten knoten : knotenListe) {
			final List<AeusseresStrassenSegment> segmenteVorKnoten = netzCacheExtended
					.getAeussereStrassenSegmenteVorStrassenKnoten(knoten, strassenDaten, richtung);
			if ((segmenteVorKnoten == null) || segmenteVorKnoten.isEmpty()) {
				startKnoten.add(knoten);
			} else {
				// Für den Fall, dass vor den Knoten zwar ein
				// AeusseresStraßensegment existiert, dieses jedoch keinen
				// Startknoten hat.
				for (final AeusseresStrassenSegment as : segmenteVorKnoten) {
					if (as.getKdAeusseresStrassenSegment().getDatum().getVonKnoten() == null) {
						startKnoten.add(knoten);
					}
				}
			}
		}

		// (2) bestimme die Startsegmente
		final List<AeusseresStrassenSegment> startSegmente = new ArrayList<>();
		for (final StrassenKnoten knoten : startKnoten) {
			final List<AeusseresStrassenSegment> segmenteNachStartKnoten = netzCacheExtended
					.getAeussereStrassenSegmenteNachStrassenKnoten(knoten, strassenDaten, richtung);
			if (segmenteNachStartKnoten != null) {
				for (final AeusseresStrassenSegment segment : segmenteNachStartKnoten) {
					if (!startSegmente.contains(segment)) {
						startSegmente.add(segment);
					}
				}
			}
		}

		// (3) bestimme die möglichen Streckenabschnitte
		int i = 0;
		List<AeusseresStrassenSegment> weitereStartSegmente = new ArrayList<>();
		final Set<AeusseresStrassenSegment> alleAnalyisiertenStartSegmente = new HashSet<>();
		final List<StreckenAbschnitt> alleStreckenAbschnitte = new ArrayList<>();

		while ((i == 0) || !weitereStartSegmente.isEmpty()) {

			final List<AeusseresStrassenSegment> aktuelleStartSegmente = (i == 0) ? startSegmente
					: new ArrayList<>(weitereStartSegmente);
			alleAnalyisiertenStartSegmente.addAll(aktuelleStartSegmente);
			weitereStartSegmente = new ArrayList<>();

			for (final AeusseresStrassenSegment startSegment : aktuelleStartSegmente) {
				final List<StreckenAbschnitt> streckenAbschnitte = new ArrayList<>();
				final StreckenAbschnitt ersterAbschnitt = new StreckenAbschnitt(this, strassenNummer, richtung);
				streckenAbschnitte.add(ersterAbschnitt);
				fuelleStreckenAbschnitte(streckenAbschnitte, ersterAbschnitt, weitereStartSegmente, startSegment,
						strassenDaten, richtung);
				alleStreckenAbschnitte.addAll(streckenAbschnitte);
			}
			weitereStartSegmente.removeAll(alleAnalyisiertenStartSegmente);
			i++;
		}
		return alleStreckenAbschnitte;
	}

	/**
	 * Hilfsmethode, welche rekursiv einen Streckenabschnitt füllt.
	 *
	 * @param streckenAbschnitte   die Ergebnisliste aller Streckenabschnitte für
	 *                             die gegebene Strasse (collecting parameter
	 *                             pattern)
	 * @param abschnitt            der zu füllende Streckenabschnitt, nicht
	 *                             <code>null</code>
	 * @param weitereStartSegmente eine Liste mit möglichen weiteren
	 *                             Start-Straßensegmenten (collecting parameter
	 *                             pattern)
	 * @param segment              das Segment, mit welchem der Streckenabschnitt
	 *                             erweitert werden soll
	 * @param strassenDaten        die Straße, nicht <code>null</code>
	 * @param richtung             die Straßenrichtung (positiv oder negativ)
	 */
	private void fuelleStreckenAbschnitte(final List<StreckenAbschnitt> streckenAbschnitte,
			final StreckenAbschnitt abschnitt, final List<AeusseresStrassenSegment> weitereStartSegmente,
			final AeusseresStrassenSegment segment, final Daten strassenDaten, final AttTmcRichtung richtung) {

		final StrassenKnoten nachKnoten = segment.getKdAeusseresStrassenSegment().getDatum().getNachKnoten();
		if (nachKnoten == null) {
			return;
		}

		// Äußeres Straßensegment nur hinzufügen, wenn es auch ein inneres
		// Straßensegment gibt, welche dieses äußere Straßensegement mit dem
		// aktuell letzten äußeren Straßensegment verbindet
		boolean fuegeSegmentHinzu = false;
		final AeusseresStrassenSegment aktuellesEndSegment = abschnitt.getEndASS();
		if (aktuellesEndSegment == null) {
			fuegeSegmentHinzu = true;
		} else {
			final StrassenKnoten aktuellerNachKnoten = aktuellesEndSegment.getKdAeusseresStrassenSegment().getDatum()
					.getNachKnoten();
			final StrassenKnoten vonKnoten = segment.getKdAeusseresStrassenSegment().getDatum().getVonKnoten();
			if ((aktuellerNachKnoten != null) && (vonKnoten != null) && aktuellerNachKnoten.equals(vonKnoten)) {
				final Collection<InneresStrassenSegment> innereSegmente = vonKnoten.getInnereStrassenSegmente();
				if (innereSegmente != null) {
					for (final InneresStrassenSegment inneresSegment : innereSegmente) {
						final AeusseresStrassenSegment vonSegment = inneresSegment.getKdInneresStrassenSegment()
								.getDatum().getVonStrassenSegment();
						final AeusseresStrassenSegment nachSegment = inneresSegment.getKdInneresStrassenSegment()
								.getDatum().getNachStrassenSegment();
						if (aktuellesEndSegment.equals(vonSegment) && segment.equals(nachSegment)) {
							fuegeSegmentHinzu = true;
							break;
						}
					}
				}
			}
		}

		if (fuegeSegmentHinzu) {
			abschnitt.addSegment(segment);
		} else {
			if (!weitereStartSegmente.contains(segment)) {
				weitereStartSegmente.add(segment);
			}
			return;
		}

		final List<AeusseresStrassenSegment> nachfolgerSegmente = netzCacheExtended
				.getAeussereStrassenSegmenteNachStrassenKnoten(nachKnoten, strassenDaten, richtung);
		if (nachfolgerSegmente == null) {
			return;
		}

		final Iterator<AeusseresStrassenSegment> iter = nachfolgerSegmente.iterator();
		final StreckenAbschnitt tempAbschnitt = new StreckenAbschnitt(abschnitt);
		if (iter.hasNext()) {
			fuelleStreckenAbschnitte(streckenAbschnitte, abschnitt, weitereStartSegmente, iter.next(), strassenDaten,
					richtung);
		}

		while (iter.hasNext()) {
			final StreckenAbschnitt neuerAbschnitt = new StreckenAbschnitt(tempAbschnitt);
			streckenAbschnitte.add(neuerAbschnitt);
			fuelleStreckenAbschnitte(streckenAbschnitte, neuerAbschnitt, weitereStartSegmente, iter.next(),
					strassenDaten, richtung);
		}
	}

	/**
	 * Liefert die Streckenabschnitte zurück, welche auf der gegebenen Strasse
	 * liegen.
	 *
	 * @param strasse  die Strasse, nicht <code>null</code>
	 * @param richtung die Richtung
	 *
	 * @return die Streckenabschnitte, welche auf der gegebenen Strasse liegen,
	 *         niemals <code>null</code>
	 */
	public List<StreckenAbschnitt> getStreckenAbschnitte(final Strasse strasse, final AttTmcRichtung richtung) {
		ensureInit();
		return getStreckenAbschnitteInternal(strasse, richtung);
	}

	private List<StreckenAbschnitt> getStreckenAbschnitteInternal(final Strasse strasse,
			final AttTmcRichtung richtung) {
		if (strasse == null) {
			throw new IllegalArgumentException("Übergebener Straße darf nicht null sein.");
		}
		final String strassenNummer = netzCacheExtended.getStrassenNummer(strasse);
		if (strassenNummer != null) {
			final Map<String, List<StreckenAbschnitt>> richtungsMap = streckenAbschnitteMap.get(richtung);
			if (richtungsMap != null) {
				final List<StreckenAbschnitt> ergebnis = richtungsMap.get(strassenNummer);
				if (ergebnis != null) {
					return ergebnis;
				}
			}
		}
		return new ArrayList<>();
	}

	/**
	 * Liefert die Streckenabschnitte zurück, welche auf der gegebenen Straße liegen
	 * und deren äußere Straßensegmente alle auf '0' endende PID haben.
	 *
	 * @param strasse  die Strasse, nicht <code>null</code>
	 * @param richtung die Richtung (positiv oder negativ)
	 *
	 * @return die Standardstreckenabschnitte, niemals <code>null</code>
	 */
	public List<StreckenAbschnitt> getStandardStreckenAbschnitte(final Strasse strasse, final AttTmcRichtung richtung) {
		ensureInit();
		return getStandardStreckenAbschnitteInternal(strasse, richtung);
	}

	private List<StreckenAbschnitt> getStandardStreckenAbschnitteInternal(final Strasse strasse,
			final AttTmcRichtung richtung) {
		final List<StreckenAbschnitt> ergebnis = new ArrayList<>();
		for (final StreckenAbschnitt abschnitt : getStreckenAbschnitteInternal(strasse, richtung)) {
			if (abschnitt.istStandardStreckenAbschnitt()) {
				ergebnis.add(abschnitt);
			}
		}
		return ergebnis;
	}

	/**
	 * Liefert den bevorzugten Streckenabschnitt für eine gegebene Straße zurück.
	 * Der bevorzugte Streckenabschnitt stellt der längste (also aus den meisten
	 * äußeren Straßensegmenten bestehende) Standardstreckenabschnitt dar.
	 *
	 * @param strasse  die Strasse, nicht <code>null</code>
	 * @param richtung die Richtung (positiv oder negativ)
	 *
	 * @return den bevorzugten Streckenabschnitt für eine gegebene Straße, kann
	 *         <code>null</code> sein
	 */
	public StreckenAbschnitt getStandardStreckenAbschnitt(final Strasse strasse, final AttTmcRichtung richtung) {
		ensureInit();
		StreckenAbschnitt ergebnis = null;
		int maxAnzahlSegment = 0;
		for (final StreckenAbschnitt abschnitt : getStandardStreckenAbschnitteInternal(strasse, richtung)) {
			final int anzahlSegmente = abschnitt.getAeussereStrassenSegmente().size();
			if (anzahlSegmente > maxAnzahlSegment) {
				maxAnzahlSegment = anzahlSegmente;
				ergebnis = abschnitt;
			}
		}

		// wenn kein Standardstreckenabschnitt gefunden wurde, nimm den ersten
		// verfügbaren Streckenabschnitt
		if (ergebnis == null) {
			final List<StreckenAbschnitt> alleAbschnitte = getStreckenAbschnitteInternal(strasse, richtung);
			if (!alleAbschnitte.isEmpty()) {
				ergebnis = alleAbschnitte.get(0);
			}
		}
		return ergebnis;
	}

	/**
	 * Liefert die Streckenabschnitte zurück, welche auf der gegebenen Strasse
	 * liegen sowie die gegebenen Segmente enthält.
	 *
	 * @param strasse  die Strasse, nicht <code>null</code>
	 * @param richtung die Richtung
	 * @param segmente Liste von Straßensegmenten, welche im Streckenabschnitt
	 *                 enthalten sein müssen, nicht <code>null</code>
	 *
	 * @return Streckenabschnitte, welche auf der gegebenen Strasse liegen sowie die
	 *         gegebenen Segmente enthält, niemals <code>null</code>
	 */
	public List<StreckenAbschnitt> getStreckenAbschnitte(final Strasse strasse, final AttTmcRichtung richtung,
			final List<StrassenSegment> segmente) {
		ensureInit();
		return getStreckenAbschnitteInternal(strasse, richtung, segmente);
	}

	private List<StreckenAbschnitt> getStreckenAbschnitteInternal(final Strasse strasse, final AttTmcRichtung richtung,
			final List<StrassenSegment> segmente) {
		if (segmente == null) {
			throw new IllegalArgumentException(
					"Übergebener Liste mit Straßensegmenten darf leer aber nicht null sein.");
		}

		final List<StreckenAbschnitt> streckenAbschnitte = new ArrayList<>(
				getStreckenAbschnitteInternal(strasse, richtung));

		// Klone Streckenabschnitte
		final List<StreckenAbschnitt> ergebnisListe = new ArrayList<>();
		for (final StreckenAbschnitt abschnitt : streckenAbschnitte) {
			ergebnisListe.add(new StreckenAbschnitt(abschnitt));
		}

		for (final Iterator<StreckenAbschnitt> iter = ergebnisListe.iterator(); iter.hasNext();) {
			final List<InneresStrassenSegment> innereSegmente = new ArrayList<>();
			final StreckenAbschnitt abschnitt = iter.next();
			for (final StrassenSegment segment : segmente) {
				if (segment instanceof AeusseresStrassenSegment) {
					if (!abschnitt.getAeussereStrassenSegmente().contains(segment)) {
						iter.remove();
						break;
					}
				} else if (segment instanceof InneresStrassenSegment) {
					innereSegmente.add((InneresStrassenSegment) segment);
				}
			}
			for (final InneresStrassenSegment inneresSegment : innereSegmente) {
				abschnitt.setInneresStrassenSegment(inneresSegment);
			}
		}
		return ergebnisListe;
	}

	/**
	 * Liefert die Streckenabschnitte zurück, welche den gegebenen Start- und
	 * Endknoten auf der gegebenen Straße verbinden.
	 *
	 * <p>
	 * Die gegebenen Start- und Endknoten müssen auf der Straße liegen und dürfen
	 * nicht identisch sein.
	 * </p>
	 *
	 * @param strasse     die Strasse, nicht <code>null</code>
	 * @param richtung    die Richtung (positiv oder negativ)
	 * @param startKnoten der Startstraßenknoten, nicht <code>null</code>
	 * @param endKnoten   der Endstraßenknoten, nicht <code>null</code>
	 *
	 * @return die Streckenabschnitte, welche den gegebenen Start- und Endknoten auf
	 *         der gegebenen Straße verbinden, niemals <code>null</code>
	 */
	public List<StreckenAbschnitt> getStreckenAbschnitte(final Strasse strasse, final AttTmcRichtung richtung,
			final StrassenKnoten startKnoten, final StrassenKnoten endKnoten) {
		ensureInit();
		return getStreckenAbschnitteInternal(strasse, richtung, startKnoten, endKnoten);
	}

	private List<StreckenAbschnitt> getStreckenAbschnitteInternal(final Strasse strasse, final AttTmcRichtung richtung,
			final StrassenKnoten startKnoten, final StrassenKnoten endKnoten) {
		if ((startKnoten == null) || (endKnoten == null)) {
			throw new IllegalArgumentException("Übergebenener Start- und Endstraßenknoten darf nicht null sein.");
		}

		final List<StreckenAbschnitt> ergebnis = new ArrayList<>();
		for (final StreckenAbschnitt abschnitt : getStreckenAbschnitteInternal(strasse, richtung)) {
			if (abschnitt.getStrassenKnoten().contains(startKnoten)
					&& abschnitt.getStrassenKnoten().contains(endKnoten)) {
				final StreckenAbschnitt neuerAbschnitt = new StreckenAbschnitt(abschnitt);
				neuerAbschnitt.setStartKnoten(startKnoten);
				neuerAbschnitt.setEndKnoten(endKnoten);
				if (!ergebnis.contains(neuerAbschnitt)) {
					ergebnis.add(neuerAbschnitt);
				}
			}
		}
		return ergebnis;
	}

	/**
	 * Liefert deb Streckenabschnitt zurück, welcher den gegebenen Start- und
	 * Endknoten auf der gegebenen Straße mit äußeren Straßensegmenten verbindet,
	 * welche eine auf '0' endende PID haben.
	 *
	 * @param strasse     die Strasse, nicht <code>null</code>
	 * @param richtung    die Richtung (positiv oder negativ)
	 * @param startKnoten der Startstraßenknoten, nicht <code>null</code>
	 * @param endKnoten   der Endstraßenknoten, nicht <code>null</code>
	 *
	 * @return den Standardstraßenabschnitt, bestehend aus äußeren Straßensegmenten
	 *         mit PID endend auf '0', kann <code>null</code> sein
	 */
	public StreckenAbschnitt getStandardStreckenAbschnitt(final Strasse strasse, final AttTmcRichtung richtung,
			final StrassenKnoten startKnoten, final StrassenKnoten endKnoten) {
		ensureInit();
		final List<StreckenAbschnitt> abschnitte = getStreckenAbschnitteInternal(strasse, richtung, startKnoten,
				endKnoten);
		for (final StreckenAbschnitt abschnitt : abschnitte) {
			if (abschnitt.istStandardStreckenAbschnitt()) {
				return abschnitt;
			}
		}
		if (!abschnitte.isEmpty()) {
			return abschnitte.get(0);
		}
		return null;
	}

	/**
	 * Sortiert eine gegebene Liste von äußeren Strassensegmenten.
	 *
	 * <p>
	 * Beachte: Um die äußeren Strassensegmente sortieren zu können, müssen diese
	 * alle auf derselben Straße (in gleicher Richtung) liegen.
	 * </p>
	 *
	 * @param aeussereSegmente die zu sortierenden Liste äußerer Strassensegmente,
	 *                         nicht <code>null</code>
	 *
	 * @return sortierte Liste von Strassensegmenten, niemals <code>null</code>
	 */
	public List<AeusseresStrassenSegment> sortiereStrassenSegmente(
			final List<AeusseresStrassenSegment> aeussereSegmente) {
		ensureInit();
		if (aeussereSegmente == null) {
			throw new IllegalArgumentException("Übergebene Strassensegmente dürfen nicht null sein");
		}
		if (aeussereSegmente.isEmpty()) {
			return aeussereSegmente;
		}

		final List<AeusseresStrassenSegment> segmente = new ArrayList<>(aeussereSegmente);

		// (1) ermittel zunächst das erste äußere Straßensegment, um die
		// dazugehörige Straße und Richtung zu bestimmen
		Strasse strasse = null;
		AttTmcRichtung richtung = null;
		for (final AeusseresStrassenSegment segment : segmente) {
			strasse = segment.getKdStrassenSegment().getDatum().getGehoertZuStrasse();
			richtung = segment.getKdAeusseresStrassenSegment().getDatum().getTmcRichtung();
			break;
		}
		if ((strasse == null) || (richtung == null)) {
			return segmente;
		}

		// (2) Ermittel eine Liste sortierter Straßensegmente ausgehende von
		// einem existierenden Streckenabschnitt, welcher die gegebenen
		// unsortierten Straßensegmente enthält
		final List<StreckenAbschnitt> streckenAbschnitte = getStreckenAbschnitteInternal(strasse, richtung,
				new ArrayList<StrassenSegment>(segmente));
		if (streckenAbschnitte.isEmpty()) {
			return segmente;
		}
		final StreckenAbschnitt streckenAbschnitt = streckenAbschnitte.get(0);
		final List<StrassenSegment> sortierteAbschittsSegmente = streckenAbschnitt.getStrassenSegmente();

		// (3) Sortiere die gegebenen Straßensegmente
		final List<StrassenSegment> ergebnisListe = new ArrayList<>(segmente);
		Collections.sort(ergebnisListe, new Comparator<StrassenSegment>() {
			@Override
			public int compare(final StrassenSegment s1, final StrassenSegment s2) {
				final int i1 = sortierteAbschittsSegmente.indexOf(s1);
				final int i2 = sortierteAbschittsSegmente.indexOf(s2);
				if (i1 < i2) {
					return -1;
				}
				if (i1 > i2) {
					return +1;
				}
				return 0;
			}
		});

		return segmente;
	}

	/**
	 * Sortiert die Straßenknoten in Durchfahrtsrichtung.
	 *
	 * @param monitor der Monitor
	 *
	 * @return <code>false</code>, wenn der Monitor gecancelt wurde, ansonsten
	 *         <code>true</code>
	 */
	private boolean sortiereStrassenKnoten(final IMonitor monitor) {

		monitor.setTaskName("Sortiere Straßenknoten");

		int i = 0;
		final int totalWork = 1000000;
		final List<Strasse> strassenMitSegmenten = netzCacheExtended.getStrassenMitSegmenten();
		final int anzahlStrassen = strassenMitSegmenten.size() * AttTmcRichtung.getZustaende().size();
		final int work = totalWork / anzahlStrassen;

		for (final Strasse strasse : strassenMitSegmenten) {
			for (final AttTmcRichtung richtung : AttTmcRichtung.getZustaende()) {

				monitor.subTask("Sortiere Straßenknoten: " + strasse.getName() + " - " + richtung.toString() + " (" + i
						+ "/" + anzahlStrassen + ")");

				final List<StreckenAbschnitt> streckenAbschnitte = new ArrayList<>(
						getStreckenAbschnitteInternal(strasse, richtung));
				Collections.sort(streckenAbschnitte, new Comparator<StreckenAbschnitt>() {
					@Override
					public int compare(final StreckenAbschnitt o1, final StreckenAbschnitt o2) {

						if (o1.istStandardStreckenAbschnitt() && !o2.istStandardStreckenAbschnitt()) {
							return -1;
						} else if (!o1.istStandardStreckenAbschnitt() && o2.istStandardStreckenAbschnitt()) {
							return +1;
						} else {
							return new Integer(o1.getStrassenKnoten().size())
									.compareTo(new Integer(o2.getStrassenKnoten().size()));
						}
					}
				});

				final List<StreckenAbschnitt> tempStreckenAbschnitte = new ArrayList<>(streckenAbschnitte);
				final List<StrassenKnoten> knoten = netzCacheExtended.getStrassenKnoten(strasse, richtung);
				Collections.sort(knoten, new Comparator<StrassenKnoten>() {
					@Override
					public int compare(final StrassenKnoten o1, final StrassenKnoten o2) {

						int j = 0;
						int index11 = -1;
						int index12 = -1;
						int index21 = -1;
						int index22 = -1;

						for (final StreckenAbschnitt abschnitt : tempStreckenAbschnitte) {
							if ((index11 == -1) && abschnitt.getStrassenKnoten().contains(o1)) {
								index11 = j;
								index12 = abschnitt.getStrassenKnoten().indexOf(o1);
							}
							if ((index21 == -1) && abschnitt.getStrassenKnoten().contains(o2)) {
								index21 = j;
								index22 = abschnitt.getStrassenKnoten().indexOf(o2);
							}
							if ((index11 != -1) && (index21 != -1)) {
								break;
							}
							j++;
						}

						final int compare = new Integer(index11).compareTo(new Integer(index21));
						if (compare != 0) {
							return compare;
						}
						return new Integer(index12).compareTo(new Integer(index22));
					}
				});

				monitor.worked(work);
				if (monitor.isCanceled()) {
					monitor.done();
					return false;
				}
				i++;
			}
		}
		return true;
	}

	/**
	 * Liefert den erweiterten Netz-Cache zurück, von welchem dieser Cache abhängig
	 * ist.
	 *
	 * @return den erweiterten Netz-Cache, von welchem dieser Cache abhängig ist
	 */
	protected NetzCacheExtended getNetzCacheExtended() {
		return netzCacheExtended;
	}

	/**
	 * Gibt das Netz zurück, welches diesem Cache zugrunde liegt.
	 *
	 * @return das Netz, welches diesem Cache zugrunde liegt
	 */
	public VerkehrsModellNetz getNetz() {
		return netzCache.getNetz();
	}

	/**
	 * Liefet den StrassenTeilSegmente-Cache zurück, von welchem dieser Cache
	 * abhängig ist.
	 *
	 * @return der StrassenTeilSegmente-Cache, von welchem dieser Cache abhängig ist
	 */
	public StrassenTeilSegmenteCache getStrassenTeilSegmenteCache() {
		return strassenTeilSegmenteCache;
	}
}
