/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2011 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.bitctrl.modell.util.cache;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.bitctrl.util.monitor.IMonitor;
import com.bitctrl.util.monitor.SubMonitor;

import de.bsvrz.sys.funclib.bitctrl.modell.SystemObjekt;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkextlsglobal.objekte.De;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkextlsglobal.objekte.Eak;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkextlsglobal.objekte.SteuerModul;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Stellt die TLS-Hierarchie als Cache zur Verfuegung.
 * 
 * @author BitCtrl Systems GmbH, thierfelder
 * 
 */
public class TlsCache extends AbstractCache {

	private Map<SteuerModul, Set<Eak>> sst2EakMap;

	private Map<SteuerModul, Set<De>> sst2DeMap;

	private Map<Eak, Set<De>> eak2DeMap;

	private Map<De, SteuerModul> de2SstMap;

	private Map<De, Eak> de2EakMap;

	private AbstractDavVerbindungsCache verbindungsCache;

	/**
	 * Standardkonstruktor.
	 * 
	 * @param verbindungsCache z.B.
	 *                         <code>new SimpleDavVerbindungsCache(DefaultObjektFactory.getInstanz());</code>
	 *                         .
	 */
	public TlsCache(final AbstractDavVerbindungsCache verbindungsCache) {
		super("TLS-Cache", verbindungsCache);
	}

	protected void preInit(final AbstractDavVerbindungsCache verbindungsCache1) {
		verbindungsCache = verbindungsCache1;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	@SuppressWarnings({ "unchecked" })
	protected synchronized boolean doInit(final IMonitor monitor) {
		final long start = System.currentTimeMillis();

		final List<SteuerModul> steuerModule = (List<SteuerModul>) verbindungsCache.getObjektFactory()
				.bestimmeModellobjekte(SteuerModul.PID);

		final SubMonitor subMonitor = SubMonitor.convert(monitor);
		subMonitor.beginTask("Initialisiere TLS-Cache ...", steuerModule.size());

		sst2EakMap = new HashMap<>();
		sst2DeMap = new HashMap<>();
		eak2DeMap = new HashMap<>();
		de2SstMap = new HashMap<>();
		de2EakMap = new HashMap<>();

		for (final SystemObjekt de : verbindungsCache.getObjektFactory().bestimmeModellobjekte(De.PID)) {
			de2SstMap.put((De) de, null);
			de2EakMap.put((De) de, null);
		}
		for (final SystemObjekt eak : verbindungsCache.getObjektFactory().bestimmeModellobjekte(Eak.PID)) {
			eak2DeMap.put((Eak) eak, null);
		}

		for (final SteuerModul steuerModul : steuerModule) {
			sst2EakMap.put(steuerModul, new HashSet<>());
			sst2DeMap.put(steuerModul, new HashSet<>());
			for (final Eak eak : steuerModul.getEak()) {
				sst2EakMap.get(steuerModul).add(eak);
				eak2DeMap.put(eak, new HashSet<>());
				for (final De de : eak.getDe()) {
					eak2DeMap.get(eak).add(de);
					sst2DeMap.get(steuerModul).add(de);
					de2EakMap.put(de, eak);
					de2SstMap.put(de, steuerModul);
				}
			}
			subMonitor.worked(1);
		}

		Debug.getLogger().info(
				"Benoetigte Zeit zum Initialisieren des TLS-Caches: " + (System.currentTimeMillis() - start) + " ms");
		return true;
	}

	/**
	 * Erfragt die Menge aller Steuermodule.
	 * 
	 * @return die Menge aller Steuermodule.
	 */
	public final Set<SteuerModul> getSteuerModulSet() {
		ensureInit();
		return sst2EakMap.keySet();
	}

	/**
	 * Erfragt die Menge aller EAK.
	 * 
	 * @return die Menge aller EAK.
	 */
	public final Set<Eak> getEakSet() {
		ensureInit();
		return eak2DeMap.keySet();
	}

	/**
	 * Erfragt die Menge aller DE.
	 * 
	 * @return die Menge aller DE.
	 */
	public final Set<De> getDeSet() {
		ensureInit();
		return de2EakMap.keySet();
	}

	/**
	 * Erfragt die Menge aller EAK, die am uebergebenen Steuermodul referenziert
	 * sind.
	 * 
	 * @param sst ein Steuermodul.
	 * @return eine ggf. leere Menge DE, die am uebergebenen EAK referenziert sind.
	 */
	public final Set<Eak> getEakSet(final SteuerModul sst) {
		ensureInit();
		return sst2EakMap.get(sst);
	}

	/**
	 * Erfragt die Menge der DE, die am uebergebenen EAK referenziert sind.
	 * 
	 * @param eak ein EAK.
	 * @return eine ggf. leere Menge DE, die am uebergebenen EAK referenziert sind.
	 */
	public final Set<De> getDeSet(final Eak eak) {
		ensureInit();
		final Set<De> result = eak2DeMap.get(eak);
		return result == null ? Collections.emptySet() : result;
	}

	/**
	 * Erfragt das Steuermodul, an dem das uebergebene DE konfiguriert ist.
	 * 
	 * @param de ein DE.
	 * @return das Steuermodul, an dem das uebergebene DE konfiguriert ist oder
	 *         <code>null</code>, wenn das DE keinem Steuermodul zugeordnet ist.
	 */
	public final SteuerModul getSteuerModul(final De de) {
		ensureInit();
		return de2SstMap.get(de);
	}

	/**
	 * Erfragt den EAK, an dem das uebergebene DE konfiguriert ist.
	 * 
	 * @param de ein DE.
	 * @return der EAK, an dem das uebergebene DE konfiguriert ist oder
	 *         <code>null</code>, wenn das DE keinem EAK zugeordnet ist.
	 */
	public final Eak getEak(final De de) {
		ensureInit();
		return de2EakMap.get(de);
	}

	/**
	 * Erfragt die Menge der DE, die (indirekt ueber die EAK) am uebergebenen
	 * Steuermodul referenziert sind.
	 * 
	 * @param sst ein Steuermodul.
	 * @return eine ggf. leere Menge DE, die am uebergebenen Steuermodul
	 *         referenziert sind.
	 */
	public final Set<De> getDeSet(final SteuerModul sst) {
		ensureInit();
		return sst2DeMap.get(sst);
	}

}
