/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2011 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.bitctrl.modell.util.cache;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.bitctrl.util.monitor.IMonitor;
import com.bitctrl.util.monitor.SubMonitor;

import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.SystemObjekt;
import de.bsvrz.sys.funclib.bitctrl.modell.metamodellglobal.attribute.AttJaNein;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.attribute.AttRdsTMCRichtung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcglobal.attribute.AttTmcLocationCode;
import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcglobal.konfigurationsdaten.KdTmcLocationCode;
import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcglobal.objekte.TmcGebiet;
import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcglobal.objekte.TmcLinie;
import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcglobal.objekte.TmcLocationCode;
import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcglobal.objekte.TmcPunkt;
import de.bsvrz.sys.funclib.bitctrl.modell.util.rdstmc.TmcGebietWrapper;
import de.bsvrz.sys.funclib.bitctrl.modell.util.rdstmc.TmcLinieWrapper;
import de.bsvrz.sys.funclib.bitctrl.modell.util.rdstmc.TmcLocationCodeWrapper;
import de.bsvrz.sys.funclib.bitctrl.modell.util.rdstmc.TmcPunktWrapper;

/**
 * <p>
 * Diese Hilfsklasse cacht sämtliche für dieses Plug-in benötigten Informationen
 * bzgl. TMC-Location-Codes.
 * </p>
 *
 * @author BitCtrl Systems GmbH, Andreas Meissner
 */
public class TmcLocationCodeCache extends AbstractCache {

	/**
	 * Map, welche zu einem AttTmcLocationCode den dazugehörigen TmcLocationCode
	 * abspeichert.
	 *
	 * @see AttTmcLocationCode
	 * @see TmcLocationCode
	 */
	private Map<AttTmcLocationCode, TmcLocationCode> tmcLocationCodeMap;

	/**
	 * Map, welche zu einem TmcLocationCode (TmcPunkt, TmcLinie oder TmcGebiet) den
	 * dazugehorigen TmcLocationCodeWrapper (TmcPunktWrapper, TmcLinieWrapper oder
	 * TmcGebietWrapper) speichert.
	 */
	private Map<TmcLocationCode, TmcLocationCodeWrapper> tmcLocationCodeWrapperMap;

	/**
	 * Map zur Zuordnung Richtung-&gt;TmcLinie-&gt;TmcLinieWrapper.
	 */
	private Map<AttRdsTMCRichtung, Map<TmcLinie, TmcLinieWrapper>> richtung2Line2LinieWrapperMap;

	/**
	 * Map, welche zu einer TCM-Linie die dazugehörigen Liste der TMC-Punkte
	 * abspeichert.
	 */
	private Map<TmcLinieWrapper, List<TmcPunktWrapper>> linie2PunkteMap;

	/**
	 * Der Verbindungscache.
	 */
	private AbstractDavVerbindungsCache verbindungsCache;

	/**
	 * Der Konstruktor.
	 *
	 * @param verbindungsCache Verbindungscache
	 */
	public TmcLocationCodeCache(final AbstractDavVerbindungsCache verbindungsCache) {
		super(verbindungsCache);
	}

	/**
	 * Vorinitialisierung.
	 *
	 * @param vCache Verbindungscache
	 */
	protected void preInit(final AbstractDavVerbindungsCache vCache) {
		verbindungsCache = vCache;
	}

	@Override
	protected boolean doInit(final IMonitor monitor) {

		tmcLocationCodeMap = new HashMap<>();
		tmcLocationCodeWrapperMap = new HashMap<>();
		richtung2Line2LinieWrapperMap = new HashMap<>();
		linie2PunkteMap = new HashMap<>();

		for (final AttRdsTMCRichtung richtung : AttRdsTMCRichtung.getZustaende()) {
			richtung2Line2LinieWrapperMap.put(richtung, new HashMap<TmcLinie, TmcLinieWrapper>());
		}

		final SubMonitor subMonitor = SubMonitor.convert(monitor);
		subMonitor.beginTask("Initialisiere TMC-Location-Codes", 120000);

		if (intialisiereTmcLocationen(subMonitor.newChild(100000))) {
			if (intialisiereReferenzen(subMonitor.newChild(10000))) {
				return sortiereTmcPunkte(subMonitor.newChild(10000));
			}
		}
		return false;
	}

	/**
	 * Initialisiert die TMC-Location-Codes.
	 *
	 * @param monitor der Monitor
	 *
	 * @return <code>true</code>, wenn die Method erfolgreich ausgeführt wurde,
	 *         ansonsten <code>false</code>
	 */
	@SuppressWarnings("unchecked")
	private boolean intialisiereTmcLocationen(final IMonitor monitor) {

		final ObjektFactory objektFactory = verbindungsCache.getObjektFactory();

		int totalWork = 1000000;
		int work = totalWork / 3;
		final String taskName = "Lade TMC-Location-Codes";
		monitor.setTaskName(taskName);
		monitor.beginTask(taskName, totalWork);

		// (1) Lade SystemObjekte
		monitor.subTask("Lade SystemObjekte");
		final Collection<? extends TmcLocationCode> tmcLocations = (Collection<? extends TmcLocationCode>) objektFactory
				.bestimmeModellobjekte(TmcLocationCode.PID);
		final Collection<SystemObject> tmcLocationSOs = new ArrayList<>(tmcLocations.size());
		for (final SystemObjekt objekt : tmcLocations) {
			tmcLocationSOs.add(objekt.getSystemObject());
		}
		monitor.worked(work);
		totalWork -= work;

		// (2) Lade Daten
		final DataModel dataModel = objektFactory.getDav().getDataModel();
		if (isBenutzeBlockAbruf()) {
			monitor.subTask("Lade Daten");
			dataModel.getConfigurationData(tmcLocationSOs, dataModel.getAttributeGroup(KdTmcLocationCode.PID));
			monitor.worked(work);
		} else {
			int i = 0;
			final int anzahlTmcLocationSO = tmcLocationSOs.size();
			if (anzahlTmcLocationSO == 0) {
				return true;
			}
			final int currentWork = work / anzahlTmcLocationSO;
			for (final SystemObject so : tmcLocationSOs) {
				monitor.subTask(new StringBuffer("Lade Daten ").append(i).append(" von ").append(anzahlTmcLocationSO)
						.toString());
				so.getConfigurationData(dataModel.getAttributeGroup(KdTmcLocationCode.PID));
				monitor.worked(currentWork);
				if (monitor.isCanceled()) {
					monitor.done();
					return false;
				}
				i++;
			}
		}
		totalWork -= work;

		// (3) Initialisiere TMC-Location-Codes
		final int anzahlTmcLocationen = tmcLocations.size();
		if (anzahlTmcLocationen == 0) {
			return true;
		}
		work = totalWork / anzahlTmcLocationen;

		int i = 0;
		for (final TmcLocationCode tmcLocation : tmcLocations) {

			try {

				monitor.subTask(new StringBuffer("Initialisiere TMC-Location-Code ").append(i).append(" von ")
						.append(anzahlTmcLocationen).toString());

				if (tmcLocation instanceof TmcLinie) {
					final TmcLinieWrapper tmcLinieWrapper = new TmcLinieWrapper((TmcLinie) tmcLocation,
							AttRdsTMCRichtung.ZUSTAND_0_POSITIV);
					fuegeZuRichtungLine2LinieWrapperMapHinzu(tmcLinieWrapper);
					speicherTmcLocationCode(tmcLocation, tmcLinieWrapper);
				}

				if (tmcLocation instanceof TmcPunkt) {
					speicherTmcLocationCode(tmcLocation, new TmcPunktWrapper((TmcPunkt) tmcLocation));
				}

				if (tmcLocation instanceof TmcGebiet) {
					speicherTmcLocationCode(tmcLocation, new TmcGebietWrapper((TmcGebiet) tmcLocation));
				}
			} catch (final RuntimeException e) {
				e.printStackTrace();
			} finally {
				monitor.worked(work);
				if (monitor.isCanceled()) {
					monitor.done();
					return false;
				}
				i++;
			}
		}
		return true;
	}

	/**
	 * Initialisiert die Referenzen zwischen den TMC-Location-Codes.
	 *
	 * @param monitor der Monitor
	 *
	 * @return <code>true</code>, wenn die Method erfolgreich ausgeführt wurde,
	 *         ansonsten <code>false</code>
	 */
	private boolean intialisiereReferenzen(final IMonitor monitor) {

		final int totalWork = 1000000;
		final String taskName = "Initialisiere TMC-Referenzen";
		monitor.setTaskName(taskName);
		monitor.beginTask(taskName, totalWork);

		final Collection<TmcLocationCodeWrapper> locationCodeWrapper = tmcLocationCodeWrapperMap.values();
		final int anzahlWrapper = locationCodeWrapper.size();
		if (anzahlWrapper == 0) {
			return true;
		}
		final int work = totalWork / anzahlWrapper;

		final int i = 0;
		for (final TmcLocationCodeWrapper wrapper : locationCodeWrapper) {

			monitor.subTask(new StringBuffer("Initialisiere TMC-Referenzen ").append(i).append(" von ")
					.append(anzahlWrapper).toString());

			if (wrapper instanceof TmcPunktWrapper) {
				final TmcPunktWrapper punktWrapper = (TmcPunktWrapper) wrapper;
				// Ermittle TMC-Linie, auf welcher dieser TMC-Punkt liegt
				final TmcLinie tmcLinie = punktWrapper.getIstTeilvonTmcLinie();
				if (tmcLinie == null) {
					continue;
				}
				ermittelTmcLinien(tmcLinie, punktWrapper);
			}

			monitor.worked(work);
			if (monitor.isCanceled()) {
				monitor.done();
				return false;
			}
		}

		return true;
	}

	/**
	 * Sortiert die TMC-Punkte hinsichtlich der Durchfahrtsrichtung auf der
	 * dazugehörigen TMC-Linie.
	 *
	 * @param monitor der Monitor
	 *
	 * @return <code>true</code>, wenn die Method erfolgreich ausgeführt wurde,
	 *         ansonsten <code>false</code>
	 */
	private boolean sortiereTmcPunkte(final IMonitor monitor) {

		final int totalWork = 1000000;
		final String taskName = "Sortiere TMC-Punkte";
		monitor.setTaskName(taskName);
		monitor.beginTask(taskName, totalWork);

		for (final TmcLinieWrapper linie : linie2PunkteMap.keySet()) {
			final List<TmcPunktWrapper> tempTmcPunkte = linie2PunkteMap.get(linie);
			if ((tempTmcPunkte == null) || tempTmcPunkte.isEmpty()) {
				continue;
			}
			final List<TmcPunktWrapper> tmcPunkte = sortiereTmcPunkte(linie, tempTmcPunkte);
			linie2PunkteMap.put(linie, tmcPunkte);
			if (monitor.isCanceled()) {
				monitor.done();
				return false;
			}
		}
		return true;
	}

	/**
	 * Sortiert die gegebene Liste von TMC-Punkten abhängig von der
	 * Straße(nrichtung), auf welcher sich diese befinden.
	 *
	 * @param strasse   die Straße (TMC-Linie), auf welcher sich die TMC-Punkte
	 *                  befinden
	 * @param tmcPunkte die zu sortierende Liste von TMC-Punkten
	 *
	 * @return die sortierte List von TMC-Punkten
	 */
	private List<TmcPunktWrapper> sortiereTmcPunkte(final TmcLinieWrapper strasse,
			final List<TmcPunktWrapper> tmcPunkte) {
		final List<TmcPunktWrapper> ergebnis = new ArrayList<>();

		if (tmcPunkte.size() == 0) {
			return ergebnis;
		}

		final TmcPunktWrapper startPunkt = tmcPunkte.get(0);

		final List<TmcPunktWrapper> nachfolger = getNachfolgendeTmcPunktWrapperInternal(startPunkt,
				strasse.getRichtung(), false);
		final List<TmcPunktWrapper> vorgaenger = getVorgaengerTmcPunktWrapperInternal(startPunkt, strasse.getRichtung(),
				false);

		Collections.reverse(vorgaenger);
		ergebnis.addAll(vorgaenger);
		ergebnis.add(startPunkt);
		ergebnis.addAll(nachfolger);

		return ergebnis;
	}

	/**
	 * Ermittelt startend mit einer gegebenen TMC-Linie rekursiv sämtliche
	 * übergeordneten TMC-Linien und fügt den gegebenen TMC-Punkt diesen TMC-Linien
	 * hinzu.
	 *
	 * @param tmcLinie die TMC-Linie
	 * @param tmcPunkt der TMC-Punkt
	 */
	private void ermittelTmcLinien(final TmcLinie tmcLinie, final TmcPunktWrapper tmcPunkt) {

		// Weise TmcPunkt der dazugehörigen Straße zu
		if (tmcPunkt.getVorhandenPositiveRichtung() == AttJaNein.ZUSTAND_1_JA) {
			fuegeZuStrassenMapHinzu(tmcPunkt, AttRdsTMCRichtung.ZUSTAND_0_POSITIV, tmcLinie);
		}
		if (tmcPunkt.getVorhandenNegativeRichtung() == AttJaNein.ZUSTAND_1_JA) {
			fuegeZuStrassenMapHinzu(tmcPunkt, AttRdsTMCRichtung.ZUSTAND_1_NEGATIV, tmcLinie);
		}
		final TmcLinieWrapper tmcLinieWrapper = (TmcLinieWrapper) tmcLocationCodeWrapperMap.get(tmcLinie);

		if (tmcLinieWrapper.getIstTeilvonTmcLinie() != null) {
			ermittelTmcLinien(tmcLinieWrapper.getIstTeilvonTmcLinie(), tmcPunkt);
		}
		if (tmcLinieWrapper.getIstTeilvonTmcGebiet() != null) {
			ermittelTmcGebiete(tmcLinieWrapper.getIstTeilvonTmcGebiet());
		}
	}

	/**
	 * Ermittelt startend mit einem gegebenen TMC-Gebiet rekursiv sämtliche
	 * übergeordneten TMC-Gebiete.
	 *
	 * @param tmcGebiet das TMC-Gebiet
	 */
	private void ermittelTmcGebiete(final TmcGebiet tmcGebiet) {
		TmcGebietWrapper tmcGebietWrapper = (TmcGebietWrapper) tmcLocationCodeWrapperMap.get(tmcGebiet);
		if (tmcGebietWrapper != null) {
			return;
		}
		tmcGebietWrapper = new TmcGebietWrapper(tmcGebiet);
		speicherTmcLocationCode(tmcGebiet, tmcGebietWrapper);
		if (tmcGebietWrapper.getIstTeilvonTmcGebiet() != null) {
			ermittelTmcGebiete(tmcGebietWrapper.getIstTeilvonTmcGebiet());
		}
	}

	/**
	 * Speichert den gegebenen TMC-LocationCode und dazugehörigen Wrapper in den
	 * entsprechenden Maps ab.
	 *
	 * @param tmcLocationCode der TMC-LocationCode
	 * @param wrapper         der dazugehörige Wrapper
	 */
	private void speicherTmcLocationCode(final TmcLocationCode tmcLocationCode, final TmcLocationCodeWrapper wrapper) {
		tmcLocationCodeWrapperMap.put(tmcLocationCode, wrapper);
		final AttTmcLocationCode attTmcLocationCode = wrapper.getAttTmcLocationCode();
		if (attTmcLocationCode != null) {
			tmcLocationCodeMap.put(attTmcLocationCode, tmcLocationCode);
		}
	}

	/**
	 * Fügt den gegebenen TmcLinieWrapper der entsprechenden Map hinzu und
	 * initialisiert die {@link #linie2PunkteMap}.
	 *
	 * @param wrapper der TmcLinieWrapper
	 */
	private void fuegeZuRichtungLine2LinieWrapperMapHinzu(final TmcLinieWrapper wrapper) {
		richtung2Line2LinieWrapperMap.get(wrapper.getRichtung()).put(wrapper.getTmcLinie(), wrapper);
		linie2PunkteMap.put(wrapper, new ArrayList<TmcPunktWrapper>());
	}

	/**
	 * Fügt den gegebenen TMC-Punkt den entsprechenden Maps hinzu.
	 *
	 * @param tmcPunkt der TMC-Punkt
	 * @param richtung die Straßenrichtung der Straße, auf welcher der gegebene
	 *                 Punkt liegt
	 * @param tmcLinie die Straße (bzw. TMC-Linie), auf welcher der gegebene Punkt
	 *                 liegt
	 */
	private void fuegeZuStrassenMapHinzu(final TmcPunktWrapper tmcPunkt, final AttRdsTMCRichtung richtung,
			final TmcLinie tmcLinie) {

		final Map<TmcLinie, TmcLinieWrapper> map = richtung2Line2LinieWrapperMap.get(richtung);

		TmcLinieWrapper tmcLinieWrapper = map.get(tmcLinie);
		if (tmcLinieWrapper == null) {
			tmcLinieWrapper = new TmcLinieWrapper(tmcLinie, richtung);
			fuegeZuRichtungLine2LinieWrapperMapHinzu(tmcLinieWrapper);
			speicherTmcLocationCode(tmcLinie, tmcLinieWrapper);
		}

		List<TmcPunktWrapper> punkte = linie2PunkteMap.get(tmcLinieWrapper);
		if (punkte == null) {
			punkte = new ArrayList<>();
			linie2PunkteMap.put(tmcLinieWrapper, punkte);
		}

		punkte.add(tmcPunkt);
	}

	/**
	 * Gibt alle TmcLinien zur&uuml;ck.
	 *
	 * @return Liste aller TmcLinien
	 */
	public List<TmcLinieWrapper> getTmcLinien() {
		return new ArrayList<>(linie2PunkteMap.keySet());
	}

	/**
	 * Liefert sämtliche TmcLinien abhängig von dem gegebenen Filter zurück.
	 *
	 * @param filter der Filter
	 *
	 * @return sämtliche TmcLinien abhängig von dem gegebenen Filters
	 */
	public List<TmcLinieWrapper> getTmcLinien(final TmcLinienFilter filter) {
		final List<TmcLinieWrapper> gefilterteLinien = new ArrayList<>();
		for (final TmcLinieWrapper l : linie2PunkteMap.keySet()) {
			if (filter.passiertFilter(l)) {
				gefilterteLinien.add(l);
			}
		}
		return gefilterteLinien;
	}

	/**
	 * Liefert zu einem gegebenen AttTmcLocationCode den dazugehörigen
	 * TmcLocationCode oder <code>null</code> falls kein dazugehöriger
	 * TmcLocationCode gefunden wurde.
	 *
	 * @param attTmcLocationCode der AttTmcLocationCode, darf nicht
	 *                           <code>null</code> sein
	 *
	 * @return zu einem gegebenen AttTmcLocationCode den dazugehörigen
	 *         TmcLocationCode, kann <code>null</code> sein
	 *
	 * @see AttTmcLocationCode
	 * @see TmcLocationCode
	 */
	public TmcLocationCode getTmcLocationCode(final AttTmcLocationCode attTmcLocationCode) {
		ensureInit();
		assert attTmcLocationCode != null;
		if (tmcLocationCodeMap != null) {
			return tmcLocationCodeMap.get(attTmcLocationCode);
		}
		return null;
	}

	/**
	 * Liefert zu einem gegebenen {@link TmcLocationCode} den dazugehörigen
	 * {@link TmcLocationCodeWrapper} zurück.
	 *
	 * @param tmcLocationCode der {@link TmcLocationCode}, darf nicht
	 *                        <code>null</code> sein
	 *
	 * @return zu einem gegebenen {@link TmcLocationCode} den dazugehörigen
	 *         {@link TmcLocationCodeWrapper}, kann <code>null</code> sein
	 */
	private TmcLocationCodeWrapper getTmcLocationCodeWrapperInternal(final TmcLocationCode tmcLocationCode) {
		assert tmcLocationCode != null;
		if (tmcLocationCodeWrapperMap != null) {
			return tmcLocationCodeWrapperMap.get(tmcLocationCode);
		}
		return null;
	}

	/**
	 * Liefert zu einem gegebenen {@link TmcPunkt} den dazugehörigen
	 * {@link TmcPunktWrapper} zurück.
	 *
	 * @param tmcPunkt der {@link TmcPunkt}, darf nicht <code>null</code> sein
	 *
	 * @return zu einem gegebenen {@link TmcPunkt} den dazugehörigen
	 *         {@link TmcPunktWrapper}, kann <code>null</code> sein
	 */
	public TmcPunktWrapper getTmcPunktWrapper(final TmcPunkt tmcPunkt) {
		ensureInit();
		final TmcLocationCodeWrapper tmcLocationCodeWrapper = getTmcLocationCodeWrapperInternal(tmcPunkt);
		if (tmcLocationCodeWrapper instanceof TmcPunktWrapper) {
			return (TmcPunktWrapper) tmcLocationCodeWrapper;
		}
		return null;
	}

	/**
	 * Liefert zu einer gegebenen {@link TmcLinie} und {@link AttRdsTMCRichtung} den
	 * dazugehörigen {@link TmcLinieWrapper} zurück.
	 *
	 * @param tmcLinie der {@link TmcLinie}, darf nicht <code>null</code> sein
	 * @param richtung die {@link AttRdsTMCRichtung}, darf nicht <code>null</code>
	 *                 sein
	 *
	 * @return zu einer gegebenen {@link TmcLinie} und {@link AttRdsTMCRichtung} den
	 *         dazugehörigen {@link TmcLinieWrapper}, kann <code>null</code> sein,
	 *         kann <code>null</code> sein
	 */
	public TmcLinieWrapper getTmcLinieWrapper(final TmcLinie tmcLinie, final AttRdsTMCRichtung richtung) {
		ensureInit();
		final Map<TmcLinie, TmcLinieWrapper> map = richtung2Line2LinieWrapperMap.get(richtung);
		if (map != null) {
			return map.get(tmcLinie);
		}
		return null;
	}

	/**
	 * Liefert zu einem gegebenen {@link TmcGebiet} den dazugehörigen
	 * {@link TmcGebietWrapper} zurück.
	 *
	 * @param tmcGebiet der {@link TmcGebiet}, darf nicht <code>null</code> sein
	 *
	 * @return zu einem gegebenen {@link TmcGebiet} den dazugehörigen
	 *         {@link TmcGebietWrapper}, kann <code>null</code> sein
	 */
	public TmcGebietWrapper getTmcGebietWrapper(final TmcGebiet tmcGebiet) {
		ensureInit();
		final TmcLocationCodeWrapper tmcLocationCodeWrapper = getTmcLocationCodeWrapperInternal(tmcGebiet);
		if (tmcLocationCodeWrapper instanceof TmcGebietWrapper) {
			return (TmcGebietWrapper) tmcLocationCodeWrapper;
		}
		return null;
	}

	/**
	 * Liefert zu einem gegebenen AttTmcLocationCode den dazugehörigen
	 * TmcLocationCodeWrapper zurück.
	 *
	 * @param attTmcLocationCode der AttTmcLocationCode, darf nicht
	 *                           <code>null</code> sein
	 *
	 * @return zu einem gegebenen AttTmcLocationCode den dazugehörigen
	 *         TmcLocationCodeWrapper, kann <code>null</code> sein
	 *
	 * @see AttTmcLocationCode
	 * @see TmcLocationCodeWrapper
	 */
	public TmcPunktWrapper getTmcPunktWrapper(final AttTmcLocationCode attTmcLocationCode) {
		ensureInit();
		assert attTmcLocationCode != null;
		if (tmcLocationCodeMap != null) {
			final TmcLocationCode tmcLocationCode = tmcLocationCodeMap.get(attTmcLocationCode);
			if ((tmcLocationCode != null) && (tmcLocationCode instanceof TmcPunkt)) {
				return getTmcPunktWrapper((TmcPunkt) tmcLocationCode);
			}
		}
		return null;
	}

	/**
	 * Gibt eine geordnete Liste von TMC-Punkten zurück, welche dem gegebenen
	 * TMC-Punkt (auf der gegebenen Straße) nachfolgen.
	 *
	 * @param tmcPunktWrapper                             der TMC-Punkt, darf nicht
	 *                                                    <code>null</code> sein
	 * @param richtung                                    die Richtung, darf nicht
	 *                                                    <code>null</code> sein
	 * @param beruecksichtigeWeiterfuehrendeStrassenTeile <code>true</code>, wenn
	 *                                                    auch TMC-Punkte
	 *                                                    mitberücksichtigt werden
	 *                                                    sollen, welche auf
	 *                                                    weiterführenden
	 *                                                    Strassenteilen mit
	 *                                                    derselben Straßennummer
	 *                                                    liegen, ansonsten
	 *                                                    <code>false</code>
	 *
	 * @return eine geordnete Liste von TMC-Punkten, welche dem gegebenen TMC-Punkt
	 *         nachfolgen, niemals <code>null</code>
	 */
	public List<TmcPunktWrapper> getNachfolgendeTmcPunktWrapper(final TmcPunktWrapper tmcPunktWrapper,
			final AttRdsTMCRichtung richtung, final boolean beruecksichtigeWeiterfuehrendeStrassenTeile) {
		ensureInit();
		return getNachfolgendeTmcPunktWrapperInternal(tmcPunktWrapper, richtung,
				beruecksichtigeWeiterfuehrendeStrassenTeile);
	}

	/**
	 * Gibt eine geordnete Liste von TMC-Punkten zurück, welche dem gegebenen
	 * TMC-Punkt (auf der gegebenen Straße) nachfolgen.
	 *
	 * <p>
	 * Achtung: Methode zur internen Benutzung (ohne vorangestellten Methodenaufruf
	 * {@link #ensureInit()}.
	 * </p>
	 *
	 * @param tmcPunktWrapper                             der TMC-Punkt, darf nicht
	 *                                                    <code>null</code> sein
	 * @param richtung                                    die Richtung, darf nicht
	 *                                                    <code>null</code> sein
	 * @param beruecksichtigeWeiterfuehrendeStrassenTeile <code>true</code>, wenn
	 *                                                    auch TMC-Punkte
	 *                                                    mitberücksichtigt werden
	 *                                                    sollen, welche auf
	 *                                                    weiterführenden
	 *                                                    Strassenteilen mit
	 *                                                    derselben Straßennummer
	 *                                                    liegen, ansonsten
	 *                                                    <code>false</code>
	 *
	 * @return eine geordnete Liste von TMC-Punkten, welche dem gegebenen TMC-Punkt
	 *         nachfolgen, niemals <code>null</code>
	 */
	private List<TmcPunktWrapper> getNachfolgendeTmcPunktWrapperInternal(final TmcPunktWrapper tmcPunktWrapper,
			final AttRdsTMCRichtung richtung, final boolean beruecksichtigeWeiterfuehrendeStrassenTeile) {
		return getNachfolgendeTmcPunktWrapperInternal(tmcPunktWrapper, richtung, Integer.MAX_VALUE,
				beruecksichtigeWeiterfuehrendeStrassenTeile);
	}

	/**
	 * Gibt eine geordnete Liste von TMC-Punkten zurück, welche dem gegebenen
	 * TMC-Punkt (auf der gegebenen Straße) nachfolgen.
	 *
	 * @param tmcPunktWrapper                             der TMC-Punkt, darf nicht
	 *                                                    <code>null</code> sein
	 * @param richtung                                    die Richtung, darf nicht
	 *                                                    <code>null</code> sein
	 * @param maximaleAnzahl                              die maximale Anzahl der
	 *                                                    Nachfolger-TMC-Punkte
	 * @param beruecksichtigeWeiterfuehrendeStrassenTeile <code>true</code>, wenn
	 *                                                    auch TMC-Punkte
	 *                                                    mitberücksichtigt werden
	 *                                                    sollen, welche auf
	 *                                                    weiterführenden
	 *                                                    Strassenteilen mit
	 *                                                    derselben Straßennummer
	 *                                                    liegen, ansonsten
	 *                                                    <code>false</code>
	 *
	 * @return eine geordnete Liste von TMC-Punkten, welche dem gegebenen TMC-Punkt
	 *         nachfolgen, niemals <code>null</code>
	 */
	public List<TmcPunktWrapper> getNachfolgendeTmcPunktWrapper(final TmcPunktWrapper tmcPunktWrapper,
			final AttRdsTMCRichtung richtung, final int maximaleAnzahl,
			final boolean beruecksichtigeWeiterfuehrendeStrassenTeile) {
		ensureInit();
		return getNachfolgendeTmcPunktWrapperInternal(tmcPunktWrapper, richtung, maximaleAnzahl,
				beruecksichtigeWeiterfuehrendeStrassenTeile);
	}

	/**
	 * Gibt eine geordnete Liste von TMC-Punkten zurück, welche dem gegebenen
	 * TMC-Punkt (auf der gegebenen Straße) nachfolgen.
	 *
	 * <p>
	 * Achtung: Methode zur internen Benutzung (ohne vorangestellten Methodenaufruf
	 * {@link #ensureInit()}.
	 * </p>
	 *
	 * @param tmcPunktWrapper                             der TMC-Punkt, darf nicht
	 *                                                    <code>null</code> sein
	 * @param richtung                                    die Richtung, darf nicht
	 *                                                    <code>null</code> sein
	 * @param maximaleAnzahl                              die maximale Anzahl der
	 *                                                    Nachfolger-TMC-Punkte
	 * @param beruecksichtigeWeiterfuehrendeStrassenTeile <code>true</code>, wenn
	 *                                                    auch TMC-Punkte
	 *                                                    mitberücksichtigt werden
	 *                                                    sollen, welche auf
	 *                                                    weiterführenden
	 *                                                    Strassenteilen mit
	 *                                                    derselben Straßennummer
	 *                                                    liegen, ansonsten
	 *                                                    <code>false</code>
	 *
	 * @return eine geordnete Liste von TMC-Punkten, welche dem gegebenen TMC-Punkt
	 *         nachfolgen, niemals <code>null</code>
	 */
	private List<TmcPunktWrapper> getNachfolgendeTmcPunktWrapperInternal(final TmcPunktWrapper tmcPunktWrapper,
			final AttRdsTMCRichtung richtung, final int maximaleAnzahl,
			final boolean beruecksichtigeWeiterfuehrendeStrassenTeile) {

		assert tmcPunktWrapper != null;
		assert richtung != null;

		TmcLinie tmcLinie = tmcPunktWrapper.getIstTeilvonTmcLinie();

		final List<TmcPunktWrapper> ergebnis = new ArrayList<>();

		TmcPunktWrapper wrapper = tmcPunktWrapper;
		int i = 1;
		while ((i <= maximaleAnzahl) && (wrapper != null)) {
			final TmcPunkt nachfolger = (richtung.getValue() == AttRdsTMCRichtung.ZUSTAND_0_POSITIV.getValue())
					? wrapper.getNachfolger()
					: wrapper.getVorgaenger();
			wrapper = null;
			if (nachfolger != null) {
				wrapper = (TmcPunktWrapper) getTmcLocationCodeWrapperInternal(nachfolger);
			}

			// Überprüfe Straße
			if (wrapper != null) {
				if (wrapper == tmcPunktWrapper) {
					// System.err.println("TmcPunkt-Loop detected: "
					// + tmcPunktWrapper.getAttTmcLocationCode());
					break;
				}
				final TmcLinie nachfolgerLinie = wrapper.getIstTeilvonTmcLinie();

				if (beruecksichtigeWeiterfuehrendeStrassenTeile) {
					if (nachfolgerLinie != tmcLinie) {
						final TmcLinie nachfolgerLinieNachfolger = tmcLinie.getKdTmcLinie().getDatum().getNachfolger();
						if ((nachfolgerLinieNachfolger != null) && (nachfolgerLinie == nachfolgerLinieNachfolger)) {
							tmcLinie = nachfolgerLinieNachfolger;
						} else {
							break;
						}
					}
				} else {
					if (nachfolgerLinie != tmcLinie) {
						break;
					}
				}
			}

			if ((wrapper != null) && (((richtung.getValue() == AttRdsTMCRichtung.ZUSTAND_0_POSITIV.getValue())
					&& (wrapper.getVorhandenPositiveRichtung() == AttJaNein.ZUSTAND_1_JA))
					|| ((richtung.getValue() == AttRdsTMCRichtung.ZUSTAND_1_NEGATIV.getValue())
							&& (wrapper.getVorhandenNegativeRichtung() == AttJaNein.ZUSTAND_1_JA)))) {
				ergebnis.add(wrapper);
				i++;
			}
		}
		return ergebnis;
	}

	/**
	 * Gibt eine geordnete Liste von TMC-Punkten zurück, welche dem gegebenen
	 * TMC-Punkt (auf der gegebenen Straße) vorangestellt sind.
	 *
	 * @param tmcPunktWrapper                             der TMC-Punkt, darf nicht
	 *                                                    <code>null</code> sein
	 * @param richtung                                    die Richtung, darf nicht
	 *                                                    <code>null</code> sein
	 * @param beruecksichtigeWeiterfuehrendeStrassenTeile <code>true</code>, wenn
	 *                                                    auch TMC-Punkte
	 *                                                    mitberücksichtigt werden
	 *                                                    sollen, welche auf
	 *                                                    weiterführenden
	 *                                                    Strassenteilen mit
	 *                                                    derselben Straßennummer
	 *                                                    liegen, ansonsten
	 *                                                    <code>false</code>
	 *
	 * @return eine geordnete Liste von TMC-Punkten, welche dem gegebenen TMC-Punkt
	 *         vorangestellt sind, niemals <code>null</code>
	 */
	public List<TmcPunktWrapper> getVorgaengerTmcPunktWrapper(final TmcPunktWrapper tmcPunktWrapper,
			final AttRdsTMCRichtung richtung, final boolean beruecksichtigeWeiterfuehrendeStrassenTeile) {
		ensureInit();
		return getVorgaengerTmcPunktWrapperInternal(tmcPunktWrapper, richtung,
				beruecksichtigeWeiterfuehrendeStrassenTeile);
	}

	/**
	 * Gibt eine geordnete Liste von TMC-Punkten zurück, welche dem gegebenen
	 * TMC-Punkt (auf der gegebenen Straße) vorangestellt sind.
	 *
	 * <p>
	 * Achtung: Methode zur internen Benutzung (ohne vorangestellten Methodenaufruf
	 * {@link #ensureInit()}.
	 * </p>
	 *
	 * @param tmcPunktWrapper                             der TMC-Punkt, darf nicht
	 *                                                    <code>null</code> sein
	 * @param richtung                                    die Richtung, darf nicht
	 *                                                    <code>null</code> sein
	 * @param beruecksichtigeWeiterfuehrendeStrassenTeile <code>true</code>, wenn
	 *                                                    auch TMC-Punkte
	 *                                                    mitberücksichtigt werden
	 *                                                    sollen, welche auf
	 *                                                    weiterführenden
	 *                                                    Strassenteilen mit
	 *                                                    derselben Straßennummer
	 *                                                    liegen, ansonsten
	 *                                                    <code>false</code>
	 *
	 * @return eine geordnete Liste von TMC-Punkten, welche dem gegebenen TMC-Punkt
	 *         vorangestellt sind, niemals <code>null</code>
	 */
	private List<TmcPunktWrapper> getVorgaengerTmcPunktWrapperInternal(final TmcPunktWrapper tmcPunktWrapper,
			final AttRdsTMCRichtung richtung, final boolean beruecksichtigeWeiterfuehrendeStrassenTeile) {
		return getVorgaengerTmcPunktWrapperInternal(tmcPunktWrapper, richtung, Integer.MAX_VALUE,
				beruecksichtigeWeiterfuehrendeStrassenTeile);
	}

	/**
	 * Gibt eine geordnete Liste von TMC-Punkten zurück, welche dem gegebenen
	 * TMC-Punkt (auf der gegebenen Straße) vorangestellt sind.
	 *
	 * @param tmcPunktWrapper                             der TMC-Punkt, darf nicht
	 *                                                    <code>null</code> sein
	 * @param richtung                                    die Richtung, darf nicht
	 *                                                    <code>null</code> sein
	 * @param maximaleAnzahl                              die maximale Anzahl der
	 *                                                    Vorgänger-TMC-Punkte
	 * @param beruecksichtigeWeiterfuehrendeStrassenTeile <code>true</code>, wenn
	 *                                                    auch TMC-Punkte
	 *                                                    mitberücksichtigt werden
	 *                                                    sollen, welche auf
	 *                                                    weiterführenden
	 *                                                    Strassenteilen mit
	 *                                                    derselben Straßennummer
	 *                                                    liegen, ansonsten
	 *                                                    <code>false</code>
	 *
	 * @return eine geordnete Liste von TMC-Punkten, welche dem gegebenen TMC-Punkt
	 *         vorangestellt sind, niemals <code>null</code>
	 */
	public List<TmcPunktWrapper> getVorgaengerTmcPunktWrapper(final TmcPunktWrapper tmcPunktWrapper,
			final AttRdsTMCRichtung richtung, final int maximaleAnzahl,
			final boolean beruecksichtigeWeiterfuehrendeStrassenTeile) {
		ensureInit();
		return getVorgaengerTmcPunktWrapperInternal(tmcPunktWrapper, richtung, maximaleAnzahl,
				beruecksichtigeWeiterfuehrendeStrassenTeile);
	}

	/**
	 * Gibt eine geordnete Liste von TMC-Punkten zurück, welche dem gegebenen
	 * TMC-Punkt (auf der gegebenen Straße) vorangestellt sind.
	 *
	 * <p>
	 * Achtung: Methode zur internen Benutzung (ohne vorangestellten Methodenaufruf
	 * {@link #ensureInit()}.
	 * </p>
	 *
	 * @param tmcPunktWrapper                             der TMC-Punkt, darf nicht
	 *                                                    <code>null</code> sein
	 * @param richtung                                    die Richtung, darf nicht
	 *                                                    <code>null</code> sein
	 * @param maximaleAnzahl                              die maximale Anzahl der
	 *                                                    Vorgänger-TMC-Punkte
	 * @param beruecksichtigeWeiterfuehrendeStrassenTeile <code>true</code>, wenn
	 *                                                    auch TMC-Punkte
	 *                                                    mitberücksichtigt werden
	 *                                                    sollen, welche auf
	 *                                                    weiterführenden
	 *                                                    Strassenteilen mit
	 *                                                    derselben Straßennummer
	 *                                                    liegen, ansonsten
	 *                                                    <code>false</code>
	 *
	 * @return eine geordnete Liste von TMC-Punkten, welche dem gegebenen TMC-Punkt
	 *         vorangestellt sind, niemals <code>null</code>
	 */
	private List<TmcPunktWrapper> getVorgaengerTmcPunktWrapperInternal(final TmcPunktWrapper tmcPunktWrapper,
			final AttRdsTMCRichtung richtung, final int maximaleAnzahl,
			final boolean beruecksichtigeWeiterfuehrendeStrassenTeile) {

		assert tmcPunktWrapper != null;
		assert richtung != null;

		TmcLinie tmcLinie = tmcPunktWrapper.getIstTeilvonTmcLinie();

		final List<TmcPunktWrapper> ergebnis = new ArrayList<>();

		TmcPunktWrapper wrapper = tmcPunktWrapper;
		int i = 1;
		while ((i <= maximaleAnzahl) && (wrapper != null)) {
			final TmcPunkt vorgaenger = (richtung.getValue() == AttRdsTMCRichtung.ZUSTAND_0_POSITIV.getValue())
					? wrapper.getVorgaenger()
					: wrapper.getNachfolger();
			wrapper = null;
			if (vorgaenger != null) {
				wrapper = (TmcPunktWrapper) getTmcLocationCodeWrapperInternal(vorgaenger);
			}

			// Überprüfe Straße
			if (wrapper != null) {
				if (wrapper == tmcPunktWrapper) {
					// System.err.println("TmcPunkt-Loop detected: "
					// + tmcPunktWrapper.getAttTmcLocationCode());
					break;
				}
				final TmcLinie vorgaengerLinie = wrapper.getIstTeilvonTmcLinie();

				if (beruecksichtigeWeiterfuehrendeStrassenTeile) {
					if (vorgaengerLinie != tmcLinie) {
						final TmcLinie vorgaengerLinieVorgaenger = tmcLinie.getKdTmcLinie().getDatum().getVorgaenger();
						if ((vorgaengerLinieVorgaenger != null) && (vorgaengerLinie == vorgaengerLinieVorgaenger)) {
							tmcLinie = vorgaengerLinieVorgaenger;
						} else {
							break;
						}
					}
				} else {
					if (vorgaengerLinie != tmcLinie) {
						break;
					}
				}
			}

			if ((wrapper != null) && (((richtung.getValue() == AttRdsTMCRichtung.ZUSTAND_0_POSITIV.getValue())
					&& (wrapper.getVorhandenPositiveRichtung() == AttJaNein.ZUSTAND_1_JA))
					|| ((richtung.getValue() == AttRdsTMCRichtung.ZUSTAND_1_NEGATIV.getValue())
							&& (wrapper.getVorhandenNegativeRichtung() == AttJaNein.ZUSTAND_1_JA)))) {
				ergebnis.add(wrapper);
				i++;
			}
		}
		return ergebnis;
	}

	/**
	 * Liefert die auf der gegebenen TMC-Linie liegenden TMC-Punkte zurück.
	 *
	 * <p>
	 * Beachte: Die zurückgegebene Liste der TMC-Punkte ist bereits nach
	 * Durchfahrtrichtung sortiert.
	 * </p>
	 *
	 * @param tmcLinie die TMC-Linie
	 *
	 * @return die auf der gegebenen TMC-Linie liegenden TMC-Punkte
	 */
	public List<TmcPunktWrapper> getTmcPunktWrapper(final TmcLinieWrapper tmcLinie) {
		return linie2PunkteMap.get(tmcLinie);
	}
}
