/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2011 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.bitctrl.modell.util.geolib;

import java.util.ArrayList;
import java.util.List;

import de.bsvrz.sys.funclib.bitctrl.geolib.WGS84Koordinate;
import de.bsvrz.sys.funclib.bitctrl.geolib.WGS84Polygon;
import de.bsvrz.sys.funclib.bitctrl.geolib.WGS84Punkt;
import de.bsvrz.sys.funclib.bitctrl.modell.att.Feld;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.attribute.AttWgs84Breite;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.attribute.AttWgs84Laenge;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.konfigurationsdaten.KdBestehtAusLinienObjekten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.konfigurationsdaten.KdFlaechenKoordinaten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.konfigurationsdaten.KdLinienKoordinaten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.konfigurationsdaten.KdPunktKoordinaten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.konfigurationsdaten.KdPunktLiegtAufLinienObjekt;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.objekte.BestehtAusLinienObjekten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.objekte.Flaeche;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.objekte.FlaecheXY;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.objekte.Linie;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.objekte.LinieXY;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.objekte.Punkt;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.objekte.PunktLiegtAufLinienObjekt;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.objekte.PunktXY;

/**
 * Macht die geolib aus funclib.bitctrl für die Modell-Objekte nutzbar. Diese
 * Klasse arbeitet nur mit den WGS84-Koordinaten.
 * 
 * @author BitCtrl Systems GmbH, Falko Schumann
 * @see UTMUtil
 */
public final class WGS84Util {

	/**
	 * Bestimmt die WGS84-Koordinaten eines Punkts. Erweitert ein Objekt sowohl
	 * <code>PunktXY</code> als auch <code>PunktLiegtAufLinienObjekt</code>, wird
	 * zuerst versucht die Koordinaten über <code>PunktLiegtAufLinienObjekt</code>
	 * zu bestimmen, ist das nicht möglich, wird versucht sie über
	 * <code>PunktXY</code> zu bestimmen.
	 * 
	 * @param punkt der Punkt, dessen Koordinaten gesucht sind.
	 * @return die Koordinaten des Punkts.
	 */
	public static WGS84Koordinate konvertiere(final Punkt punkt) {
		return konvertiereIntern(punkt);
	}

	static WGS84Punkt konvertiereIntern(final Punkt punkt) {
		if (punkt instanceof PunktLiegtAufLinienObjekt) {
			try {
				return konvertiereIntern((PunktLiegtAufLinienObjekt) punkt);
			} catch (final RuntimeException ex) {
				if (punkt instanceof PunktXY) {
					// Rückfallebene, Bestimmung mit PunktXY
					return konvertiereIntern((PunktXY) punkt);
				}
				// Keine Rückfallebene vorhanden
				throw ex;
			}
		}
		if (punkt instanceof PunktXY) {
			return konvertiereIntern((PunktXY) punkt);
		}
		throw new IllegalArgumentException(
				"Der Punkt ist weder vom Typ PunktXY noch vom Typ PunktLiegtAufLinienObjekt: " + punkt);
	}

	static WGS84Punkt konvertiereIntern(final PunktXY punkt) {
		final KdPunktKoordinaten.Daten datum = punkt.getKdPunktKoordinaten().getDatum();
		if (datum == null) {
			throw new IllegalArgumentException("Für den Punkt wurden keine Koordinaten konfiguriert: " + punkt);
		}
		return new WGS84Punkt(datum.getX().doubleValue(), datum.getY().doubleValue());
	}

	static WGS84Punkt konvertiereIntern(final PunktLiegtAufLinienObjekt punkt) {
		final KdPunktLiegtAufLinienObjekt.Daten datum = punkt.getKdPunktLiegtAufLinienObjekt().getDatum();
		if (datum == null) {
			throw new IllegalArgumentException(
					"Für den Punkt wurden keine Linienreferenz und kein Offset konfiguriert: " + punkt);
		}

		final Linie linie = datum.getLinienReferenz();
		final WGS84Polygon polygon = konvertiereIntern(linie);

		final double offset = datum.getOffset().doubleValue();
		return polygon.bildPunkt(offset);
	}

	/**
	 * Bestimmt die Liste der Koordinaten für eine Linie.
	 * 
	 * @param linie die Linie, deren Koordinaten gesucht sind.
	 * @return die Liste der Koordinaten der Linie.
	 */
	public static List<WGS84Koordinate> konvertiere(final Linie linie) {
		final WGS84Polygon wgs84 = konvertiereIntern(linie);
		return (List) wgs84.getKoordinaten();
	}

	static WGS84Polygon konvertiereIntern(final Linie linie) {
		if (linie instanceof LinieXY) {
			return konvertiereIntern((LinieXY) linie);
		} else if (linie instanceof BestehtAusLinienObjekten) {
			return konvertiereIntern((BestehtAusLinienObjekten) linie);
		} else {
			throw new IllegalArgumentException(
					"Die Linie ist weder vom Typ LinieXY noch vom Typ BestehtAusLinienObjekten: " + linie);
		}
	}

	static WGS84Polygon konvertiereIntern(final LinieXY linie) {
		final KdLinienKoordinaten.Daten datum = linie.getKdLinienKoordinaten().getDatum();
		if (datum == null) {
			throw new IllegalArgumentException("Für die Linie wurden keine Koordinaten konfiguriert: " + linie);
		}

		final Feld<AttWgs84Laenge> feldX = datum.getX();
		final Feld<AttWgs84Breite> feldY = datum.getY();
		if (feldX.size() != feldY.size()) {
			throw new IllegalStateException(
					"Die Längen der Felder für x- und y-Koordinaten sind unterschiedlich: " + linie);
		}

		final List<WGS84Punkt> punkte = new ArrayList<>();
		for (int i = 0; i < feldX.size(); ++i) {
			punkte.add(new WGS84Punkt(feldX.get(i).doubleValue(), feldY.get(i).doubleValue()));
		}

		return new WGS84Polygon(punkte);
	}

	static WGS84Polygon konvertiereIntern(final BestehtAusLinienObjekten linie) {
		final KdBestehtAusLinienObjekten.Daten datum = linie.getKdBestehtAusLinienObjekten().getDatum();
		if (datum == null) {
			throw new IllegalArgumentException("Für die Linie wurden keine Koordinaten konfiguriert: " + linie);
		}

		final List<WGS84Punkt> result = new ArrayList<>();
		final Feld<Linie> linien = datum.getLinienReferenz();
		for (int i = 0; i < linien.size(); ++i) {
			final WGS84Polygon polygon = konvertiereIntern(linien.get(i));
			result.addAll(polygon.getKoordinaten());

			/*
			 * Außer bei der letzten Linie, den letzten Punkt wegwerfen, da er identisch
			 * ist, mit dem ersten, der nächsten.
			 */
			if (i != (linien.size() - 1)) {
				result.remove(result.size() - 1);
			}
		}

		return new WGS84Polygon(result);
	}

	public static List<WGS84Koordinate> konvertiere(final Flaeche flaeche) {
		if (flaeche instanceof FlaecheXY) {
			final WGS84Polygon wgs84 = konvertiereIntern((FlaecheXY) flaeche);
			return (List) wgs84.getKoordinaten();
		}

		throw new IllegalArgumentException("Die Fläche ist nicht vom Typ FlächeXY : " + flaeche);
	}

	private static WGS84Polygon konvertiereIntern(final FlaecheXY flaeche) {
		final KdFlaechenKoordinaten.Daten datum = flaeche.getKdFlaechenKoordinaten().getDatum();
		if (datum == null) {
			throw new IllegalArgumentException("Für die Fläche wurden keine Koordinaten konfiguriert: " + flaeche);
		}

		final Feld<AttWgs84Laenge> feldX = datum.getX();
		final Feld<AttWgs84Breite> feldY = datum.getY();
		if (feldX.size() != feldY.size()) {
			throw new IllegalStateException(
					"Die Längen der Felder für x- und y-Koordinaten sind unterschiedlich: " + flaeche);
		}

		final List<WGS84Punkt> punkte = new ArrayList<>();
		for (int i = 0; i < feldX.size(); ++i) {
			punkte.add(new WGS84Punkt(feldX.get(i).doubleValue(), feldY.get(i).doubleValue()));
		}

		return new WGS84Polygon(punkte);
	}

	/**
	 * Gibt die Teilstrecke des Polygons zurück, auf der der Punkt liegt.
	 * 
	 * @param punkt ein Punkt auf einer Linie.
	 * @return ein Feld mit zwei Punkten.
	 */
	public static WGS84Koordinate[] findTeilstrecke(final PunktLiegtAufLinienObjekt punkt) {
		final WGS84Punkt wgs84Punkt = konvertiereIntern(punkt);
		final Linie linie = punkt.getKdPunktLiegtAufLinienObjekt().getDatum().getLinienReferenz();
		final WGS84Polygon wgs84Polygon = konvertiereIntern(linie);
		final WGS84Polygon teilstrecke = wgs84Polygon.findeTeilstreckeKleinsterAbstand(wgs84Punkt);

		final ArrayList<WGS84Punkt> koordinaten = teilstrecke.getKoordinaten();
		assert koordinaten.size() == 2;
		final WGS84Koordinate[] result = new WGS84Koordinate[2];
		result[0] = koordinaten.get(0);
		result[1] = koordinaten.get(1);
		return result;
	}

	/**
	 * Gibt die Linie als Polygon zurueck, auf der der Punkt liegt.
	 * 
	 * @param punkt ein Punkt auf einer Linie.
	 * @return die Linie Polygon, auf der der Punkt liegt.
	 */
	public static WGS84Polygon getLinie(final PunktLiegtAufLinienObjekt punkt) {
		final Linie linie = punkt.getKdPunktLiegtAufLinienObjekt().getDatum().getLinienReferenz();
		final WGS84Polygon wgs84Polygon = konvertiereIntern(linie);
		return wgs84Polygon;
	}

	/**
	 * Verbindet Linienobjekte.
	 * 
	 * @param linien eine Menge von Linien.
	 * @return das sich ergebende Polygon.
	 */
	public static WGS84Polygon verbinde(final List<? extends Linie> linien) {
		final List<WGS84Punkt> result = new ArrayList<>();
		for (int i = 0; i < linien.size(); ++i) {
			final WGS84Polygon polygon = konvertiereIntern(linien.get(i));
			result.addAll(polygon.getKoordinaten());

			/*
			 * Außer bei der letzten Linie, den letzten Punkt wegwerfen, da er identisch
			 * ist, mit dem ersten, der nächsten.
			 */
			if (i != (linien.size() - 1)) {
				result.remove(result.size() - 1);
			}
		}
		return new WGS84Polygon(result);
	}

	/**
	 * Schneidet an einer Linienmenge am Anfang und Ende ein Stück ab.
	 * 
	 * @param linien      eine Menge von Linien.
	 * @param startOffset der Offset auf der ersten Linie, vor dem alles
	 *                    weggeschnitten werden soll.
	 * @param endOffset   der Offset auf der letzten Linie, nach der alles
	 *                    weggeschnitten werden soll.
	 * @return das sich ergebende Polygon.
	 */
	public static WGS84Polygon abschneiden(final List<? extends Linie> linien, final double startOffset,
			final double endOffset) {
		final List<WGS84Punkt> result = new ArrayList<>();
		for (int i = 0; i < linien.size(); ++i) {
			final WGS84Polygon polygon = konvertiereIntern(linien.get(i));

			if ((0 < i) && (i < (linien.size() - 1))) {
				result.addAll(polygon.getKoordinaten());
			} else if (i == 0) {
				polygon.anfangAbschneiden(startOffset);

				if (i == (linien.size() - 1)) {
					if (endOffset != 0) {
						final WGS84Polygon dummy = polygon.anfangAbschneiden(endOffset - startOffset);
						if (dummy != null) {
							result.addAll(dummy.getKoordinaten());
						}
					} else {
						result.addAll(polygon.getKoordinaten());
					}
				} else {
					result.addAll(polygon.getKoordinaten());
				}
			} else if (i == (linien.size() - 1)) {
				if (endOffset != 0) {
					final WGS84Polygon dummy = polygon.anfangAbschneiden(endOffset);
					if (dummy != null) {
						result.addAll(dummy.getKoordinaten());
					}
				}
			}

			/*
			 * Außer bei der letzten Linie, den letzten Punkt wegwerfen, da er identisch
			 * ist, mit dem ersten, der nächsten.
			 */
			if ((i != (linien.size() - 1)) && !result.isEmpty()) {
				result.remove(result.size() - 1);
			}
		}
		return new WGS84Polygon(result);
	}

	/**
	 * Schneidet aus einer Linie in der Mitte ein Stueck aus.
	 * 
	 * @param linie       eine Linie.
	 * @param startOffset der Offset auf der Linie, vor dem alles weggeschnitten
	 *                    werden soll. Wird hier ein negativer Wert angegeben, so
	 *                    soll nichts weggeschnitten werden.
	 * @param endOffset   der Offset auf der Linie, nach dem alles weggeschnitten
	 *                    werden soll. Wird hier ein negativer Wert angegeben, so
	 *                    soll nichts weggeschnitten werden.
	 * @return das sich ergebende Polygon.
	 */
	public static WGS84Polygon ausschneiden(final Linie linie, final double startOffset, final double endOffset) {
		final WGS84Polygon polygon = konvertiereIntern(linie);

		if ((startOffset > 0.0) || (endOffset >= 0.0)) {
			if (startOffset > 0.0) {
				polygon.anfangAbschneiden(startOffset);
			}

			if (endOffset >= 0.0) {
				final List<WGS84Punkt> apunkte = new ArrayList<>();
				try {
					final double schnittBei = endOffset - (startOffset >= 0.0 ? startOffset : 0.0);
					final WGS84Polygon dummy = polygon.anfangAbschneiden(schnittBei);

					/**
					 * FALLBACK: Bei sehr langen Polygonen, von denen nur ein kleines Stueck am Ende
					 * abgeschnitten wird bleibt manchmal nichts uebrig! Deshalb:
					 */
					if (dummy == null) {
						if (Math.abs(polygon.laenge() - schnittBei) < 10.0) {
							if (polygon.getKoordinaten().size() > 1) {
								apunkte.add(polygon.getKoordinaten().get(polygon.getKoordinaten().size() - 2));
								apunkte.add(polygon.getKoordinaten().get(polygon.getKoordinaten().size() - 1));
							} else {
								apunkte.add(polygon.getKoordinaten().get(polygon.getKoordinaten().size() - 1));
							}
						} else {
							apunkte.add(polygon.getKoordinaten().get(0));
							if (polygon.getKoordinaten().size() > 1) {
								apunkte.add(polygon.getKoordinaten().get(1));
							}
						}
					} else {
						apunkte.addAll(dummy.getKoordinaten());
					}
				} catch (final Exception ex) {
					apunkte.addAll(polygon.anfangAbschneiden(endOffset - (startOffset >= 0.0 ? startOffset : 0.0))
							.getKoordinaten());
				}

				return new WGS84Polygon(apunkte);
			}
		}

		return polygon;
	}

	private WGS84Util() {
		// utility class
	}

}
