/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2011 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.sys.funclib.bitctrl.modell.util.kalender;

import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import javax.swing.event.EventListenerList;

import com.bitctrl.util.Interval;

import de.bsvrz.dav.daf.main.config.ClientApplication;
import de.bsvrz.dav.daf.main.config.ConfigurationChangeException;
import de.bsvrz.sys.funclib.bitctrl.modell.AnmeldeException;
import de.bsvrz.sys.funclib.bitctrl.modell.Aspekt;
import de.bsvrz.sys.funclib.bitctrl.modell.DatensatzUpdateEvent;
import de.bsvrz.sys.funclib.bitctrl.modell.DatensatzUpdateListener;
import de.bsvrz.sys.funclib.bitctrl.modell.DatensendeException;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.OnlineDatensatz.Status;
import de.bsvrz.sys.funclib.bitctrl.modell.att.Feld;
import de.bsvrz.sys.funclib.bitctrl.modell.att.Zeitstempel;
import de.bsvrz.sys.funclib.bitctrl.modell.systemmodellglobal.objekte.Applikation;
import de.bsvrz.sys.funclib.bitctrl.modell.tmereigniskalenderglobal.attribute.AtlZusaetzlicheAttribute;
import de.bsvrz.sys.funclib.bitctrl.modell.tmereigniskalenderglobal.attribute.AttEreignisTypPrioritaet;
import de.bsvrz.sys.funclib.bitctrl.modell.tmereigniskalenderglobal.konfigurationsdaten.KdEreignisEigenschaften;
import de.bsvrz.sys.funclib.bitctrl.modell.tmereigniskalenderglobal.konfigurationsdaten.KdEreignisTypEigenschaften;
import de.bsvrz.sys.funclib.bitctrl.modell.tmereigniskalenderglobal.objekte.Ereignis;
import de.bsvrz.sys.funclib.bitctrl.modell.tmereigniskalenderglobal.objekte.EreignisTyp;
import de.bsvrz.sys.funclib.bitctrl.modell.tmereigniskalenderglobal.onlinedaten.OdEreignisKalenderAnfrage;
import de.bsvrz.sys.funclib.bitctrl.modell.tmereigniskalenderglobal.onlinedaten.OdEreignisKalenderAntwort;
import de.bsvrz.sys.funclib.bitctrl.modell.tmereigniskalenderglobal.parameter.PdEreignisParameter;
import de.bsvrz.sys.funclib.bitctrl.modell.tmereigniskalenderglobal.parameter.PdEreignisTypParameter;
import de.bsvrz.sys.funclib.bitctrl.modell.tmsystemkalenderglobal.objekte.Kalender;
import de.bsvrz.sys.funclib.bitctrl.modell.tmsystemkalenderglobal.objekte.SystemKalenderEintrag;
import de.bsvrz.sys.funclib.bitctrl.modell.tmsystemkalenderglobal.parameter.PdSystemKalenderEintrag;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.dynobj.DynObjektException;

/**
 * Diese Klasse stellt Methoden zur Verf&uuml;gung um ein einfacher Art und
 * Weise Anfragen an den Ereigniskalender zustellen.
 *
 * @author BitCtrl Systems GmbH, Schumann
 */
public final class Ereigniskalender implements DatensatzUpdateListener {

	/** Sichert die Liste des Singletons pro Datenverteilerverbindung. */
	private static Ereigniskalender singleton;

	/**
	 * Gibt einen Kalender als Singleton zur&uuml;ck.
	 *
	 * @param factory {@link ObjektFactory}
	 *
	 * @return der Kalender als Singleton.
	 */
	public static synchronized Ereigniskalender getInstanz(final ObjektFactory factory) {
		if (singleton == null) {
			singleton = new Ereigniskalender(factory);
		}
		return singleton;
	}

	/** Der Logger. */
	private final Debug log = Debug.getLogger();

	/** Angemeldete Listener. */
	private final EventListenerList listeners = new EventListenerList();

	/** Die Eigenschaft {@code kalender}. */
	private final Kalender kalender;

	/** Der Anfragedatensatz. */
	private final OdEreignisKalenderAnfrage odAnfrage;

	/** Der Aspekt zum Senden der Anfrage. */
	private final Aspekt aspAnfrage;

	/** Die ObjektFactory. */
	private final ObjektFactory factory;

	/**
	 * Initialisert die Anfrageschnittstelle.
	 *
	 * @param factory {@link ObjektFactory}
	 */
	private Ereigniskalender(final ObjektFactory factory) {
		this.factory = factory;

		final OdEreignisKalenderAntwort odAntwort;
		final Aspekt aspAntwort;
		final Applikation klient;

		// Anfragedatensatz bestimmen
		kalender = (Kalender) factory.getModellobjekt(factory.getDav().getLocalConfigurationAuthority());
		aspAnfrage = OdEreignisKalenderAnfrage.Aspekte.Anfrage;
		odAnfrage = kalender.getOdEreignisKalenderAnfrage();

		// Anmelden als Empfänger der Kalenderantworten
		klient = (Applikation) factory.getModellobjekt(factory.getDav().getLocalApplicationObject());
		odAntwort = klient.getOdEreignisKalenderAntwort();
		aspAntwort = OdEreignisKalenderAntwort.Aspekte.Antwort;
		odAntwort.setSenke(aspAntwort, true);
		odAntwort.addUpdateListener(aspAntwort, this);
		try {
			odAnfrage.anmeldenSender(aspAnfrage);
		} catch (final AnmeldeException ex) {
			log.error("Anmeldung zum Senden von Anfragen an die Ganglinienprognose konnte nicht durchgeführt werden",
					ex);
		}

		log.info("Schnittstelle zum Kalender initialisiert.");
	}

	/**
	 * Registriert einen Listener.
	 *
	 * @param listener Der neue Listener
	 */
	public void addKalenderListener(final KalenderListener listener) {
		listeners.add(KalenderListener.class, listener);
	}

	/**
	 * Legt ein Ereignis an und f&uuml;gt ihn der entsprechenden Menge am Kalender
	 * hinzu.
	 *
	 * @param pid          die PID.
	 * @param name         der Name.
	 * @param beschreibung die Beschreibung.
	 * @param typ          der Ereignistyp.
	 * @param intervall    das Intervall in dem das Ereignis g&uuml;ltig sein soll.
	 *                     Das Intervall bezieht sich auf die zeitliche und
	 *                     verkehrliche G&uuml;ltigkeit.
	 * @param quelle       die Quelle.
	 * @return der angeleget Ereignistyp.
	 * @throws ConfigurationChangeException wenn das Anlegen fehlerhaft verlief.
	 * @throws AnmeldeException             wenn das Anmelden nicht m&ouml;glich
	 *                                      war.
	 * @throws DatensendeException          wenn das Sendes Parameters mit dem
	 *                                      Intervall nicht m&ouml;glich war.
	 */
	public Ereignis anlegenEreignis(final String pid, final String name, final String beschreibung,
			final EreignisTyp typ, final Interval intervall, final String quelle)
			throws ConfigurationChangeException, AnmeldeException, DatensendeException {
		final Ereignis erg;
		final PdEreignisParameter param;
		final PdEreignisParameter.Daten datum;

		try {
			erg = ereignisObjektAnlegen(pid, name, beschreibung, typ);
		} catch (final DynObjektException ex) {
			throw new ConfigurationChangeException("Das Ereignis '" + name + "' konnte nicht angelegt werden.", ex);
		}

		kalender.getEreignisse().add(erg);

		param = erg.getPdEreignisParameter();
		param.anmeldenSender();
		datum = param.createDatum();
		datum.setBeginnZeitlicheGueltigkeit(new Zeitstempel(intervall.getStart()));
		datum.setEndeZeitlicheGueltigkeit(new Zeitstempel(intervall.getEnd()));

		datum.getVerkehrlicheGueltigkeit().clear();
		datum.setQuelle(quelle);
		param.sendeDatum(datum);

		return erg;
	}

	/**
	 * Erzeugt ein Ereignis-Objekt.
	 *
	 * @param pid
	 * @param name
	 * @param beschreibung
	 * @param typ
	 * @return {@link Ereignis}
	 * @throws DynObjektException
	 */
	private Ereignis ereignisObjektAnlegen(final String pid, final String name, final String beschreibung,
			final EreignisTyp typ) throws DynObjektException {

		// Konfigurationsdatensatz erzeugen
		final KdEreignisEigenschaften datensatz = new KdEreignisEigenschaften(null, factory);
		final KdEreignisEigenschaften.Daten datum = new KdEreignisEigenschaften.Daten(datensatz,
				KdEreignisEigenschaften.Aspekte.Eigenschaften);

		datum.setEreignisbeschreibung(beschreibung);
		datum.setEreignisTypReferenz(typ);

		return factory.createDynamischesObjekt(Ereignis.class, name, pid, datum);
	}

	/**
	 * Erzeugt ein EreignisTyp-Objekt.
	 *
	 * @param pid
	 * @param name
	 * @param attribute
	 * @return {@link EreignisTyp}
	 * @throws DynObjektException
	 */
	private EreignisTyp ereignisTypObjektAnlegen(final String pid, final String name,
			final Map<String, String> attribute) throws DynObjektException {

		// Konfigurationsdatensatz erzeugen
		final KdEreignisTypEigenschaften datensatz = new KdEreignisTypEigenschaften(null, factory);
		final KdEreignisTypEigenschaften.Daten datum = new KdEreignisTypEigenschaften.Daten(datensatz,
				KdEreignisTypEigenschaften.Aspekte.Eigenschaften);

		final Feld<AtlZusaetzlicheAttribute> zusaetzlicheAttribute = datum.getZusaetzlicheAttribute();

		for (final Entry<String, String> entry : attribute.entrySet()) {
			final AtlZusaetzlicheAttribute att = new AtlZusaetzlicheAttribute();
			att.setAttributname(entry.getKey());
			att.setAttributwert(entry.getValue());
			zusaetzlicheAttribute.add(att);
		}

		return factory.createDynamischesObjekt(EreignisTyp.class, name, pid, datum);
	}

	/**
	 * Erzeugt ein SystemKalenderEintrag-Objekt.
	 *
	 * @param pid
	 * @param name
	 * @return {@link SystemKalenderEintrag}
	 * @throws DynObjektException
	 */
	private SystemKalenderEintrag systemKalenderEintragObjektAnlegen(final String pid, final String name)
			throws DynObjektException {
		return factory.createDynamischesObjekt(SystemKalenderEintrag.class, name, pid);
	}

	/**
	 * Legt ein Ereignis an und f&uuml;gt ihn der entsprechenden Menge am Kalender
	 * hinzu.
	 *
	 * @param pid          die PID.
	 * @param name         der Name.
	 * @param beschreibung die Beschreibung.
	 * @param typ          der Ereignistyp.
	 * @param ske          ein Systemkalendereintrag, der die G&uuml;ltigkeit des
	 *                     Ereignisses beschreibt. Die zeitliche und verkehrliche
	 *                     G&uuml;ltigkeit wird als identisch angenommen.
	 * @param quelle       die Quelle.
	 * @return der angeleget Ereignistyp.
	 * @throws ConfigurationChangeException wenn das Anlegen fehlerhaft verlief.
	 * @throws AnmeldeException             wenn das Anmelden nicht m&ouml;glich
	 *                                      war.
	 * @throws DatensendeException          wenn das Sendes Parameters mit dem
	 *                                      Intervall nicht m&ouml;glich war.
	 */
	public Ereignis anlegenEreignis(final String pid, final String name, final String beschreibung,
			final EreignisTyp typ, final SystemKalenderEintrag ske, final String quelle)
			throws ConfigurationChangeException, AnmeldeException, DatensendeException {
		final Ereignis erg;
		final PdEreignisParameter param;
		final PdEreignisParameter.Daten datum;

		try {
			erg = ereignisObjektAnlegen(pid, name, beschreibung, typ);
		} catch (final DynObjektException ex) {
			throw new ConfigurationChangeException("Das Ereignis '" + name + "' konnte nicht angelegt werden.", ex);
		}

		kalender.getEreignisse().add(erg);

		param = erg.getPdEreignisParameter();
		param.anmeldenSender();
		datum = param.createDatum();

		datum.getVerkehrlicheGueltigkeit().clear();
		datum.setSystemKalenderEintragReferenz(ske);

		datum.setQuelle(quelle);
		param.sendeDatum(datum);

		return erg;
		// Ereignis erg;
		// PdEreignisParameter param;
		// PdEreignisParameter.Daten datum;
		//
		// erg = Ereignis.anlegen(pid, name, beschreibung, typ);
		// kalender.add(erg);
		//
		// param = erg.getParameterDatensatz(PdEreignisParameter.class);
		// param.anmeldenSender();
		// datum = param.erzeugeDatum();
		// datum.setSystemKalenderEintrag(ske);
		// datum.getVerkehrlicheGueltigkeit()
		// .add(new PdEreignisParameter.Daten.VerkehrlicheGueltigkeit());
		// datum.setQuelle(quelle);
		// param.sendeDaten(datum);
		//
		// return erg;
	}

	/**
	 * Legt einen Ereignistyp an und f&uuml;gt ihn der entsprechenden Menge am
	 * Kalender hinzu.
	 *
	 * @param pid        die PID.
	 * @param name       der Name.
	 * @param prioritaet die Priorit&auml;t des Ereignistyps.
	 * @return der angeleget Ereignistyp.
	 * @throws ConfigurationChangeException wenn das Anlegen fehlerhaft verlief.
	 * @throws DatensendeException          wenn der Parameter mit der
	 *                                      Priorit&auml;t nicht gesendet werden
	 *                                      konnte.
	 * @throws AnmeldeException             wenn der Parameter mit der
	 *                                      Priorit&auml;t nicht zum Senden
	 *                                      angemeldet werden konnte.
	 */
	public EreignisTyp anlegenEreignisTyp(final String pid, final String name, final int prioritaet)
			throws ConfigurationChangeException, DatensendeException, AnmeldeException {
		return anlegenEreignisTyp(pid, name, prioritaet, new HashMap<String, String>());
	}

	/**
	 * Legt einen Ereignistyp an und f&uuml;gt ihn der entsprechenden Menge am
	 * Kalender hinzu.
	 *
	 * @param pid        die PID.
	 * @param name       der Name.
	 * @param prioritaet die Priorit&auml;t des Ereignistyps.
	 * @param attribute  zus&auml;tzliche Attribute in Form einer Liste von
	 *                   Schl&uuml;ssel/Wert-Paaren.
	 * @return der angeleget Ereignistyp.
	 * @throws ConfigurationChangeException wenn das Anlegen fehlerhaft verlief.
	 * @throws DatensendeException          wenn der Parameter mit der
	 *                                      Priorit&auml;t nicht gesendet werden
	 *                                      konnte.
	 * @throws AnmeldeException             wenn der Parameter mit der
	 *                                      Priorit&auml;t nicht zum Senden
	 *                                      angemeldet werden konnte.
	 */
	public EreignisTyp anlegenEreignisTyp(final String pid, final String name, final int prioritaet,
			final Map<String, String> attribute)
			throws ConfigurationChangeException, DatensendeException, AnmeldeException {
		final EreignisTyp typ;
		final PdEreignisTypParameter param;
		final PdEreignisTypParameter.Daten datum;

		try {
			typ = ereignisTypObjektAnlegen(pid, name, attribute);
		} catch (final DynObjektException ex) {
			throw new ConfigurationChangeException("Der EreignisTyp '" + name + "' konnte nicht angelegt werden.", ex);
		}

		param = typ.getPdEreignisTypParameter();
		param.anmeldenSender();
		datum = param.createDatum();
		datum.setEreignisTypPrioritaet(new AttEreignisTypPrioritaet((long) prioritaet));
		param.sendeDatum(datum);

		kalender.getEreignisTypen().add(typ);

		return typ;
	}

	/**
	 * Legt einen Systemkalendereintrag an und f&uuml;gt ihn der entsprechenden
	 * Menge am Kalender hinzu.
	 *
	 * @param pid        die PID.
	 * @param name       der Name.
	 * @param definition die Definition des Systemkalendereintrags.
	 * @return der angeleget Ereignistyp.
	 * @throws ConfigurationChangeException wenn das Anlegen fehlerhaft verlief.
	 * @throws AnmeldeException             wenn das Anmelden zum Datensenden nicht
	 *                                      m&ouml;glich ist.
	 * @throws DatensendeException          wenn der Parameter mit der Definition
	 *                                      nicht gesendet werden konnte.
	 */
	public SystemKalenderEintrag anlegenSystemKalenderEintrag(final String pid, final String name,
			final String definition) throws ConfigurationChangeException, AnmeldeException, DatensendeException {
		final SystemKalenderEintrag ske;
		final PdSystemKalenderEintrag param;
		final PdSystemKalenderEintrag.Daten datum;

		// ske = SystemKalenderEintrag.anlegen(pid, name);
		try {
			ske = systemKalenderEintragObjektAnlegen(pid, name);
		} catch (final DynObjektException ex) {
			throw new ConfigurationChangeException(
					"Der SystemKalenderEintrag '" + name + "' konnte nicht angelegt werden.", ex);
		}
		// kalender.add(ske);
		kalender.getSystemKalenderEintraege().add(ske);

		// param = ske.getParameterDatensatz(PdSystemKalenderEintrag.class);
		param = ske.getPdSystemKalenderEintrag();
		param.anmeldenSender();
		// datum = param.erzeugeDatum();
		datum = param.createDatum();
		datum.setDefinition(definition);
		// param.sendeDaten(datum);
		param.sendeDatum(datum);

		return ske;
	}

	@Override
	public void datensatzAktualisiert(final DatensatzUpdateEvent event) {
		if (event.getDatum().dContainsDaten()) {
			fireAntwort((OdEreignisKalenderAntwort.Daten) event.getDatum());
		}
	}

	/**
	 * Fragt, ob der Kalender Anfragen entgegennimmt. Wenn die Sendesteuerung noch
	 * nicht geantwortet hat, wartet die Methode maximal 30 Sekunden. Hat die
	 * Sendesteuerung schon geantwortet, entsteht keine Verz&ouml;gerung.
	 *
	 * @return {@code true}, wenn der Kalender verwendet werden kann.
	 */
	public boolean isBereit() {
		// System.out.println(
		// "TEST: " + odAnfrage.getStatusSendesteuerung(aspAnfrage));
		for (int i = 0; i < 300; i++) {
			if (odAnfrage.getStatusSendesteuerung(aspAnfrage) != null) {
				break;
			}
			factory.getDav().sleep(100);
		}
		return odAnfrage.getStatusSendesteuerung(aspAnfrage) == Status.START;
	}

	/**
	 * Entfernt das Ereignis aus der entsprechenden Menge am Kalender und
	 * "l&ouml;scht" das Ereignis anschlie&szlig;end, in dem er invalidiert wird.
	 *
	 * @param erg das zu l&ouml;schende Ereignis.
	 * @throws ConfigurationChangeException wenn das L&ouml;schen schief ging.
	 */
	public void loeschen(final Ereignis erg) throws ConfigurationChangeException {
		kalender.getEreignisse().remove(erg);
		erg.getSystemObject().invalidate();
	}

	/**
	 * Entfernt den Ereignistyp aus der entsprechenden Menge am Kalender und
	 * "l&ouml;scht" den Typ anschlie&szlig;end, in dem er invalidiert wird.
	 *
	 * @param typ der zu l&ouml;schende Ereignistyp.
	 * @throws ConfigurationChangeException wenn das L&ouml;schen schief ging.
	 */
	public void loeschen(final EreignisTyp typ) throws ConfigurationChangeException {
		kalender.getEreignisTypen().remove(typ);
		typ.getSystemObject().invalidate();
	}

	/**
	 * Entfernt den Ereignistyp aus der entsprechenden Menge am Kalender und
	 * "l&ouml;scht" den Typ anschlie&szlig;end, in dem er invalidiert wird.
	 *
	 * @param typ der zu l&ouml;schende Ereignistyp.
	 * @throws ConfigurationChangeException wenn das L&ouml;schen schief ging.
	 */
	public void loeschen(final SystemKalenderEintrag typ) throws ConfigurationChangeException {
		kalender.getSystemKalenderEintraege().remove(typ);
		typ.getSystemObject().invalidate();
	}

	/**
	 * Entfernt einen Listener wieder aus der Liste registrierter Listener.
	 *
	 * @param listener Listener der abgemeldet werden soll
	 */
	public void removeKalenderListener(final KalenderListener listener) {
		listeners.remove(KalenderListener.class, listener);
	}

	/**
	 * Sendet eine Anfrage an den Kalender. Die anfragende Applikation wird
	 * &uuml;ber ein Event &uuml;ber die eingetroffene Antwort informiert.
	 *
	 * @param absenderZeichen ein beliebiger Text.
	 * @param anfrage         die Anfragen.
	 * @throws DatensendeException wenn beim Senden ein Fehler passiert ist.
	 */
	public void sendeAnfrage(final String absenderZeichen, final KalenderAnfrage anfrage) throws DatensendeException {
		final OdEreignisKalenderAnfrage.Daten datum;
		final ClientApplication klient;

		klient = factory.getDav().getLocalApplicationObject();

		// datum = odAnfrage.erzeugeDatum();
		// datum.setAbsender((Applikation) factory.getModellobjekt(klient));
		// datum.setAbsenderZeichen(absenderZeichen);
		// datum.setEreignisTypenOption(anfrage.getEreignisTypenOption());
		// datum.setIntervall(anfrage.getIntervall());
		// datum.getRaeumlicheGueltigkeit()
		// .addAll(anfrage.getRaeumlicheGueltigkeit());
		// datum.getEreignisTypen().addAll(anfrage.getEreignisTypen());
		//
		// odAnfrage.sendeDaten(aspAnfrage, datum);

		datum = odAnfrage.createDatum();
		datum.setAbsenderId((Applikation) factory.getModellobjekt(klient));
		datum.setAbsenderZeichen(absenderZeichen);

		datum.setEreignisTypenOption(anfrage.getEreignisTypenOption());

		datum.setAnfangszeitpunkt(new Zeitstempel(anfrage.getIntervall().getStart()));
		datum.setEndzeitpunkt(new Zeitstempel(anfrage.getIntervall().getEnd()));

		datum.getRaeumlicheGueltigkeit().addAll(anfrage.getRaeumlicheGueltigkeit());
		datum.getEreignisTypReferenz().addAll(anfrage.getEreignisTypen());

		odAnfrage.sendeDatum(aspAnfrage, datum);

		log.fine("Anfrage \"" + absenderZeichen + "\" wurde gesendet");
	}

	/**
	 * Informiert alle registrierten Listener &uuml;ber eine Antwort.
	 *
	 * @param datum das Datum mit der Antwort.
	 */
	protected synchronized void fireAntwort(final OdEreignisKalenderAntwort.Daten datum) {
		final KalenderEvent e = new KalenderEvent(this, datum);

		for (final KalenderListener l : listeners.getListeners(KalenderListener.class)) {
			l.antwortEingetroffen(e);
		}

		log.fine("Kalenderantwort wurde verteilt: " + e);
	}

}
