/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2011 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.bitctrl.modell.util.rdstmc;

import java.util.HashMap;
import java.util.Map;

/**
 * Utility-Klasse, welche diverse Hilfsmethoden für Ereignis-Quantitäten
 * bereitstellt.
 *
 * @author BitCtrl Systems GmbH, Andreas Meissner
 */
public final class QuantitaetenUtil {

	/**
	 * LEN: Betroffene Länge in km [1, 2, ..., 10, 12, ..., 20, 25, ..., 100,
	 * &gt;100].
	 */
	public static final String KENNUNG_BETROFFENE_LAENGE = "LEN";

	/**
	 * Q00: Anzahl [1, 2, ..., 30, 32, 34, 36].
	 */
	public static final String KENNUNG_ANZAHL_KLEIN = "Q00";

	/**
	 * Q01: Anzahl [1, 2, ..., 10, 20, ..., 100, 150, ..., 1000].
	 */
	public static final String KENNUNG_ANZAHL_GROSS = "Q01";

	/**
	 * VIS: Sichtweite in m [10, 20, ..., 300].
	 */
	public static final String KENNUNG_SICHTWEITE = "VIS";

	/**
	 * POC: Parkplatzbelegung in % [0, 5, ..., 100].
	 */
	public static final String KENNUNG_PARKPLATZBELEGUNG = "POC";

	/**
	 * AVV: Durchschnittsgeschwindigkeit in km/h [5, 10, ..., 160].
	 */
	public static final String KENNUNG_DURCHSCHNITTSGESCHWINDIGKEIT = "AVV";

	/**
	 * durchschnittliche Windgeschwindigkeit in km/h [5, 10, ..., 160].
	 */
	public static final String KENNUNG_WINDGESCHWINDIGKEIT = "WDS";

	/**
	 * Q05: Verzögerung in Minuten/Stunden [5, 10, ..., 50, 1, 2, ..., 12, 18, ...
	 * 72].
	 */
	public static final String KENNUNG_VERZOEGERUNG = "Q05";

	/**
	 * ATE: Lufttemperatur in °C [-50, -49, ..., 50].
	 */
	public static final String KENNUNG_LUFTTEMPERATUR = "ATE";

	/**
	 * Q07: Zeit [00.00, 00.10, ..., 23.50].
	 */
	public static final String KENNUNG_ZEIT = "Q07";

	/**
	 * WEI: Fahrzeuggewicht in t [0.1, 0.2, ..., 10.0, 10.5, ..., 60.0].
	 */
	public static final String KENNUNG_FAHRZEUGGEWICHT = "WEI";

	/**
	 * Q09: Begrenzung in m [0.1, 0.2, ..., 10.0, 10.5, ..., 80.0].
	 */
	public static final String KENNUNG_BEGRENZUNG = "Q09";

	/**
	 * PDD: Niederschlagshöhe in mm [1, 2, ..., 255].
	 */
	public static final String KENNUNG_NIEDERSCHLAGSHOEHE = "PDD";

	/**
	 * FMH: FM Frequenz in MHz [87.5, 87.6, ..., 108.0].
	 */
	public static final String KENNUNG_FM_FREQUENZ = "FMH";

	/**
	 * AMH: AM Frequenz in kHz [153, 154, ..., 288, 522, 523, ..., 1620].
	 */
	public static final String KENNUNG_AM_FREQUENZ = "AMH";

	/**
	 * Liste der gültigen Quantitäten neben der betroffenen Länge (LEN).
	 *
	 * @see #KENNUNG_BETROFFENE_LAENGE
	 */
	public static final String[] GUELTIGE_QUANTITAETEN_KENNUNGEN = { KENNUNG_ANZAHL_KLEIN, KENNUNG_ANZAHL_GROSS,
			KENNUNG_SICHTWEITE, KENNUNG_PARKPLATZBELEGUNG, KENNUNG_DURCHSCHNITTSGESCHWINDIGKEIT,
			KENNUNG_WINDGESCHWINDIGKEIT, KENNUNG_VERZOEGERUNG, KENNUNG_LUFTTEMPERATUR, KENNUNG_ZEIT,
			KENNUNG_FAHRZEUGGEWICHT, KENNUNG_BEGRENZUNG, KENNUNG_NIEDERSCHLAGSHOEHE, KENNUNG_FM_FREQUENZ,
			KENNUNG_AM_FREQUENZ };

	/**
	 * Map, welche zu einer Quantität den gültigen Wertebereich speichert.
	 */
	private static Map<String, QuantitaetenWerteBereich> spinnerWerteMap;

	static {
		spinnerWerteMap = new HashMap<>();
		spinnerWerteMap.put(KENNUNG_ANZAHL_KLEIN, new QuantitaetenWerteBereich(1, 36, 0, 1, 5, 1));
		spinnerWerteMap.put(KENNUNG_ANZAHL_GROSS, new QuantitaetenWerteBereich(1, 1000, 0, 1, 100, 100));
		spinnerWerteMap.put(KENNUNG_SICHTWEITE, new QuantitaetenWerteBereich(10, 300, 0, 10, 50, 10));
		spinnerWerteMap.put(KENNUNG_PARKPLATZBELEGUNG, new QuantitaetenWerteBereich(0, 100, 0, 5, 20, 0));
		spinnerWerteMap.put(KENNUNG_DURCHSCHNITTSGESCHWINDIGKEIT, new QuantitaetenWerteBereich(5, 160, 0, 5, 20, 50));
		spinnerWerteMap.put(KENNUNG_WINDGESCHWINDIGKEIT, new QuantitaetenWerteBereich(5, 160, 0, 5, 20, 50));
		spinnerWerteMap.put(KENNUNG_VERZOEGERUNG, new QuantitaetenWerteBereich(5, 7200, 2, 5, 100, 5));
		spinnerWerteMap.put(KENNUNG_LUFTTEMPERATUR, new QuantitaetenWerteBereich(-50, 50, 0, 1, 10, 0));
		spinnerWerteMap.put(KENNUNG_NIEDERSCHLAGSHOEHE, new QuantitaetenWerteBereich(1, 255, 0, 1, 10, 10));
		spinnerWerteMap.put(KENNUNG_ZEIT, new QuantitaetenWerteBereich(0, 2350, 2, 10, 100, 1200));
		spinnerWerteMap.put(KENNUNG_FAHRZEUGGEWICHT, new QuantitaetenWerteBereich(1, 600, 1, 1, 10, 10));
		spinnerWerteMap.put(KENNUNG_BEGRENZUNG, new QuantitaetenWerteBereich(1, 800, 1, 1, 10, 10));
		spinnerWerteMap.put(KENNUNG_FM_FREQUENZ, new QuantitaetenWerteBereich(875, 1080, 1, 1, 10, 875));
		spinnerWerteMap.put(KENNUNG_AM_FREQUENZ, new QuantitaetenWerteBereich(153, 1620, 0, 1, 100, 153));
	}

	/**
	 * Der private Default-Konstruktor.
	 */
	private QuantitaetenUtil() {
		// tut nix
	}

	/**
	 * Liefert den zu einer gegebenen Quantität gültigen Wertebereich zurück.
	 *
	 * @param kennung die Kennung der Quanität, darf nicht <code>null</code> sein
	 *
	 * @return den zu einer gegebenen Quantität gültigen Wertebereich
	 */
	public static QuantitaetenWerteBereich getSpinnerWert(final String kennung) {
		return spinnerWerteMap.get(kennung);
	}

	/**
	 * Liefert den entsprechenden Integer-Wert für die gegebene Zeichenkette zurück.
	 * Beachte, dieser Wert ist abhängig von den für die spezielle Quantität
	 * definierten Spinner-Werte.
	 *
	 * @param kennung die Kennung der dazugehörigen Quantität, darf nicht
	 *                <code>null</code> sein
	 * @param wert    die Zeichenkette, darf nicht <code>null</code> sein
	 *
	 * @return den entsprechenden Integer-Wert für die gegebene Zeichenkette
	 */
	public static Integer getIntWert(final String kennung, final String wert) {
		assert null != wert;
		assert null != kennung;

		final String s = wert.trim().replaceAll(" ", "").replaceAll(",", ".");

		if ("".equals(s)) {
			return null;
		}

		double d = 0;
		try {
			d = new Double(s);
		} catch (final NumberFormatException e) {
			return null;
		}

		if (KENNUNG_FAHRZEUGGEWICHT.equals(kennung) || KENNUNG_BEGRENZUNG.equals(kennung)
				|| KENNUNG_FM_FREQUENZ.equals(kennung)) {
			return new Integer((int) (d * 10));
		}
		if (KENNUNG_ZEIT.equals(kennung) || KENNUNG_VERZOEGERUNG.equals(kennung)) {
			return new Integer((int) (d * 100));
		}
		return new Integer((int) d);
	}

	/**
	 * Liefert die Zeichenkette für den gegeben Integer-Wert zurück. Beachte, dieser
	 * Wert ist abhängig von den für die spezielle Quantität definierten
	 * Spinner-Werte.
	 *
	 * @param kennung die Kennung der dazugehörigen Quantität, darf nicht
	 *                <code>null</code> sein
	 * @param wert    der Integer-Wert
	 *
	 * @return die Zeichenkette für den gegeben Integer-Wert
	 */
	public static String getStringWert(final String kennung, final Integer wert) {
		assert null != kennung;
		if (wert == null) {
			return "";
		}
		if (KENNUNG_FAHRZEUGGEWICHT.equals(kennung) || KENNUNG_BEGRENZUNG.equals(kennung)
				|| KENNUNG_FM_FREQUENZ.equals(kennung)) {
			final Double d = wert.doubleValue() / 10.d;
			return d.toString().replace(".", ",");
		}

		if (KENNUNG_ZEIT.equals(kennung)) {
			final Double d = wert.doubleValue() / 100.d;
			String ergebnis = d.toString();
			ergebnis = ergebnis + "0";
			if (d < 10) {
				ergebnis = "0" + ergebnis;
			}
			return ergebnis;
		}

		if (KENNUNG_VERZOEGERUNG.equals(kennung)) {
			final Double d = wert.doubleValue() / 100.d;
			String ergebnis = d.toString();
			if (d < 1) {
				if (ergebnis.length() == 3) {
					ergebnis = ergebnis + "0";
				}
			} else {
				return "" + d.intValue();
			}
			return ergebnis;
		}

		return wert.toString();
	}

	/**
	 * Liefert den Standardwert für die gegebene Quantität zurück.
	 *
	 * @param kennung die Kennung der dazugehörigen Quantität, darf nicht
	 *                <code>null</code> sein
	 *
	 * @return den Standardwert für die gegebene Quantität
	 */
	public static String getDefaultWert(final String kennung) {
		assert null != kennung;
		final QuantitaetenWerteBereich spinnerWert = getSpinnerWert(kennung);
		if (spinnerWert != null) {
			return getStringWert(kennung, spinnerWert.getDefaultValue());
		}
		return "";
	}

	/**
	 * Liefert den bereinigten Integer-Wert für die gegebene Quantität zurück.
	 *
	 * @param kennung   die Kennung der dazugehörigen Quantität, darf nicht
	 *                  <code>null</code> sein
	 * @param alterWert der vorherige Wert, darf <code>null</code> sein
	 * @param neuerWert der neue Wert, darf nicht <code>null</code> sein
	 *
	 * @return den bereinigten Integer-Wert für die gegebene Quantität
	 */
	public static Integer getValidIntWert(final String kennung, final Integer alterWert, final Integer neuerWert) {
		assert null != kennung;
		assert null != neuerWert;

		final boolean aufsteigend = (alterWert != null) ? alterWert < neuerWert : true;

		final QuantitaetenWerteBereich spinnerWert = getSpinnerWert(kennung);
		if (spinnerWert == null) {
			return neuerWert;
		}

		// Überprüfe Wertebereich
		if (neuerWert > spinnerWert.getMax()) {
			return spinnerWert.getMax();
		}
		if (neuerWert < spinnerWert.getMin()) {
			return spinnerWert.getMin();
		}

		// Quantität Anzahl (Q00)
		if (KENNUNG_ANZAHL_KLEIN.equals(kennung)) {

			switch (neuerWert) {
			case 31:
				return aufsteigend ? 32 : 30;
			case 33:
				return aufsteigend ? 34 : 32;
			case 35:
				return aufsteigend ? 36 : 34;
			default:
				return neuerWert;
			}
		}

		// Quantität Anzahl (Q01)
		if (KENNUNG_ANZAHL_GROSS.equals(kennung)) {
			if ((neuerWert > 10) && (neuerWert < 100)) {
				int quotient = neuerWert / 10;
				if ((neuerWert % 10) > 0) {
					quotient = aufsteigend ? quotient + 1 : quotient;
				}
				return quotient * 10;
			} else if ((neuerWert > 100) && (neuerWert < 1000)) {
				int modulo = neuerWert % 100;
				int quotient = neuerWert / 100;
				if ((modulo > 0) && (modulo < 50)) {
					modulo = aufsteigend ? 50 : 0;
				} else if (modulo > 50) {
					modulo = aufsteigend ? 0 : 50;
					quotient = aufsteigend ? quotient + 1 : quotient;
				}
				return (quotient * 100) + modulo;
			}
		}

		// Quantität Fahrzeuggewicht (WEI) und Begrenzung (Q09)
		if ((KENNUNG_FAHRZEUGGEWICHT.equals(kennung) || KENNUNG_BEGRENZUNG.equals(kennung)) && (neuerWert > 100)) {

			int modulo = neuerWert % 10;
			int quotient = neuerWert / 10;
			if ((modulo > 0) && (modulo < 5)) {
				modulo = aufsteigend ? 5 : 0;
			} else if (modulo > 5) {
				modulo = aufsteigend ? 0 : 5;
				quotient = aufsteigend ? quotient + 1 : quotient;
			}
			return (quotient * 10) + modulo;
		}

		// Quantität Verzögerung (Q05)
		if (KENNUNG_VERZOEGERUNG.equals(kennung)) {

			if ((neuerWert > 0) && (neuerWert < 50)) {
				int modulo = neuerWert % 10;
				int quotient = neuerWert / 10;
				if ((modulo > 0) && (modulo < 5)) {
					modulo = aufsteigend ? 5 : 0;
				} else if (modulo > 5) {
					modulo = aufsteigend ? 0 : 5;
					quotient = aufsteigend ? quotient + 1 : quotient;
				}
				return (quotient * 10) + modulo;
			}

			if ((neuerWert > 50) && (neuerWert < 100)) {
				return aufsteigend ? 100 : 50;
			}

			int ergebnis = neuerWert;
			if (neuerWert > 100) {
				int modulo = neuerWert % 10;
				int quotient = neuerWert / 10;
				if (modulo > 0) {
					quotient = aufsteigend ? quotient + 1 : quotient;
				}
				ergebnis = quotient * 10;

				modulo = ergebnis % 100;
				quotient = ergebnis / 100;
				if (modulo > 0) {
					quotient = aufsteigend ? quotient + 1 : quotient;
				}
				ergebnis = quotient * 100;

				if (ergebnis > 1200) {
					modulo = ergebnis % 600;
					quotient = ergebnis / 600;
					if (modulo > 0) {
						quotient = aufsteigend ? quotient + 1 : quotient;
					}
					return quotient * 600;
				}
				return ergebnis;
			}
		}

		// Quantität Zeit (Q07)
		if (KENNUNG_ZEIT.equals(kennung)) {

			int ergebnis = neuerWert;
			int modulo = neuerWert % 10;
			int quotient = neuerWert / 10;
			if (modulo > 0) {
				quotient = aufsteigend ? quotient + 1 : quotient;
			}
			ergebnis = quotient * 10;

			modulo = ergebnis % 100;
			quotient = ergebnis / 100;
			if (modulo > 50) {
				modulo = aufsteigend ? 0 : 50;
				quotient = aufsteigend ? quotient + 1 : quotient;
			}
			return (quotient * 100) + modulo;
		}

		// Quantität AM Frequenz (AMH)
		if (KENNUNG_AM_FREQUENZ.equals(kennung)) {
			if ((neuerWert > 288) && (neuerWert < 531)) {
				return aufsteigend ? 531 : 288;
			}
		}

		return neuerWert;
	}

	/**
	 * Liefert die ausführliche Zeichenkette zur Quantität Verzögerung (Q05) zurück.
	 *
	 * <p>
	 * Der Rückgabewert hat dabei z.B. folgende Form: '5 Minuten', '12 Stunden'.
	 * </p>
	 *
	 * @param quantitaetIntWert der Integer-Wert, darf nicht <code>null</code> sein
	 *
	 * @return die ausführliche Zeichenkette zur Quantität Verzögerung (Q05)
	 */
	public static String getVerzoegerungString(final Integer quantitaetIntWert) {
		assert null != quantitaetIntWert;
		return getVerzoegerungWertString(quantitaetIntWert) + " " + getVerzoegerungEinheitString(quantitaetIntWert);
	}

	/**
	 * Liefert die ausführliche Zeichenkette des WERTES zur Quantität Verzögerung
	 * (Q05) zurück.
	 *
	 * @param quantitaetIntWert der Integer-Wert, darf nicht <code>null</code> sein
	 *
	 * @return die ausführliche Zeichenkette des WERTES zur Quantität Verzögerung
	 *         (Q05)
	 */
	public static String getVerzoegerungWertString(final Integer quantitaetIntWert) {
		assert null != quantitaetIntWert;
		final int i = getValidIntWert(KENNUNG_VERZOEGERUNG, null, quantitaetIntWert);
		return "" + ((i <= 50) ? i : i / 100);
	}

	/**
	 * Liefert die ausführliche Zeichenkette der EINHEIT zur Quantität Verzögerung
	 * (Q05) zurück.
	 *
	 * <p>
	 * Der Rückgabewert hat dabei entweder: 'Minuten' oder 'Stunden'.
	 * </p>
	 *
	 * @param quantitaetIntWert der Integer-Wert, darf nicht <code>null</code> sein
	 *
	 * @return die ausführliche Zeichenkette der EINHEIT zur Quantität Verzögerung
	 *         (Q05)
	 */
	public static String getVerzoegerungEinheitString(final Integer quantitaetIntWert) {
		assert null != quantitaetIntWert;
		final int i = getValidIntWert(KENNUNG_VERZOEGERUNG, null, quantitaetIntWert);
		return (i <= 50) ? "Minuten" : "Stunden";
	}

	/**
	 * Liefert den entsprechenden Integer-Wert zu einer ausführlichen Zeichenkette
	 * zur Quantität Verzögerung (Q05) zurück.
	 *
	 * @param verzoegerungString die Zeichenkette, darf nicht <code>null</code> sein
	 *
	 * @return den entsprechenden Integer-Wert zu einer ausführlichen Zeichenkette
	 *         zur Quantität Verzögerung (Q05)
	 */
	public static int getVerzoegerungInt(final String verzoegerungString) {
		assert null != verzoegerungString;

		boolean minutenModus = true;
		String s = verzoegerungString.trim().toLowerCase();
		if (s.contains("Minuten".toLowerCase())) {
			s = s.replace("Minuten".toLowerCase(), "");
		} else if (s.contains("Stunden".toLowerCase())) {
			s = s.replace("Stunden".toLowerCase(), "");
			minutenModus = false;
		}

		double d = 0;
		try {
			d = new Double(s);
		} catch (final NumberFormatException e) {
			return getSpinnerWert(KENNUNG_VERZOEGERUNG).getDefaultValue();
		}

		if (minutenModus) {
			return getValidIntWert(KENNUNG_VERZOEGERUNG, null, (int) d);
		}
		return getValidIntWert(KENNUNG_VERZOEGERUNG, null, (int) (d * 100));
	}

	/**
	 * Liefert <code>true</code> zurück, wenn es sich bei der gegebenen Quantität um
	 * eine Quantität handelt, welche man mit 5 Bits codieren kann, ansonsten
	 * <code>false</code>.
	 *
	 * @param kennung die Kennung der Quantität, darf nicht <code>null</code> sein
	 *
	 * @return <code>true</code>, wenn es sich bei der gegebenen Quantität um eine
	 *         Quantität handelt, welche man mit 5 Bits codieren kann, ansonsten
	 *         <code>false</code>
	 */
	public static boolean isKleineQuantitaet(final String kennung) {
		assert null != kennung;
		return KENNUNG_ANZAHL_KLEIN.equals(kennung) || KENNUNG_ANZAHL_GROSS.equals(kennung)
				|| KENNUNG_SICHTWEITE.equals(kennung) || KENNUNG_PARKPLATZBELEGUNG.equals(kennung)
				|| KENNUNG_DURCHSCHNITTSGESCHWINDIGKEIT.equals(kennung) || KENNUNG_WINDGESCHWINDIGKEIT.equals(kennung)
				|| KENNUNG_VERZOEGERUNG.equals(kennung);
	}
}
