/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2011 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.bitctrl.modell.util.rdstmc;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;

import de.bsvrz.sys.funclib.bitctrl.modell.Aspekt;
import de.bsvrz.sys.funclib.bitctrl.modell.att.Feld;
import de.bsvrz.sys.funclib.bitctrl.modell.att.WerteBereich;
import de.bsvrz.sys.funclib.bitctrl.modell.att.Zeitstempel;
import de.bsvrz.sys.funclib.bitctrl.modell.metamodellglobal.attribute.AttJaNein;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.attribute.AtlRdsAttribute;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.attribute.AtlRdsEreignis;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.attribute.AtlRdsEreignisDaten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.attribute.AtlRdsLocation;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.attribute.AtlRdsLocationDaten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.attribute.AtlRdsNachrichten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.attribute.AtlRdsNachrichtenDaten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.attribute.AtlRdsVerwaltung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.attribute.AttRdsEreignisTyp;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.attribute.AttRdsLocationMethode;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.attribute.AttRdsNachrichtenKlasse;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.attribute.AttRdsPrioritaet;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.attribute.AttRdsQuelle;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.attribute.AttRdsStatus;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.attribute.AttRdsTMCRichtung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.attribute.AttRdsWichtung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.attribute.AttRdsZustand;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.attribute.AttZahl0Bis65535;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.objekte.RdsMeldung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.parameter.PdRdsMeldung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.parameter.PdRdsMeldung.Daten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmsttmccodes.attribute.AttRdsEmpfehlungsCode;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmsttmccodes.attribute.AttRdsEreignisArt;
import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcglobal.attribute.AttTmcLocationCode;
import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcglobal.objekte.TmcLinie;
import de.bsvrz.sys.funclib.bitctrl.modell.util.ObjektMitZeitBereich;
import de.bsvrz.sys.funclib.bitctrl.modell.util.cache.RdsCache;
import de.bsvrz.sys.funclib.bitctrl.modell.util.cache.RdsMeldungenCache;
import de.bsvrz.sys.funclib.bitctrl.modell.util.cache.TmcLocationCodeCache;

/**
 * Wrapper, welcher für ein RdsMeldung-SystemObjekt sämtliche Daten vorhält.
 *
 * @author BitCtrl Systems GmbH, Andreas Meissner
 *
 * @see RdsMeldung
 */
public class RdsMeldungWrapper extends AbstractWrapper implements ObjektMitZeitBereich {

	/** Separator: '. ' (Punkt). */
	private static final String PUNKT_SEPARATOR = ". ";
	/** Separator: ', ' (Komma). */
	private static final String KOMMA_SEPARATOR = ", ";
	/** Separator: ' ' (Leerzeichen). */
	private static final String LEERZEICHEN_SEPARATOR = " ";

	/** Property-Key für das Dirty-Flag. */
	private static final String PROP_DIRTY = "dirty";

	/**
	 * Maximale Anzahl der erlaubten Bits. Die Maximale Anzahl der Bits entspricht 4
	 * RDS-Gruppen mit je 28 Bits für Free-Form-Content.
	 */
	public static final int BITS_MAX = 4 * 28;

	/** Anzahl der Bits, die für ein Label benötigt werden. */
	private static final int BITS_LABEL = 4;
	/** Anzahl der Bits, die für einen Separator benötigt werden. */
	private static final int BITS_SEPARATOR = BITS_LABEL + 0;
	/** Anzahl der Bits, die für einen Steuercode benötigt werden. */
	private static final int BITS_STEUERCODE = BITS_LABEL + 3;

	/**
	 * Anzahl der Bits, die für die Angabe des expliziten Startzeitpunkts benötigt
	 * werden.
	 */
	private static final int BITS_EXPLIZITE_START_ZEIT = BITS_LABEL + 8;

	/**
	 * Anzahl der Bits, die für die Angabe des expliziten Endzeitpunkts benötigt
	 * werden.
	 */
	private static final int BITS_EXPLIZITE_STOP_ZEIT = BITS_LABEL + 8;

	/** Anzahl der Bits, die für die betroffene Länge benötigt werden. */
	private static final int BITS_BETROFFENE_LAENGE = BITS_LABEL + 5;
	/** Anzahl der Bits, die für 5-Bit Quantitäten benötigt werden. */
	private static final int BITS_QUANTITAET_KLEIN = BITS_LABEL + 5;
	/** Anzahl der Bits, die für 8-Bit Quantitäten benötigt werden. */
	private static final int BITS_QUANTITAET_GROSS = BITS_LABEL + 8;
	/** Anzahl der Bits, die für Empfehlungen benötigt werden. */
	private static final int BITS_EMPFEHLUNG = BITS_LABEL + 8;
	/** Anzahl der Bits, die für zusätzliche Ereignisse benötigt werden. */
	private static final int BITS_ZUSAETLICHES_EREIGNIS = BITS_LABEL + 11;
	/** Anzahl der Bits, welche standardmäßig benötigt werden. */
	private static final int BITS_DEFAULT = BITS_EXPLIZITE_START_ZEIT + BITS_EXPLIZITE_STOP_ZEIT;

	/** Die Properties, welche eine Modifikation der Meldung darstellen. */
	private static final String[] PROPERTIES_MODIFIKATION = { "status", "wichtung", "prioritaet", "zustand",
			"aktivierungsZeit", "ablaufZeit" };

	/** Die Properties, welche einer Erweiterung der Meldung darstellen. */
	private static final String[] PROPERTIES_ERWEITERUNG = { "vorhersage", "strasse", "richtung", "primaereLokation",
			"sekundaereLokation", "ereignisInBeideRichtungen", "ereignisTyp", "ereignisTypen", "ereignisse", /*
																												 * EreignisWrapper:
																												 */
			"kategorie", "dauer", "betroffeneLaenge", "quantitaet", "quantitaetStringWert", "quantitaetIntWert",
			"code" };

	/** Der {@link RdsMeldungenCache}. */
	private final RdsMeldungenCache rdsMeldungenCache;
	/** Der {@link TmcLocationCodeCache}. */
	private final TmcLocationCodeCache tmcLocationCodeCache;
	/** Der {@link RdsCache}. */
	private final RdsCache rdsCache;

	/** Die in diesem Wrapper gekapselte {@link RdsMeldung}. */
	private final RdsMeldung rdsMeldung;
	/** Das zu der gekapseltem {@link RdsMeldung} gehörige Datum. */
	private final Daten datum;
	/** Der zu der gekapseltem {@link RdsMeldung} gehörige Aspekt. */
	private final Aspekt aspekt;

	/**
	 * Flag, welches angibt, ob die RDS-Meldung gueltig ist (also vollständig
	 * geladen werden konnte).
	 */
	private boolean gueltig;

	/** Der aktuelle Fehler-Text. */
	private final StringBuilder errorText;

	/** Die Versionsnummer. */
	private int versionsNummer;
	/** Das Dirty-Flag. */
	private boolean dirty;

	/**
	 * Flag, welches angibt, ob es sich um eine neu anzulegenden RDS-Meldung
	 * handelt.
	 */
	private final boolean neuesElement;

	/** Flag, welches angibt, ob der Inhalt geändert wurde. */
	private boolean inhaltGeaendert;

	/**
	 * Die Anzahl der für diese RDS-Meldung benötigten Anzahl von Bits im
	 * Free-Form-Content der RDS-Gruppen.
	 */
	private int bits;

	// Meldungstext

	/** Der aktuelle Meldungstext. */
	private String meldungsText;
	/** Der aus den Nachrichtendaten geladene Meldungstext. */
	private String meldungsTextGeladen;
	/** Die aus den Nachrichtendaten geladene Straßennummer. */
	private String strassenNummerTextGeladen;
	/** Die aus den Nachrichtendaten geladene Segmentinformation. */
	private String segmentTextGeladen;
	/** Die aus den Nachrichtendaten geladene Primärlokation. */
	private String primaereLokationTextGeladen;
	/** Die aus den Nachrichtendaten geladene Sekundärlokation. */
	private String sekundaereLokationTextGeladen;

	// Verwaltuns-Informationen

	/** Der Status. */
	private AttRdsStatus status;
	/** Die Wichtung (0..100). */
	private byte wichtung;
	/** Die Priorität/Dringlichkeit. */
	private AttRdsPrioritaet prioritaet;
	/** Die abhängig von den gesetzten Ereignis-Codes empfohlene Priorität. */
	private AttRdsPrioritaet empfohlenePrioritaet;
	/** Der Zustand. */
	private AttRdsZustand zustand;
	/**
	 * Flag, welches angibt, ob es sich bei dieser Meldung um eine Vorhersage
	 * handelt.
	 */
	private boolean vorhersage;

	// Zeit-Informationen

	/** Die Aktivierungszeit (Start der zeitlichen Gültigkeit). */
	private Date aktivierungsZeit;
	/** Die Ablaufzeit (End der zeitlichen Gültigkeit). */
	private Date ablaufZeit;

	// Location Informationen

	/** Die Straße. */
	private TmcLinieWrapper strasse;
	/** Die Primärlokation. */
	private TmcPunktWrapper primaereLokation;
	/** Die Sekundärlokation. */
	private TmcPunktWrapper sekundaereLokation;
	/** Die Lokationen. */
	private List<TmcPunktWrapper> tmcLokationen;
	/** Die Ausdehnung. */
	private int ausdehnung;

	// Ereignis Informationen

	/**
	 * Flag, welches angibt, ob diese Meldung für beide Straßenrichtungen gelten
	 * soll.
	 */
	private boolean ereignisInBeideRichtungen;

	/**
	 * Der abhängig von den gesetzten Ereignis-Codes empfohlene Wert für das Flag,
	 * welches angibt,ob diese Meldung für beide Straßenrichtungen gelten soll.
	 */
	private boolean empfohleneEreignisInBeideRichtungen;

	/**
	 * Der Ereignis-Typ.
	 *
	 * @deprecated Dieses Feld wird nur noch aus Kompatibilitätsgründen behalten.
	 *             Bei einer RDS-Meldungen können mehrere Ereignistypen gesetzt
	 *             werden (siehe {@link #ereignisTypen}).
	 */
	@Deprecated
	private AttRdsEreignisTyp ereignisTyp;

	/** Die Ereignis-Typen. */
	private List<AttRdsEreignisTyp> ereignisTypen;

	/** Der Ereignisse. */
	private List<EreignisWrapper> ereignisse;

	/**
	 * Der Konstruktor zum Anlegen einer neuen RDS-Meldung.
	 *
	 * @param cache der Cache, welcher diese RDS-Meldung verwaltet, darf nicht
	 *              <code>null</code> sein
	 */
	public RdsMeldungWrapper(final RdsMeldungenCache cache) {

		assert null != cache;

		rdsMeldungenCache = cache;
		tmcLocationCodeCache = rdsMeldungenCache.getTmcLocationCodeCache();
		rdsCache = rdsMeldungenCache.getRdsCache();

		rdsMeldung = null;
		datum = null;
		aspekt = null;
		gueltig = true;
		errorText = new StringBuilder();

		versionsNummer = 0;
		neuesElement = true;
		dirty = true;
		inhaltGeaendert = false;
		bits = BITS_DEFAULT;

		status = AttRdsStatus.ZUSTAND_0_UNBEKANNT;
		prioritaet = AttRdsPrioritaet.ZUSTAND_0_NORMAL;
		empfohlenePrioritaet = AttRdsPrioritaet.ZUSTAND_0_NORMAL;
		wichtung = (byte) WerteBereich.getWerteBereich(AttRdsWichtung.class).getMinimum();
		zustand = AttRdsZustand.ZUSTAND_0_NICHT_QUITTIERT;
		vorhersage = false;

		aktivierungsZeit = new Date();
		ablaufZeit = new Date();

		meldungsText = "unbekannt";
		meldungsTextGeladen = "";
		strassenNummerTextGeladen = "";
		segmentTextGeladen = "";
		primaereLokationTextGeladen = "unbekannt";
		sekundaereLokationTextGeladen = "unbekannt";

		strasse = null;
		primaereLokation = null;
		sekundaereLokation = null;
		tmcLokationen = null;
		ausdehnung = 0;

		ereignisInBeideRichtungen = false;
		empfohleneEreignisInBeideRichtungen = false;
		ereignisTyp = AttRdsEreignisTyp.ZUSTAND_0_STOERUNG;
		ereignisTypen = new ArrayList<>();
		ereignisTypen.add(ereignisTyp);
		ereignisse = new ArrayList<>();
		ereignisse.add(new EreignisWrapper(this));
	}

	/**
	 * Der Konstruktor zum Anlegen einer neuen RDS-Meldung.
	 *
	 * @param cache                  der Cache, welcher diese RDS-Meldung verwaltet,
	 *                               darf nicht <code>null</code> sein
	 * @param tmcLocationCodeWrapper der TMC-Lokation, für welche diese RDS-Meldung
	 *                               definiert sein soll, darf <code>null</code>
	 *                               sein
	 *
	 */
	public RdsMeldungWrapper(final RdsMeldungenCache cache, final TmcLocationCodeWrapper tmcLocationCodeWrapper) {
		this(cache);
		if (tmcLocationCodeWrapper != null) {

			if (tmcLocationCodeWrapper instanceof TmcPunktWrapper) {
				final TmcPunktWrapper tmcPunktWrapper = (TmcPunktWrapper) tmcLocationCodeWrapper;
				strasse = tmcLocationCodeCache.getTmcLinieWrapper(tmcPunktWrapper.getIstTeilvonTmcLinie(),
						AttRdsTMCRichtung.ZUSTAND_0_POSITIV);
				primaereLokation = tmcPunktWrapper;
			} else if (tmcLocationCodeWrapper instanceof TmcLinieWrapper) {
				final TmcLinieWrapper tmcLinieWrapper = (TmcLinieWrapper) tmcLocationCodeWrapper;
				strasse = tmcLinieWrapper;
				final List<TmcPunktWrapper> tmcPunkteWrapper = tmcLocationCodeCache.getTmcPunktWrapper(tmcLinieWrapper);
				if ((tmcPunkteWrapper != null) && !tmcPunkteWrapper.isEmpty()) {
					primaereLokation = tmcPunkteWrapper.get(0);
				}
			}
			if (primaereLokation != null) {
				sekundaereLokation = primaereLokation;
				tmcLokationen = new ArrayList<>();
				tmcLokationen.add(primaereLokation);
			}
		}
	}

	/**
	 * Der Konstruktor.
	 *
	 * @param cache      der Cache, welcher diese RDS-Meldung verwaltet, darf nicht
	 *                   <code>null</code> sein
	 * @param rdsMeldung die rdsMeldung, darf nicht <code>null</code> sein
	 * @param datum      das unter einem gültigen Aspekt (generiert, versendet oder
	 *                   empfangen) geladene Datum der dazugehörigen RDS-Meldung,
	 *                   darf nicht <code>null</code> sein
	 */
	public RdsMeldungWrapper(final RdsMeldungenCache cache, final RdsMeldung rdsMeldung, final Daten datum) {
		final long aspektId = datum.dGetAspekt().getId();

		assert null != cache;
		assert null != rdsMeldung;
		assert (aspektId == PdRdsMeldung.Aspekte.RdsGeneriert.getId())
				|| (aspektId == PdRdsMeldung.Aspekte.RdsSenden.getId())
				|| (aspektId == PdRdsMeldung.Aspekte.RdsVersendet.getId())
				|| (aspektId == PdRdsMeldung.Aspekte.RdsEmpfangen.getId());

		errorText = new StringBuilder();

		rdsMeldungenCache = cache;
		tmcLocationCodeCache = rdsMeldungenCache.getTmcLocationCodeCache();
		rdsCache = rdsMeldungenCache.getRdsCache();

		this.rdsMeldung = rdsMeldung;
		this.datum = datum;
		aspekt = datum.dGetAspekt();
		neuesElement = false;
		inhaltGeaendert = false;
		bits = BITS_DEFAULT;
		gueltig = true;

		// Lese Verwaltungs-Informationen
		versionsNummer = 0;
		status = AttRdsStatus.ZUSTAND_0_UNBEKANNT;
		prioritaet = AttRdsPrioritaet.ZUSTAND_0_NORMAL;
		empfohlenePrioritaet = AttRdsPrioritaet.ZUSTAND_0_NORMAL;
		wichtung = (byte) WerteBereich.getWerteBereich(AttRdsWichtung.class).getMinimum();
		zustand = AttRdsZustand.ZUSTAND_0_NICHT_QUITTIERT;

		aktivierungsZeit = new Date(0);
		ablaufZeit = new Date(0);

		if ((datum.getVersion() != null) && (datum.getVersion().getVerwaltungsInformationen() != null)) {
			final AtlRdsVerwaltung rdsVerwaltung = datum.getVersion().getVerwaltungsInformationen();

			// Versionsnummer
			if (rdsVerwaltung.getVersionsNummer() != null) {
				versionsNummer = rdsVerwaltung.getVersionsNummer().getValue();
			} else {
				setzeUngueltig("Versionsnummer");
			}

			// Status
			if (rdsVerwaltung.getStatus() != null) {
				status = rdsVerwaltung.getStatus();
			} else {
				setzeUngueltig("Status");
			}

			// Priorität
			if (rdsVerwaltung.getPrioritaet() != null) {
				prioritaet = rdsVerwaltung.getPrioritaet();
			} else {
				setzeUngueltig("Priorität");
			}

			// Wichtung
			final AttRdsWichtung rdsWichtung = rdsVerwaltung.getWichtung();
			if (rdsWichtung != null) {
				wichtung = rdsWichtung.getValue();
			} else {
				setzeUngueltig("Wichtung");
			}

			// Zustand
			if (rdsVerwaltung.getZustand() != null) {
				zustand = rdsVerwaltung.getZustand();
			} else {
				setzeUngueltig("Zustand");
			}

			// Aktivierungszeit
			final Zeitstempel rdsAktivierungsZeit = rdsVerwaltung.getAktivierungsZeit();
			if (rdsAktivierungsZeit != null) {
				aktivierungsZeit = new Date(rdsAktivierungsZeit.getTime());
			} else {
				setzeUngueltig("Aktivierungszeit");
			}

			// Ablaufzeit
			final Zeitstempel rdsAblaufZeit = rdsVerwaltung.getAblaufZeit();
			if (rdsAblaufZeit != null) {
				ablaufZeit = new Date(rdsAblaufZeit.getTime());
			} else {
				setzeUngueltig("Ablaufzeit");
			}
		} else {
			setzeUngueltig("Verwaltungsinformationen");
		}

		meldungsTextGeladen = "";
		if ((datum.getVersion() != null) && (datum.getVersion().getVerkehrsInformationen() != null)
				&& (datum.getVersion().getVerkehrsInformationen().getNachrichten() != null)) {

			final AtlRdsNachrichten nachrichten = datum.getVersion().getVerkehrsInformationen().getNachrichten();

			if (nachrichten.getNachrichtenDaten() != null) {

				strassenNummerTextGeladen = "";
				segmentTextGeladen = "";
				String inhalt = "";
				String freierText = "";

				for (final AtlRdsNachrichtenDaten nachrichtenDatum : nachrichten.getNachrichtenDaten()) {
					if ((nachrichtenDatum == null) || (nachrichtenDatum.getNachrichtenKlasse() == null)) {
						continue;
					}

					String text = nachrichtenDatum.getNachrichtenText();
					if (text != null) {
						text = text.trim();
						if (AttRdsNachrichtenKlasse.ZUSTAND_0_STRASSENNUMMER.getValue() == nachrichtenDatum
								.getNachrichtenKlasse().getValue()) {
							strassenNummerTextGeladen = text;
						} else if (AttRdsNachrichtenKlasse.ZUSTAND_1_SEGMENT.getValue() == nachrichtenDatum
								.getNachrichtenKlasse().getValue()) {
							segmentTextGeladen = text;
						} else if (AttRdsNachrichtenKlasse.ZUSTAND_2_INHALT.getValue() == nachrichtenDatum
								.getNachrichtenKlasse().getValue()) {
							inhalt = text;
							if (inhalt.endsWith(".") || inhalt.endsWith(",")) {
								inhalt = inhalt.substring(0, inhalt.length() - 1);
								inhalt = inhalt.trim();
							}
						} else if (AttRdsNachrichtenKlasse.ZUSTAND_3_FREIER_TEXT.getValue() == nachrichtenDatum
								.getNachrichtenKlasse().getValue()) {
							freierText = text;
							if (freierText.startsWith(".") || freierText.startsWith(",")) {
								freierText = freierText.substring(1);
								freierText = freierText.trim();
							}
							if (freierText.endsWith(".") || freierText.endsWith(",")) {
								freierText = freierText.substring(0, freierText.length() - 1);
								freierText = freierText.trim();
							}
						}
					}
				}

				final StringBuffer buf = new StringBuffer(strassenNummerTextGeladen);
				if (!"".equals(segmentTextGeladen)) {
					if (buf.length() > 0) {
						buf.append(LEERZEICHEN_SEPARATOR);
					}
					buf.append(segmentTextGeladen);
				}
				if (!"".equals(inhalt)) {
					if (buf.length() > 0) {
						buf.append(LEERZEICHEN_SEPARATOR);
					}
					buf.append(inhalt);
				}
				if (!"".equals(freierText)) {
					if (buf.length() > 0) {
						buf.append(PUNKT_SEPARATOR);
					}
					buf.append(freierText);
				}
				if (buf.length() > 0) {
					buf.append(".");
				}
				meldungsTextGeladen = buf.toString();

				// Meldungstext enthält manchmal \n und/oder \t
				meldungsTextGeladen = meldungsTextGeladen.replaceAll("\\s+", " ");
			} else {
				setzeUngueltig("Nachrichtendaten");
			}
		} else {
			setzeUngueltig("Nachrichten");
		}

		// Lese Lokationen
		strasse = null;
		primaereLokation = null;
		primaereLokationTextGeladen = "unbekannt";
		sekundaereLokation = null;
		sekundaereLokationTextGeladen = "unbekannt";
		tmcLokationen = new ArrayList<>();
		ausdehnung = 0;

		AttRdsTMCRichtung tmcRichtung = AttRdsTMCRichtung.ZUSTAND_0_POSITIV;
		if ((datum.getVersion() != null) && (datum.getVersion().getVerkehrsInformationen() != null)
				&& (datum.getVersion().getVerkehrsInformationen().getLocationDaten() != null)) {

			final AtlRdsLocationDaten locationDaten = datum.getVersion().getVerkehrsInformationen().getLocationDaten();

			// Richtung
			tmcRichtung = locationDaten.getTMCRichtung();

			if (tmcRichtung != null) {
				// Beachte: Richtung wird beim Einlesen vertauscht!
				tmcRichtung = tmcRichtung.getValue() == AttRdsTMCRichtung.ZUSTAND_0_POSITIV.getValue()
						? AttRdsTMCRichtung.ZUSTAND_1_NEGATIV
						: AttRdsTMCRichtung.ZUSTAND_0_POSITIV;
			} else {
				tmcRichtung = AttRdsTMCRichtung.ZUSTAND_0_POSITIV;
				setzeUngueltig("Richtung");
			}

			// Lokationen
			AttRdsLocationMethode locationMethode = locationDaten.getLocationMethode();
			final AttZahl0Bis65535 locationExtent = locationDaten.getLocationExtent();

			if (locationMethode == null) {
				throw new RuntimeException("Location Methode nicht gesetzt.");
			}
			if (locationMethode == AttRdsLocationMethode.ZUSTAND_0_ERSTE_FELDPOSITION_IST_PRIMAERE_LOCATION_UND_LETZTE_FELDPOSITION_IST_SEKUNDAERE_LOCATION) {
				locationMethode = AttRdsLocationMethode.ZUSTAND_2_PRIMAERE_LOCATION_UND_SEKUNDAERE_LOCATION_SIND_GUELTIG;
			}
			if ((locationMethode != AttRdsLocationMethode.ZUSTAND_1_PRIMAERE_LOCATION_UND_EXTENT_SIND_GUELTIG)
					&& (locationMethode != AttRdsLocationMethode.ZUSTAND_2_PRIMAERE_LOCATION_UND_SEKUNDAERE_LOCATION_SIND_GUELTIG)) {
				throw new RuntimeException(
						"Location Methode '" + locationMethode.toString() + "' wird nicht unterstützt.");
			}
			if ((locationMethode == AttRdsLocationMethode.ZUSTAND_1_PRIMAERE_LOCATION_UND_EXTENT_SIND_GUELTIG)
					&& (locationExtent == null)) {
				throw new RuntimeException("Location Extend nicht gesetzt");
			}

			final Feld<AtlRdsLocation> rdsLocations = locationDaten.getRDSLocation();
			if ((rdsLocations != null) && !rdsLocations.isEmpty()) {
				for (final AtlRdsLocation rdsLocation : rdsLocations) {
					final AttTmcLocationCode attTmcLocationCode = rdsLocation.getLocationCode();
					if (attTmcLocationCode != null) {
						final TmcPunktWrapper wrapper = tmcLocationCodeCache.getTmcPunktWrapper(attTmcLocationCode);
						if (wrapper != null) {
							// Beachte: TMC-Lokation werden in umgedrehter
							// Reihenfolge eingelesen!
							tmcLokationen.add(0, wrapper);
						}
					}
				}
			} else {
				setzeUngueltig("RDS-Lokationen");
			}

			if (!tmcLokationen.isEmpty()) {

				// Primäre Lokation
				primaereLokation = tmcLokationen.get(0);
				primaereLokationTextGeladen = primaereLokation.getTmcName();

				// Straße
				final TmcLinie tmcLinie = primaereLokation.getIstTeilvonTmcLinie();
				if (tmcLinie != null) {
					final TmcLinieWrapper tmcLinieWrapper = tmcLocationCodeCache.getTmcLinieWrapper(tmcLinie,
							tmcRichtung);
					if (tmcLinieWrapper != null) {
						strasse = tmcLinieWrapper;
					}
				}

				// Sekundäre Lokation
				if (locationMethode != AttRdsLocationMethode.ZUSTAND_1_PRIMAERE_LOCATION_UND_EXTENT_SIND_GUELTIG) {
					int extent = locationExtent.getValue();
					if (extent > Konstanten.MAXIMALE_ANZAHL_NACHFOLGER_LOKATIONEN) {
						extent = Konstanten.MAXIMALE_ANZAHL_NACHFOLGER_LOKATIONEN;
					}
					tmcLokationen = new ArrayList<>();
					tmcLokationen.add(primaereLokation);
					tmcLokationen.addAll(tmcLocationCodeCache.getNachfolgendeTmcPunktWrapper(primaereLokation,
							tmcRichtung, extent, true));
				}
				ausdehnung = tmcLokationen.size() - 1;
				sekundaereLokation = tmcLokationen.get(ausdehnung);
				sekundaereLokationTextGeladen = sekundaereLokation.getTmcName();
			} else {
				tmcLokationen = null;
			}
		} else {
			setzeUngueltig("Location-Informationen");
		}

		// Überprüfe Gültigkeit der Lokationen
		if (strasse == null) {
			setzeUngueltig("Straße");
		}
		if (primaereLokation == null) {
			setzeUngueltig("Primärlokation");
		}
		if (sekundaereLokation == null) {
			setzeUngueltig("Sekundärlokation");
		}
		if ((primaereLokation != null) && (sekundaereLokation != null)) {
			final List<TmcPunktWrapper> moeglicheSekundaerLokationen = new ArrayList<>();
			moeglicheSekundaerLokationen.add(primaereLokation);
			moeglicheSekundaerLokationen.addAll(tmcLocationCodeCache.getNachfolgendeTmcPunktWrapper(primaereLokation,
					tmcRichtung, Konstanten.MAXIMALE_ANZAHL_NACHFOLGER_LOKATIONEN, true));
			if (!moeglicheSekundaerLokationen.contains(sekundaereLokation)) {
				strasse = null;
				primaereLokation = null;
				sekundaereLokation = null;
				setzeUngueltig("Sekundärlokation");
			}
		}

		// Lese Ereignis-Informationen
		ereignisInBeideRichtungen = false;
		empfohleneEreignisInBeideRichtungen = false;
		ereignisTyp = AttRdsEreignisTyp.ZUSTAND_0_STOERUNG;
		ereignisTypen = new ArrayList<>();
		ereignisse = new ArrayList<>();

		if ((datum.getVersion() != null) && (datum.getVersion().getVerkehrsInformationen() != null)
				&& (datum.getVersion().getVerkehrsInformationen().getEreignis() != null)) {

			final AtlRdsEreignis rdsEreignis = datum.getVersion().getVerkehrsInformationen().getEreignis();

			// Flag: Ereignis in beiden Richtungen
			if (rdsEreignis.getEreignisInBeidenRichtungen() != null) {
				ereignisInBeideRichtungen = rdsEreignis.getEreignisInBeidenRichtungen()
						.getValue() == AttJaNein.ZUSTAND_1_JA.getValue();
			} else {
				setzeUngueltig("Räumliche Gültigkeit");
			}

			// Ereignistyp
			final Feld<AttRdsEreignisTyp> typen = rdsEreignis.getEreignisTyp();
			if ((typen != null) && !typen.isEmpty()) {
				for (final AttRdsEreignisTyp typ : typen) {
					ereignisTypen.add(typ);
					ereignisTyp = typ;
				}
			} else {
				setzeUngueltig("Ereignistyp");
			}

			// Ereignisdaten
			final Feld<AtlRdsEreignisDaten> ereignisDaten = rdsEreignis.getEreignisDaten();
			if ((ereignisDaten != null) && !ereignisDaten.isEmpty()) {
				for (final AtlRdsEreignisDaten ereignisDatum : ereignisDaten) {
					ereignisse.add(new EreignisWrapper(this, ereignisDatum));
				}
			} else {
				setzeUngueltig("Ereignisdaten");
			}
		} else {
			setzeUngueltig("Ereignis-Informationen");
		}

		// zusätzliche Attribute
		if ((datum.getVersion() != null) && (datum.getVersion().getVerkehrsInformationen() != null)
				&& (datum.getVersion().getVerkehrsInformationen().getZusatzAttribute() != null)) {

			final AtlRdsAttribute zusatzAttribute = datum.getVersion().getVerkehrsInformationen().getZusatzAttribute();
			vorhersage = zusatzAttribute.getVorhersage() == AttJaNein.ZUSTAND_1_JA;
		} else {
			setzeUngueltig("Zusatzattribute");
		}

		// letzte Initialisierungen
		aktualisiereEreignisAbhaengigeDaten(false, true);
		setMeldungsText();
		setAusdehnung();
		dirty = false;
	}

	/**
	 * Setzt diese Meldung ungültig und erweitert die Fehlermeldung.
	 *
	 * @param ungueltigesDatum der Name des Datums, welches nicht geladen werden
	 *                         konnte
	 */
	/* package */void setzeUngueltig(final String ungueltigesDatum) {
		if (errorText.length() == 0) {
			errorText.append(
					"Folgende Daten konnten nicht geladen bzw. zugewiesen werden und werden somit nicht im Editor dargestellt: ");
		} else {
			errorText.append(", ");
		}
		errorText.append(ungueltigesDatum);
		gueltig = false;
	}

	/**
	 * Der Konstruktor.
	 *
	 * @param wrapper ein anderer RdsMeldungWrapper
	 */
	public RdsMeldungWrapper(final RdsMeldungWrapper wrapper) {

		rdsMeldungenCache = wrapper.rdsMeldungenCache;
		tmcLocationCodeCache = wrapper.tmcLocationCodeCache;
		rdsCache = wrapper.rdsCache;

		rdsMeldung = wrapper.rdsMeldung;
		datum = wrapper.datum;
		aspekt = wrapper.aspekt;
		neuesElement = wrapper.neuesElement;
		dirty = wrapper.dirty;
		inhaltGeaendert = wrapper.inhaltGeaendert;
		bits = wrapper.bits;
		gueltig = wrapper.gueltig;
		errorText = new StringBuilder(wrapper.errorText);

		versionsNummer = wrapper.versionsNummer;
		status = AttRdsStatus.getZustand(wrapper.status.getValue());
		prioritaet = AttRdsPrioritaet.getZustand(wrapper.prioritaet.getValue());
		empfohlenePrioritaet = AttRdsPrioritaet.getZustand(wrapper.empfohlenePrioritaet.getValue());
		zustand = AttRdsZustand.getZustand(wrapper.zustand.getValue());
		wichtung = wrapper.wichtung;
		vorhersage = wrapper.vorhersage;

		aktivierungsZeit = new Date(wrapper.aktivierungsZeit.getTime());
		ablaufZeit = new Date(wrapper.ablaufZeit.getTime());

		meldungsText = new String(wrapper.meldungsText);
		meldungsTextGeladen = new String(wrapper.meldungsTextGeladen);
		strassenNummerTextGeladen = new String(wrapper.strassenNummerTextGeladen);
		segmentTextGeladen = new String(wrapper.segmentTextGeladen);
		primaereLokationTextGeladen = new String(wrapper.primaereLokationTextGeladen);
		sekundaereLokationTextGeladen = new String(wrapper.sekundaereLokationTextGeladen);

		strasse = wrapper.strasse;
		primaereLokation = wrapper.primaereLokation;
		sekundaereLokation = wrapper.sekundaereLokation;
		if (wrapper.tmcLokationen != null) {
			tmcLokationen = new ArrayList<>(wrapper.tmcLokationen);
		} else {
			tmcLokationen = null;
		}
		ausdehnung = wrapper.ausdehnung;

		ereignisInBeideRichtungen = wrapper.ereignisInBeideRichtungen;
		empfohleneEreignisInBeideRichtungen = wrapper.empfohleneEreignisInBeideRichtungen;
		ereignisTyp = AttRdsEreignisTyp.getZustand(wrapper.ereignisTyp.getValue());
		ereignisTypen = new ArrayList<>(wrapper.ereignisTypen.size());
		for (final AttRdsEreignisTyp typ : wrapper.ereignisTypen) {
			if (!ereignisTypen.contains(typ)) {
				ereignisTypen.add(typ);
			}
		}
		ereignisse = new ArrayList<>(wrapper.ereignisse.size());
		for (final EreignisWrapper ereignisWrapper : wrapper.ereignisse) {
			ereignisse.add(new EreignisWrapper(this, ereignisWrapper));
		}
	}

	/**
	 * Liefert das in diesem Objekt gekapselte RdsMeldung-Systemobjekt zurück.
	 *
	 * @return das in diesem Objekt gekapselte RdsMeldung-Systemobjekt, kann
	 *         <code>null</code> sein
	 */
	public RdsMeldung getRdsMeldung() {
		return rdsMeldung;
	}

	/**
	 * Liefert das zugrundeliegende Datum dieser RDS-Meldung zurück.
	 *
	 * @return das zugrundeliegende Datum dieser RDS-Meldung, kann <code>null</code>
	 *         sein
	 */
	public Daten getDatum() {
		return datum;
	}

	/**
	 * Liefert den Aspekt des in diesem RdsMeldungWrapper geladenden Datensatzes
	 * zurück.
	 *
	 * @return den Aspekt des in diesem RdsMeldungWrapper geladenden Datensatzes,
	 *         kann <code>null</code> sein
	 */
	public Aspekt getAspekt() {
		return aspekt;
	}

	/**
	 * Liefert den Fehlertext zurück. Der Fehlertext existiert nur, wenn diese
	 * Meldung ungültig ist.
	 *
	 * @return den Fehlertext, niemals <code>null</code>, kann bzw. sollte leer sein
	 *
	 * @see #isGueltig()
	 */
	public String getErrorText() {
		return errorText.toString();
	}

	/**
	 * Liefert die aktuelle Versionsnummer dieser RDS-Meldung zurück.
	 *
	 * @return die aktuelle Versionsnummer dieser RDS-Meldung
	 */
	public int getVersionsNummer() {
		return versionsNummer;
	}

	/**
	 * Liefert die Quelle dieser RDS-Meldung zurück. Die Quelle gibt an, ob eine
	 * RDS-Meldung "automatisch" (durch die SW-Einheit RDS/TMC Meldungserstellung)
	 * oder "manuell" (durch den Benutzer über die Benutzerschnittstelle) angelegt
	 * wurde.
	 *
	 * @return die Quelle dieser RDS-Meldung
	 */
	public AttRdsQuelle getQuelle() {
		if (datum != null) {
			return datum.getVersion().getVerwaltungsInformationen().getQuelle();
		}
		return AttRdsQuelle.ZUSTAND_0_MANUELL;
	}

	/**
	 * Liefert den aktuellen Dirty-State zurück.
	 *
	 * @return den aktuellen Dirty-State
	 */
	public boolean isDirty() {
		if (isEditierbar()) {
			return dirty;
		}
		return false;
	}

	/**
	 * Setzt den Dirty-State dieses Wrappers.
	 *
	 * @param neuerWert der neue Dirty-State
	 */
	public void setDirty(final boolean neuerWert) {
		if (dirty != neuerWert) {
			final boolean alterWert = dirty;
			dirty = isEditierbar() ? neuerWert : false;
			firePropertyChange(PROP_DIRTY, alterWert, neuerWert);
		}
	}

	/**
	 * Liefert true zurück, wenn es sich bei diesem Wrapper, um eine neu anzulegende
	 * Engstelle handelt, ansonsten false.
	 *
	 * @return true, wenn es sich bei diesem Wrapper, um eine neu anzulegende
	 *         Engstelle handelt, ansonsten false
	 */
	public boolean isNeuesElement() {
		return neuesElement;
	}

	/**
	 * Liefert den Namen der primären Lokation dieser RDS-Meldung zurück.
	 *
	 * @return den Namen der primären Lokation dieser RDS-Meldung, niemals
	 *         <code>null</code>
	 */
	public String getPrimaereLokationText() {
		if (primaereLokation != null) {
			return primaereLokation.getTmcName();
		}
		return primaereLokationTextGeladen;
	}

	/**
	 * Liefert den Namen der sekundären Lokation dieser RDS-Meldung zurück.
	 *
	 * @return den Namen der sekundären Lokation dieser RDS-Meldung, niemals
	 *         <code>null</code>
	 */
	public String getSekundaereLokationText() {
		if (sekundaereLokation != null) {
			return sekundaereLokation.getTmcName();
		}
		return sekundaereLokationTextGeladen;
	}

	/**
	 * Liefert eine Zeichenkette zurück, welcher die Straße dieser RDS-Meldung
	 * beschreibt.
	 *
	 * <p>
	 * Diese Zeichenkette setzt sich zusammen aus der Straßennummer und dem
	 * Segment-Text dieser RDS-Meldung.
	 * </p>
	 *
	 * @return eine Zeichenkette, welcher die Straße dieser RDS-Meldung beschreibt
	 *
	 * @see #getStrassenNummerText()
	 * @see #getSegmentText()
	 */
	public String getStrassenText() {
		final StringBuffer buf = new StringBuffer();
		final String strassenNummerText = getStrassenNummerText();
		if (!"".equals(strassenNummerText)) {
			buf.append(strassenNummerText);
		}
		final String segmentText = getSegmentText();
		if (!"".equals(segmentText)) {
			if (buf.length() > 0) {
				buf.append(" ");
			}
			buf.append(segmentText);
		}
		final String ergebnis = buf.toString();
		if ("".equals(ergebnis)) {
			return "unbekannt";
		}
		return ergebnis;
	}

	/**
	 * Liefert die Stra&szlig;ennummer der Stra&szlig;e zur&uuml;ck, auf welche
	 * diese RDS-Meldung sich bezieht.
	 *
	 * @return die Stra&szlig;ennummer der Stra&szlig;e, auf welche diese
	 *         RDS-Meldung sich bezieht, niemals <code>null</code>
	 */
	public String getStrassenNummerText() {
		if ((strassenNummerTextGeladen != null) && !"".equals(strassenNummerTextGeladen)) {
			return strassenNummerTextGeladen;

		}
		// if (strasse != null && isEditierbar()) {
		if (strasse != null) {
			final String strassenNummer = strasse.getTmcStrassenNummer();
			if (strassenNummer != null) {
				return strassenNummer;
			}
		}
		return "";
	}

	/**
	 * Liefert das Segment (&lt;Knoten&gt; Richtung &lt;Knoten&gt;) zur&uuml;ck, auf
	 * welchem diese RDS-Meldung liegt.
	 *
	 * @return das Segment, auf welchem diese RDS-Meldung liegt, niemals
	 *         <code>null</code>
	 */
	public String getSegmentText() {
		if ((segmentTextGeladen != null) && !"".equals(segmentTextGeladen)) {
			return segmentTextGeladen;
		}

		if (strasse != null) {
			String tmcErsterName = strasse.getTmcErsterName();
			String tmcZweiterName = strasse.getTmcZweiterName();
			final AttRdsTMCRichtung richtung = strasse.getRichtung();
			if ((tmcErsterName != null) && (tmcZweiterName != null) && (richtung != null)) {
				tmcErsterName = tmcErsterName.trim();
				tmcZweiterName = tmcZweiterName.trim();

				TmcLinieWrapper strasse2 = null;
				if ((sekundaereLokation != null) && (sekundaereLokation.getIstTeilvonTmcLinie() != null)) {
					strasse2 = tmcLocationCodeCache.getTmcLinieWrapper(sekundaereLokation.getIstTeilvonTmcLinie(),
							richtung);
				}
				if ((strasse2 != null) && !strasse.equals(strasse2)) {
					final String alternativerZweiterNamer = strasse2.getTmcZweiterName();
					if (alternativerZweiterNamer != null) {
						tmcZweiterName = alternativerZweiterNamer.trim();
					}
				}

				if (richtung == AttRdsTMCRichtung.ZUSTAND_0_POSITIV) {
					return new StringBuffer(tmcErsterName.trim()).append(" Richtung ").append(tmcZweiterName)
							.toString();
				}
				return new StringBuffer(tmcZweiterName.trim()).append(" Richtung ").append(tmcErsterName).toString();
			}
		}
		return "";
	}

	/**
	 * Liefert eine Zeichenkette zurück, welche die Ausdehnung dieser RDS-Meldung
	 * beschreibt.
	 *
	 * @return eine Zeichenkette, welche die Ausdehnung dieser RDS-Meldung
	 *         beschreibt niemals <code>null</code>
	 */
	public String getAusdehnungText() {
		final StringBuffer buf = new StringBuffer();
		if ((primaereLokation != null) && (primaereLokation.getTmcName() != null) && (sekundaereLokation != null)
				&& (sekundaereLokation.getTmcName() != null)) {

			if (primaereLokation != sekundaereLokation) {
				buf.append("zwischen ").append(primaereLokation.getTmcName()).append(" und ")
						.append(sekundaereLokation.getTmcName()).toString();
			} else {
				buf.append(primaereLokation.getTmcName());
			}
			if (ereignisInBeideRichtungen) {
				buf.append(" in beiden Richtungen");
			}
		}
		return buf.toString();
	}

	/**
	 * Liefert eine Zeichenkette zurück, welche die Ereignisse dieser RDS-Meldung
	 * beschreibt.
	 *
	 * @return eine Zeichenkette, welche die Ereignisse dieser RDS-Meldung
	 *         beschreibt niemals <code>null</code>
	 */
	public String getEreignisText() {
		if ((getEreignisse() != null) && !getEreignisse().isEmpty()) {
			final StringBuilder buf = new StringBuilder();
			for (final Iterator<EreignisWrapper> iter = getEreignisse().iterator(); iter.hasNext();) {
				final EreignisWrapper ereignisWrapper = iter.next();
				final String ereignisText = ereignisWrapper.getText();
				if ((ereignisText == null) || "".equals(ereignisText)) {
					continue;
				}
				buf.append(ereignisText);
				if (iter.hasNext()) {
					buf.append(KOMMA_SEPARATOR);
				}
			}
			return buf.toString();
		}
		return "";
	}

	/**
	 * Liefert den Nachrichten-Inhalt dieser RDS-Meldung zurück. Der
	 * Nachrichten-Inhalt setzt sich zusammen aus dem Ausdehnungs-Text und dem
	 * Ereignis-Text.
	 *
	 * @return den Nachrichten-Inhalt dieser RDS-Meldung, niemals <code>null</code>
	 *
	 * @see #getAusdehnungText()
	 * @see #getEreignisText()
	 */
	public String getInhaltText() {
		final String ausdehnungText = getAusdehnungText();
		final String ereignisText = getEreignisText();
		if (!"".equals(ausdehnungText) && !"".equals(ereignisText)) {
			return new StringBuffer(ausdehnungText).append(LEERZEICHEN_SEPARATOR).append(ereignisText).toString();
		}
		if ("".equals(ausdehnungText)) {
			return ereignisText;
		}
		if ("".equals(ereignisText)) {
			return ausdehnungText;
		}
		return "";
	}

	/**
	 * Liefert den freien Nachrichtentext dieser RDS-Meldung zurück.
	 *
	 * @return den freien Nachrichtentext dieser RDS-Meldung
	 */
	public String getFreierText() {
		if (datum != null) {
			return datum.getVersion().getVerkehrsInformationen().getFreierText().getText();
		}
		return "";
	}

	/**
	 * Liefert den Meldungstext dieser RDS-Meldung zurück.
	 *
	 * <p>
	 * Der Meldungstext kann nicht editiert werden. Er wird aus Informationen zur
	 * Lokation und zum Ereignis generiert.
	 * </p>
	 *
	 * <p>
	 * Der Text setzt sich nach folgendem Schema zusammen: "&lt;Straße&gt;",
	 * "&lt;Ort1&gt;" Richtung "&lt;Ort2&gt;", zwischen "&lt;Primärlokation&gt;" und
	 * "&lt;Sekundärlokation&gt;", "&lt;Ereignis&gt;".
	 * </p>
	 *
	 * @return den Meldungstext dieser RDS-Meldung zurück, niemals <code>null</code>
	 */
	public String getMeldungsText() {
		if ((meldungsTextGeladen != null) && !"".equals(meldungsTextGeladen)) {
			return meldungsTextGeladen;
		}
		return meldungsText;
	}

	/**
	 * Generiert abhängig von der gesetzten Straße, TMC-Lokationen und Ereignisse
	 * den Meldungstext.
	 */
	void setMeldungsText() {
		final String alterWert = meldungsText;
		final StringBuilder buf = new StringBuilder("");

		// Straßenname + Segment
		final String strassenNummerText = getStrassenNummerText();
		if (!"".equals(strassenNummerText)) {
			buf.append(strassenNummerText);
		}
		final String segmentText = getSegmentText();
		if (!"".equals(segmentText)) {
			if (buf.length() > 0) {
				buf.append(LEERZEICHEN_SEPARATOR);
			}
			buf.append(segmentText);
		}

		// Nachricht
		final String nachrichtenText = getInhaltText();
		if (!"".equals(nachrichtenText)) {
			if (buf.length() > 0) {
				if (primaereLokation != sekundaereLokation) {
					buf.append(LEERZEICHEN_SEPARATOR);
				} else {
					buf.append(KOMMA_SEPARATOR);
				}
			}
			buf.append(nachrichtenText);
		}

		// Freier Text
		final String freierText = getFreierText();
		if (!"".equals(freierText)) {
			if (buf.length() > 0) {
				buf.append(PUNKT_SEPARATOR);
			}
			buf.append(freierText);
		}

		// Abschluss
		if (buf.length() > 0) {
			buf.append(".");
		}

		meldungsText = buf.toString();
		if ("".equals(meldungsText)) {
			meldungsText = "unbekannt";
		}
		firePropertyChange("meldungsText", alterWert, meldungsText);
	}

	/**
	 * Liefert die Anzahl der für diese RDS-Meldung benötigten Anzahl von Bits im
	 * Free-Form-Content der RDS-Gruppen zurück.
	 *
	 * <p>
	 * Beachte: Es können nur RDS-Meldung erstellt werden, deren Anzahl der Bits die
	 * maximale Anzahl nicht überschreitet.
	 * </p>
	 *
	 * @return die Anzahl der für diese RDS-Meldung benötigten Anzahl von Bits im
	 *         Free-Form-Content der RDS-Gruppen
	 *
	 * @see #BITS_MAX
	 */
	public int getBits() {
		return bits;
	}

	/**
	 * Ermittelt die Anzahl der für diese RDS-Meldung benötigten Anzahl von Bits im
	 * Free-Form-Content der RDS-Gruppen.
	 *
	 * <p>
	 * Beachte: Es können nur RDS-Meldung erstellt werden, deren Anzahl der Bits die
	 * maximale Anzahl nicht überschreitet.
	 * </p>
	 *
	 * @see #BITS_MAX
	 */
	/* package */void setBits() {
		final int alterWert = bits;
		int neuerWert = BITS_DEFAULT;

		// (0) Analysiere Ereignis-Daten
		int anzahlGeaenderterDauern = 0;
		int anzahlBetroffenerLaengen = 0;
		int anzahlKleinerQuantitaeten = 0;
		int anzahlGrosserQuantitaeten = 0;
		int anzahlEmpfehlungen = 0;

		for (final EreignisWrapper ereignis : getEreignisse()) {
			if (ereignis.getDauer() != ereignis.getEmpfohleneDauer()) {
				anzahlGeaenderterDauern++;
			}
			if (ereignis.getBetroffeneLaenge() != null) {
				anzahlBetroffenerLaengen++;
			}
			final RdsQuantitaetWrapper quantitaet = ereignis.getQuantitaet();
			if (quantitaet != null) {
				if (QuantitaetenUtil.isKleineQuantitaet(quantitaet.getKennung())) {
					anzahlKleinerQuantitaeten++;
				} else {
					anzahlGrosserQuantitaeten++;
				}
			}
			if (ereignis.getEmpfehlungsCode() != AttRdsEmpfehlungsCode.ZUSTAND_0_NICHT_DEFINIERT) {
				anzahlEmpfehlungen++;
			}
		}

		// (1) Steuercodes (Kennung/Label 1)

		// (1.1) Priorität/Urgency (Steuercode 0 oder 1)
		if (Math.abs(getEmpfohlenePrioritaet().getValue() - getPrioritaet().getValue()) > 0) {
			neuerWert += BITS_STEUERCODE;
		}

		// (1.2) 'In beide Richtungen'-Flag/Directionality (Steuercode 2)
		if (isEreignisInBeideRichtungen() != isEmpfohleneEreignisInBeideRichtungen()) {
			neuerWert += BITS_STEUERCODE;
		}

		// (1.3) Dauer/Duration (Steuercode 3)
		if (anzahlGeaenderterDauern > 0) {
			neuerWert += BITS_STEUERCODE;
		}

		// (1.4) Ausdehnung/Extent (Steuercode 6 und/oder 7)
		if (getAusdehnung() > 7) {
			neuerWert += BITS_STEUERCODE;
		}
		if (getAusdehnung() > 23) {
			neuerWert += BITS_STEUERCODE;
		}

		// (2) Betroffene(r) Länge/Abschnitt (Kennung 2)
		neuerWert += (anzahlBetroffenerLaengen * BITS_BETROFFENE_LAENGE);
		if (anzahlBetroffenerLaengen > 1) {
			neuerWert += (anzahlBetroffenerLaengen - 1) * BITS_SEPARATOR;
		}

		// (3) Quantitäten/Quantifier (Kennung 4 bzw. 5)
		neuerWert += (anzahlKleinerQuantitaeten * BITS_QUANTITAET_KLEIN);
		neuerWert += (anzahlGrosserQuantitaeten * BITS_QUANTITAET_GROSS);

		// (4) Empfehlungen/Supplementary Information (Kennung 6)
		neuerWert += (anzahlEmpfehlungen * BITS_EMPFEHLUNG);

		// (5) Zusätzliche Ereignisse/Additional Events (Kennung 9)
		final int anzahlZusaetlicherEreignisse = ereignisse.size() - anzahlEmpfehlungen - 1;
		if (anzahlZusaetlicherEreignisse > 0) {
			neuerWert += (anzahlZusaetlicherEreignisse * BITS_ZUSAETLICHES_EREIGNIS);
		}

		if (bits != neuerWert) {
			bits = neuerWert;
			firePropertyChange("bits", alterWert, bits);
		}
	}

	/**
	 * Liefert die Straße zurück, welche dieser RDS-Meldung zugeordnet ist.
	 *
	 * @return die Straße, welche dieser RDS-Meldung zugeordnet ist, kann
	 *         <code>null</code> sein
	 */
	public TmcLinieWrapper getStrasse() {
		return strasse;
	}

	/**
	 * Setzt die Straße dieser RDS-Meldung.
	 *
	 * @param neueStrasse die neue Straße
	 */
	public void setStrasse(final TmcLinieWrapper neueStrasse) {
		if (((neueStrasse == null) && (strasse != null)) || ((neueStrasse != null) && !neueStrasse.equals(strasse))) {
			final TmcLinieWrapper alterWert = strasse;
			strasse = neueStrasse;
			firePropertyChange("strasse", alterWert, neueStrasse);
			setMeldungsText();
		}
	}

	/**
	 * Liefert den aktuellen Status dieser RDS-Meldung zurück.
	 *
	 * @return den aktuellen Status dieser Meldung, niemals <code>null</code>
	 */
	public AttRdsStatus getStatus() {
		return status;
	}

	/**
	 * Setzt den Status dieser RDS-Meldung.
	 *
	 * @param neuerWert der neue Status, darf nicht <code>null</code> sein
	 */
	public void setStatus(final AttRdsStatus neuerWert) {
		assert null != neuerWert;
		if (neuerWert != status) {
			final AttRdsStatus alterWert = status;
			status = neuerWert;
			firePropertyChange("status", alterWert, neuerWert);
		}
	}

	/**
	 * Ermittelt den Status dieser RDS-Meldung abhängig von einer Änderung die an
	 * dieser RDS-Meldung gemacht wurde.
	 *
	 * @param propertyName der Name der Property, welche sich geändert hat
	 */
	void setStatus(final String propertyName) {
		if (isNeuesElement()) {
			return;
		}
		if (Arrays.asList(PROPERTIES_ERWEITERUNG).contains(propertyName)) {
			inhaltGeaendert = true;
			setStatus(AttRdsStatus.ZUSTAND_6_ERWEITERUNG);
		} else if (Arrays.asList(PROPERTIES_MODIFIKATION).contains(propertyName) && !inhaltGeaendert) {
			setStatus(AttRdsStatus.ZUSTAND_2_MODIFIKATION);
		}
	}

	/**
	 * Liefert die Wichtung dieser RDS-Meldung zurück.
	 *
	 * @return Wichtung dieser RDS-Meldung
	 */
	public byte getWichtung() {
		return wichtung;
	}

	/**
	 * Setzt die Wichtung dieser RDS-Meldung.
	 *
	 * @param neuerWert die neue Wichtung, muss ein gültiger Wert des Wertebereichs
	 *                  des Typen <code>AttRdsWichtung</code> sein.
	 *
	 * @see AttRdsWichtung
	 */
	public void setWichtung(final byte neuerWert) {
		final WerteBereich werteBereich = WerteBereich.getWerteBereich(AttRdsWichtung.class);
		assert neuerWert >= werteBereich.getMinimum();
		assert neuerWert <= werteBereich.getMaximum();
		if (neuerWert != wichtung) {
			final byte alterWert = wichtung;
			wichtung = neuerWert;
			firePropertyChange("wichtung", alterWert, neuerWert);
		}
	}

	/**
	 * Liefert die aktuelle Priorität dieser RDS-Meldung zurück.
	 *
	 * @return die aktuelle Priorität dieser RDS-Meldung, niemals <code>null</code>
	 *
	 * @see #setPrioritaet(AttRdsPrioritaet)
	 * @see #getEmpfohlenePrioritaet()
	 */
	public AttRdsPrioritaet getPrioritaet() {
		return prioritaet;
	}

	/**
	 * Setzt die aktuelle Priorität dieser RDS-Meldung.
	 *
	 * <p>
	 * Beachte: Die aktuelle Priorität darf höchsten eine Stufe höher oder niedriger
	 * als die empfohlene Priorität sein.
	 * </p>
	 *
	 * @param neuerWert die neue aktuelle Priorität, darf nicht <code>null</code>
	 *                  sein, darf sich nur um eine Stufe von der empfohlenen
	 *                  Priorität unterscheiden
	 *
	 * @see #getPrioritaet()
	 * @see #getEmpfohlenePrioritaet()
	 */
	public void setPrioritaet(final AttRdsPrioritaet neuerWert) {
		assert null != neuerWert;
		assert Math.abs(neuerWert.getValue() - getEmpfohlenePrioritaet().getValue()) <= 1;
		if (neuerWert != prioritaet) {
			final AttRdsPrioritaet alterWert = prioritaet;
			prioritaet = neuerWert;
			firePropertyChange("prioritaet", alterWert, neuerWert);
			setBits();
		}
	}

	/**
	 * Liefert die empfohlene Priorität dieser RDS-Meldung zurück.
	 *
	 * @return die empfohlene Priorität dieser RDS-Meldung, niemals
	 *         <code>null</code>
	 *
	 * @see #getPrioritaet()
	 * @see #setPrioritaet(AttRdsPrioritaet)
	 */
	public AttRdsPrioritaet getEmpfohlenePrioritaet() {
		return empfohlenePrioritaet;
	}

	/**
	 * Setzt die empfohlene Priorität dieser RDS-Meldung.
	 *
	 * @param neuerWert die neue empfohlene Priorität, darf nicht <code>null</code>
	 *                  sein
	 *
	 * @see #getPrioritaet()
	 * @see #setPrioritaet(AttRdsPrioritaet)
	 * @see #getEmpfohlenePrioritaet()
	 */
	private void setEmpfohlenePrioritaet(final AttRdsPrioritaet neuerWert) {
		assert null != neuerWert;
		if (neuerWert != empfohlenePrioritaet) {
			final AttRdsPrioritaet alterWert = empfohlenePrioritaet;
			empfohlenePrioritaet = neuerWert;
			firePropertyChange("empfohlenePrioritaet", alterWert, neuerWert);
			setBits();
		}
	}

	/**
	 * Setzt den aktuellen und empfohlenen Wert für die Priorität und das 'In beide
	 * Richtungen'- sowie den Wert des Vorhersage-Flags abhängig von den gesetzten
	 * Ereignissen.
	 *
	 * @param aktuelleDaten   <code>true</code>, wenn die aktuellen Daten
	 *                        aktualisiert werden sollen, ansonsten
	 *                        <code>false</code>
	 * @param empfohleneDaten <code>true</code>, wenn die empfohlenen Daten
	 *                        aktualisiert werden sollen, ansonsten
	 *                        <code>false</code>
	 */
	/* package */void aktualisiereEreignisAbhaengigeDaten(final boolean aktuelleDaten, final boolean empfohleneDaten) {

		if (!aktuelleDaten && !empfohleneDaten && ((ereignisse == null) || ereignisse.isEmpty())) {
			return;
		}

		AttRdsPrioritaet hoechstePrioritaet = AttRdsPrioritaet.ZUSTAND_0_NORMAL;
		boolean inBeideRichtungen = false;
		int i = 0;
		for (final EreignisWrapper ereignis : ereignisse) {
			final RdsTmcCodeWrapper rdsTmcCodeWrapper = rdsCache.getRdsTmcCodeWrapper(ereignis.getCode());
			if (rdsTmcCodeWrapper == null) {
				continue;
			}
			if (rdsTmcCodeWrapper.getPrioritaet().getValue() > hoechstePrioritaet.getValue()) {
				hoechstePrioritaet = rdsTmcCodeWrapper.getPrioritaet();
			}
			if (rdsTmcCodeWrapper.getEreignisInBeidenRichtungen() == AttJaNein.ZUSTAND_1_JA) {
				inBeideRichtungen = true;
			}
			if ((i == 0) && aktuelleDaten) {
				setVorhersage(rdsTmcCodeWrapper.getEreignisart() == AttRdsEreignisArt.ZUSTAND_2_VORHERSAGE);
			}
			i++;
		}

		if (empfohleneDaten) {
			setEmpfohlenePrioritaet(hoechstePrioritaet);
			setEmpfohleneEreignisInBeideRichtungen(inBeideRichtungen);
		}
		if (aktuelleDaten) {
			setPrioritaet(hoechstePrioritaet);
			setEreignisInBeideRichtungen(inBeideRichtungen);
		}
	}

	/**
	 * Liefert den Zeitpunkt des Beginns der Gültigkeit zurück.
	 *
	 * @return Zeitpunkt des Beginns der Gültigkeit, niemals <code>null</code>
	 */
	public Date getAktivierungsZeit() {
		return aktivierungsZeit;
	}

	/**
	 * Setzt den Zeitpunkt des Beginns der Gültigkeit.
	 *
	 * @param neuerWert der neue Zeitpunkt
	 */
	public void setAktivierungsZeit(final Date neuerWert) {
		Date derNeueWert = neuerWert;
		if (derNeueWert == null) {
			derNeueWert = new Date();
		}
		if (!derNeueWert.equals(aktivierungsZeit)) {
			final Date alterWert = aktivierungsZeit;
			aktivierungsZeit = new Date(derNeueWert.getTime());
			firePropertyChange("aktivierungsZeit", alterWert, derNeueWert);
			if (ablaufZeit.before(aktivierungsZeit)) {
				setAblaufZeit(aktivierungsZeit);
			}
		}
	}

	/**
	 * Setzt den Startzeitpunkt dieses Objektes mit Zeitbereich.
	 *
	 * @param startZeit der neue Startzeitpunkt
	 */
	@Override
	public void setStartZeit(final Date startZeit) {
		setAktivierungsZeit(startZeit);
	}

	/**
	 * Liefert den Zeitpunkt des Endes der Gültigkeit.
	 *
	 * @return Zeitpunkt des Endes der Gültigkeit, niemals <code>null</code>
	 */
	public Date getAblaufZeit() {
		return ablaufZeit;
	}

	/**
	 * Setzt den Zeitpunkt des Endes der Gültigkeit.
	 *
	 * @param neuerWert der neuen Zeitpunkt
	 */
	public void setAblaufZeit(final Date neuerWert) {
		Date derNeueWert = neuerWert;
		if (derNeueWert == null) {
			derNeueWert = new Date();
		}
		if (!derNeueWert.equals(ablaufZeit)) {
			final Date alterWert = ablaufZeit;
			ablaufZeit = new Date(derNeueWert.getTime());
			firePropertyChange("ablaufZeit", alterWert, derNeueWert);
			if (aktivierungsZeit.after(ablaufZeit)) {
				setAktivierungsZeit(ablaufZeit);
			}
		}
	}

	/**
	 * Setzt den Endzeitpunkt dieses Objektes mit Zeitbereich.
	 *
	 * @param endZeit der neue Endzeitpunkt
	 */
	@Override
	public void setEndZeit(final Date endZeit) {
		setAblaufZeit(endZeit);
	}

	/**
	 * Liefert <code>true</code> zurück, wenn die Ereignis-Information
	 * bi-direktional ist, ansonsten <code>false</code>.
	 *
	 * @return <code>true</code>, wenn die Ereignis-Information bi-direktional ist,
	 *         ansonsten <code>false</code>
	 */
	public boolean isEreignisInBeideRichtungen() {
		return ereignisInBeideRichtungen;
	}

	/**
	 * Definiert, ob die Ereignis-Information bi-direktional ist oder nicht.
	 *
	 * @param neuerWert <code>true</code>, wenn die Ereignis-Information
	 *                  bi-direktional ist, ansonsten <code>false</code>
	 */
	public void setEreignisInBeideRichtungen(final boolean neuerWert) {
		if (neuerWert != ereignisInBeideRichtungen) {
			final boolean alterWert = ereignisInBeideRichtungen;
			ereignisInBeideRichtungen = neuerWert;
			firePropertyChange("ereignisInBeideRichtungen", alterWert, neuerWert);
			setMeldungsText();
			setBits();
		}
	}

	/**
	 * Liefert die empfohlene Direktionalität ('Ereignis in beide Richtungen'-Flag)
	 * zurück. Die empfohlene Direktionalität entspricht der höchsten
	 * Direktionalität (der größten räumlichen Gültigkeit) der gesetzten
	 * Ereignisdaten.
	 *
	 * <p>
	 * Ist also ein Ereignisdatum, welches für beide Richtungen gelten soll gesetzt,
	 * so ist auch dies auch die empfohlene Direktionalität der RDS-Meldung.
	 * </p>
	 *
	 * @return die empfohlene Direktionalität ('Ereignis in beide Richtungen'-Flag)
	 */
	public boolean isEmpfohleneEreignisInBeideRichtungen() {
		return empfohleneEreignisInBeideRichtungen;
	}

	/**
	 * Setzt die empfohlene Direktionalität ('Ereignis in beide Richtungen'-Flag)
	 * dieser RDS-Meldung.
	 *
	 * @param neuerWert der neue Wert
	 */
	private void setEmpfohleneEreignisInBeideRichtungen(final boolean neuerWert) {
		if (neuerWert != empfohleneEreignisInBeideRichtungen) {
			final boolean alterWert = empfohleneEreignisInBeideRichtungen;
			empfohleneEreignisInBeideRichtungen = neuerWert;
			firePropertyChange("empfohleneEreignisInBeideRichtungen", alterWert, neuerWert);
			setBits();
		}
	}

	/**
	 * Liefert den ersten Ereignis-Typ dieser RDS-Meldung zurück.
	 *
	 * @return den ersten Ereignis-Typ dieser RDS-Meldung, niemals <code>null</code>
	 *
	 * @deprecated nutze {@link #getEreignisTypen()}
	 */
	@Deprecated
	public AttRdsEreignisTyp getEreignisTyp() {
		return ereignisTyp;
	}

	/**
	 * Liefert die Typen der Ereignis-Information dieser RDS-Meldung zurück.
	 *
	 * @return die Typen der Ereignis-Information dieser RDS-Meldung, niemals
	 *         <code>null</code>
	 */
	public List<AttRdsEreignisTyp> getEreignisTypen() {
		return ereignisTypen;
	}

	/**
	 * Setzt den alleinige Typ der Ereignis-Information dieser RDS-Meldung.
	 *
	 * @param neuerWert der neue Typ, darf nicht <code>null</code> sein
	 *
	 * @deprecated nutze {@link #setEreignisTypen(List)}
	 */
	@Deprecated
	public void setEreignisTyp(final AttRdsEreignisTyp neuerWert) {
		assert null != neuerWert;

		final AttRdsEreignisTyp alterWert = ereignisTyp;
		ereignisTyp = neuerWert;
		firePropertyChange("ereignisTyp", alterWert, neuerWert);

		final List<AttRdsEreignisTyp> alterWert2 = new ArrayList<>(ereignisTypen);
		ereignisTypen = new ArrayList<>();
		ereignisTypen.add(ereignisTyp);
		firePropertyChange("ereignisTypen", alterWert2, ereignisTypen);
	}

	/**
	 * Setzt die Typen der Ereignis-Information dieser RDS-Meldung.
	 *
	 * @param neuerWert der neuen Typen, darf nicht <code>null</code> sein, darf
	 *                  nicht leer sein
	 */
	public void setEreignisTypen(final List<AttRdsEreignisTyp> neuerWert) {
		assert (null != neuerWert) && !neuerWert.isEmpty();

		final List<AttRdsEreignisTyp> alterWert = new ArrayList<>(ereignisTypen);
		ereignisTypen = neuerWert;
		firePropertyChange("ereignisTypen", alterWert, neuerWert);

		final AttRdsEreignisTyp alterWert2 = ereignisTyp;
		ereignisTyp = ereignisTypen.get(0);
		firePropertyChange("ereignisTyp", alterWert2, ereignisTyp);
	}

	/**
	 * Liefert die primäre Lokation dieser RDS-Meldung zurück.
	 *
	 * @return primäre Lokation dieser RDS-Meldung
	 */
	public TmcPunktWrapper getPrimaereLokation() {
		return primaereLokation;
	}

	/**
	 * Setzt die primäre Lokation dieser RDS-Meldung.
	 *
	 * @param neuerWert die neue primäre Lokation, darf nicht <code>null</code> sein
	 */
	public void setPrimaereLokation(final TmcPunktWrapper neuerWert) {
		setPrimaereLokation(neuerWert, true);
	}

	/**
	 * Setzt die primäre Lokation dieser RDS-Meldung.
	 *
	 * @param neuerWert                      die neue primäre Lokation, darf nicht
	 *                                       <code>null</code> sein
	 * @param aktualisiereSekundaereLokation <code>true</code>, wenn sekundäre
	 *                                       Lokation aktualisiert werden soll,
	 *                                       ansonsten <code>false</code>
	 */
	private void setPrimaereLokation(final TmcPunktWrapper neuerWert, final boolean aktualisiereSekundaereLokation) {

		assert null != neuerWert;

		if (!neuerWert.equals(primaereLokation)) {
			tmcLokationen = null;
			final TmcPunktWrapper alterWert = primaereLokation;
			primaereLokation = neuerWert;
			firePropertyChange("primaereLokation", alterWert, neuerWert);

			// Aktualisiere ggf. sekundäre Lokation
			if (aktualisiereSekundaereLokation) {
				final List<TmcPunktWrapper> nachfolger = new ArrayList<>();
				nachfolger.add(primaereLokation);
				nachfolger.addAll(tmcLocationCodeCache.getNachfolgendeTmcPunktWrapper(primaereLokation,
						strasse.getRichtung(), Konstanten.MAXIMALE_ANZAHL_NACHFOLGER_LOKATIONEN, true));
				if (!nachfolger.contains(sekundaereLokation) && !nachfolger.isEmpty()) {
					setSekundaereLokation(nachfolger.get(0), false);
				}
				setMeldungsText();
				setAusdehnung();
			}
		}
	}

	/**
	 * Liefert die sekundäre Lokation dieser RDS-Meldung zurück.
	 *
	 * @return sekundäre Lokation dieser RDS-Meldung
	 */
	public TmcPunktWrapper getSekundaereLokation() {
		return sekundaereLokation;
	}

	/**
	 * Setzt die sekundäre Lokation dieser RDS-Meldung.
	 *
	 * @param neuerWert die neue sekundäre Lokation, darf nicht <code>null</code>
	 *                  sein
	 */
	public void setSekundaereLokation(final TmcPunktWrapper neuerWert) {
		setSekundaereLokation(neuerWert, true);
	}

	/**
	 * Setzt die sekundäre Lokation dieser RDS-Meldung.
	 *
	 * @param neuerWert                    die neue sekundäre Lokation, darf nicht
	 *                                     <code>null</code> sein
	 * @param aktualisierePrimaereLokation <code>true</code>, wenn primäre Lokation
	 *                                     aktualisiert werden soll, ansonsten
	 *                                     <code>false</code>
	 */
	private void setSekundaereLokation(final TmcPunktWrapper neuerWert, final boolean aktualisierePrimaereLokation) {
		assert null != neuerWert;
		if (!neuerWert.equals(sekundaereLokation)) {
			tmcLokationen = null;
			final TmcPunktWrapper alterWert = sekundaereLokation;
			sekundaereLokation = neuerWert;
			firePropertyChange("sekundaereLokation", alterWert, neuerWert);

			// Aktualisiere ggf. primäre Lokation
			if (aktualisierePrimaereLokation) {
				final List<TmcPunktWrapper> vorgaenger = new ArrayList<>();
				vorgaenger.add(sekundaereLokation);
				vorgaenger.addAll(tmcLocationCodeCache.getVorgaengerTmcPunktWrapper(sekundaereLokation,
						strasse.getRichtung(), Konstanten.MAXIMALE_ANZAHL_NACHFOLGER_LOKATIONEN, true));
				if (!vorgaenger.contains(primaereLokation) && !vorgaenger.isEmpty()) {
					setPrimaereLokation(vorgaenger.get(0), false);
				}
				setMeldungsText();
				setAusdehnung();
			}
		}
	}

	/**
	 * Liefert die Liste der TMC-Lokationen dieser RDS-Meldung zurück (startend mit
	 * der primären Lokation).
	 *
	 * @return die Liste der TMC-Lokationen dieser RDS-Meldung
	 */
	public List<TmcPunktWrapper> getLokationen() {
		final List<TmcPunktWrapper> lokationen = new ArrayList<>();

		if (primaereLokation != null) {
			lokationen.add(primaereLokation);

			if (!primaereLokation.equals(sekundaereLokation)) {
				final List<TmcPunktWrapper> nachfolger = tmcLocationCodeCache
						.getNachfolgendeTmcPunktWrapper(primaereLokation, strasse.getRichtung(), true);
				for (final TmcPunktWrapper tmcPunktWrapper : nachfolger) {
					lokationen.add(tmcPunktWrapper);
					if (tmcPunktWrapper.equals(sekundaereLokation)) {
						return lokationen;
					}
				}
			}
		}
		return lokationen;
	}

	/**
	 * Liefert die Ausdehnung dieser RDS-Meldung zurück. Die Ausdehnung entspricht
	 * der Anzahl der TMC-Punkte, für welche zusätzlich zur primären Lokation diese
	 * RDS-Meldung gilt.
	 *
	 * @return die Ausdehnung dieser RDS-Meldung
	 */
	public int getAusdehnung() {
		return ausdehnung;
	}

	/**
	 * Ermittelt die Ausdehnung dieser RDS-Meldung.
	 */
	private void setAusdehnung() {
		final int alterWert = ausdehnung;
		int neuerWert = getLokationen().size() - 1;
		if (neuerWert < 0) {
			neuerWert = 0;
		}
		if (neuerWert != alterWert) {
			ausdehnung = neuerWert;
			firePropertyChange("ausdehnung", alterWert, neuerWert);
			setBits();
		}
	}

	/**
	 * Liefert die Liste der Ereignisse dieser RDS-Meldung zurück.
	 *
	 * @return Liste der Ereignisse dieser RDS-Meldung, niemals <code>null</code>
	 */
	public List<EreignisWrapper> getEreignisse() {
		return ereignisse;
	}

	/**
	 * Setzt die Ereignisse dieser RDS-Meldung.
	 *
	 * @param neuerWert die neue Ereignis-Liste, darf nicht <code>null</code> sein
	 */
	public void setEreignisse(final List<EreignisWrapper> neuerWert) {
		assert null != neuerWert;

		final List<EreignisWrapper> alterWert = new ArrayList<>(ereignisse);
		ereignisse = neuerWert;
		firePropertyChange("ereignisse", alterWert, neuerWert);

		aktualisiereEreignisAbhaengigeDaten(true, true);
		setMeldungsText();
		setBits();
	}

	/**
	 * Liefert den Zustand dieser RDS-Meldung zurück.
	 *
	 * @return den Zustand dieser RDS-Meldung
	 */
	public AttRdsZustand getZustand() {
		return zustand;
	}

	/**
	 * Setzt den Zustand dieser RDS-Meldung.
	 *
	 * @param neuerWert der neue Zustand, darf nicht <code>null</code> sein
	 */
	public void setZustand(final AttRdsZustand neuerWert) {
		assert null != neuerWert;
		if (neuerWert != zustand) {
			final AttRdsZustand alterWert = zustand;
			zustand = neuerWert;
			firePropertyChange("zustand", alterWert, neuerWert);
		}
	}

	/**
	 * Liefert <code>true</code> zurück, wenn es sich bei dieser RDS-Meldung um eine
	 * Vorhersage handelt (also die Ereignis-Art des ersten Ereignis-Datums auf den
	 * Wert "Vorhersage" gesetzt ist).
	 *
	 * @return <code>true</code>, wenn es sich bei dieser RDS-Meldung um eine
	 *         Vorhersage handelt, ansonsten <code>false</code>
	 */
	public boolean isVorhersage() {
		return vorhersage;
	}

	/**
	 * Setzt das Vorhersage-Flag.
	 *
	 * @param neuerWert das neue Vorhersage-Flag
	 */
	public void setVorhersage(final boolean neuerWert) {
		if (neuerWert != vorhersage) {
			final boolean alterWert = vorhersage;
			vorhersage = neuerWert;
			firePropertyChange("vorhersage", alterWert, neuerWert);
		}
	}

	@Override
	protected void firePropertyChange(final String propertyName, final Object oldValue, final Object newValue) {
		super.firePropertyChange(propertyName, oldValue, newValue);
		if (!PROP_DIRTY.equals(propertyName) && (oldValue != null) && !oldValue.equals(newValue)) {
			setDirty(true);
		}
		setStatus(propertyName);
	}

	@Override
	public int hashCode() {
		return Objects.hash(aspekt, rdsMeldung);
	}

	@Override
	public boolean equals(final Object obj) {
		if (this == obj) {
			return true;
		}
		if (obj == null) {
			return false;
		}
		if (getClass() != obj.getClass()) {
			return false;
		}
		final RdsMeldungWrapper other = (RdsMeldungWrapper) obj;
		if (!Objects.equals(aspekt, other.aspekt)) {
			return false;
		}
		if (!Objects.equals(rdsMeldung, other.rdsMeldung)) {
			return false;
		}
		return true;
	}

	/**
	 * Liefert den RDS-Cache zurück.
	 *
	 * @return den RDS-Cache
	 */
	/* package */RdsCache getRdsCache() {
		return rdsCache;
	}

	/**
	 * Liefert <code>true</code> zurück, wenn diese RDS-Meldung gültig ist, also
	 * sämtliche Daten geladen werden konnten.
	 *
	 * @return <code>true</code> zurück, wenn diese RDS-Meldung gültig ist,
	 *         ansonsten <code>false</code>
	 */
	public boolean isGueltig() {
		return gueltig;
	}

	/**
	 * Liefert <code>true</code> zurück, wenn diese RDS-Meldunge editierbar ist.
	 *
	 * <p>
	 * Es sind nur generierte Meldungen (Aspekt: RDSGeneriert) editierbar, deren
	 * Daten komplett geladen werden konnten. Zudem darf die Meldung nicht bereits
	 * verworfen, aufgehoben oder gelöscht sein.
	 * </p>
	 *
	 * @return <code>true</code>, wenn diese RDS-Meldung editierbar ist, ansonsten
	 *         <code>false</code>
	 *
	 * @see #isGueltig()
	 */
	public boolean isEditierbar() {
		return isNeuesElement() || (isGueltig() && (getAspekt() == PdRdsMeldung.Aspekte.RdsGeneriert)
				&& (getZustand() != AttRdsZustand.ZUSTAND_3_VERWORFEN)
				&& (getStatus() != AttRdsStatus.ZUSTAND_4_AUFHEBUNG)
				&& (getStatus() != AttRdsStatus.ZUSTAND_5_LOESCHUNG));
	}

	/**
	 * Liefert <code>true</code> zurück, wenn diese RDS-Meldung quittiert werden
	 * darf.
	 *
	 * <p>
	 * Es dürfen nur nicht-quittierte (Zustand: "nicht quittiert"), generierte
	 * (Aspekt: RDSGeneriert) Meldungen quittiert werden.
	 * </p>
	 *
	 * @return <code>true</code>, wenn diese RDS-Meldung quittiert werden darf,
	 *         ansonsten <code>false</code>
	 */
	public boolean isQuittierbar() {
		return (getAspekt() == PdRdsMeldung.Aspekte.RdsGeneriert)
				&& (getZustand() == AttRdsZustand.ZUSTAND_0_NICHT_QUITTIERT);
	}

	/**
	 * Liefert <code>true</code> zurück, wenn diese RDS-Meldung gelöscht (verworfen
	 * bzw. aufgehoben) werden darf.
	 *
	 * <p>
	 * Empfangene Meldungen (Aspekt: RDSEmpfangen) dürfen immer gelöscht werden.
	 * Sämtliche andere Meldungen (Aspekt: RDSGeneriert, RDSSenden, RDSVersendet)
	 * dürfen nur gelöscht werden, wenn diese nicht bereits verworfen (Zustand:
	 * verworfen), aufgehoben (Status: Aufhebung) oder gelöscht (Status: Löschung)
	 * wurden.
	 * </p>
	 *
	 * @return <code>true</code>, wenn diese RDS-Meldung gelöscht (verworfen bzw.
	 *         aufgehoben) werden darf, ansonsten <code>false</code>
	 */
	public boolean isLoeschbar() {
		return (getAspekt() == PdRdsMeldung.Aspekte.RdsEmpfangen)
				|| ((getZustand() != AttRdsZustand.ZUSTAND_3_VERWORFEN)
						&& (getStatus() != AttRdsStatus.ZUSTAND_4_AUFHEBUNG)
						&& (getStatus() != AttRdsStatus.ZUSTAND_5_LOESCHUNG));
	}

	@Override
	public String toString() {
		if (getMeldungsText() != null) {
			final String s = getMeldungsText().trim();
			if (!"".equals(s)) {
				return s;
			}
		}
		return super.toString();
	}
}
