/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2011 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.bitctrl.modell.util.rdstmc;

import de.bsvrz.sys.funclib.bitctrl.modell.att.Zahl;
import de.bsvrz.sys.funclib.bitctrl.modell.metamodellglobal.attribute.AttJaNein;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.attribute.AttRdsEreignisDauer;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.attribute.AttRdsEreignisKategorie;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.attribute.AttRdsPrioritaet;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.objekte.RdsQuantitaet;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmsttmccodes.attribute.AttRdsEmpfehlungsCode;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmsttmccodes.attribute.AttRdsEreignisArt;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmsttmccodes.attribute.AttRdsEreignisCode;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmsttmccodes.attribute.AttRdsVorhersageCode;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmsttmccodes.konfigurationsdaten.KdRdsTmcCode.Daten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmsttmccodes.objekte.RdsTmcCode;

/**
 * Wrapper, welcher für ein RdsTmcCode-SystemObjekt sämtliche Daten vorhält.
 *
 * @author BitCtrl Systems GmbH, Andreas Meissner
 *
 * @see RdsTmcCode
 */
public class RdsTmcCodeWrapper {

	/** Der in diesem Wrapper gekapselte {@link RdsTmcCode}. */
	private final RdsTmcCode rdsTmcCode;

	/** Das zu dem gekapselten {@link RdsTmcCode} gehörige Datum. */
	private final Daten datum;

	/** Der Code (Ereignis-, Vorhersage- oder Empfehlungs-Code. */
	private Zahl<Integer> code;

	/**
	 * Der Konstruktor.
	 *
	 * @param rdsTmcCode der RDS-TMC-Code, darf nicht <code>null</code> sein
	 */
	public RdsTmcCodeWrapper(final RdsTmcCode rdsTmcCode) {
		assert rdsTmcCode.getKdRdsTmcCode() != null;
		assert rdsTmcCode.getKdRdsTmcCode().getDatum() != null;

		this.rdsTmcCode = rdsTmcCode;
		datum = rdsTmcCode.getKdRdsTmcCode().getDatum();
	}

	/**
	 * Liefert das in diesem Objekt gekapselte {@link RdsTmcCode}-Systemobjekt
	 * zurück.
	 *
	 * @return das in diesem Objekt gekapselte {@link RdsTmcCode}-Systemobjekt,
	 *         niemals <code>null</code>
	 */
	public RdsTmcCode getRdsTmcCode() {
		return rdsTmcCode;
	}

	/**
	 * Liefert den gesetzten Code zurück. Kann ein Ereignis-, Vorhersage- oder
	 * Empfehlungs-Code sein.
	 *
	 * @return den gesetzten Code (Ereignis-, Vorhersage- oder Empfehlungs-Code).
	 *
	 * @see de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmsttmccodes.konfigurationsdaten.KdRdsTmcCode.Daten#getEreignisCode()
	 * @see de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmsttmccodes.konfigurationsdaten.KdRdsTmcCode.Daten#getVorhersageCode()
	 * @see de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmsttmccodes.konfigurationsdaten.KdRdsTmcCode.Daten#getEmpfehlungsCode()
	 */
	public Zahl<Integer> getCode() {
		if (code == null) {
			if ((datum.getEreignisCode() != null)
					&& (datum.getEreignisCode() != AttRdsEreignisCode.ZUSTAND_0_NICHT_DEFINIERT)) {
				code = datum.getEreignisCode();
			} else if ((datum.getVorhersageCode() != null)
					&& (datum.getVorhersageCode() != AttRdsVorhersageCode.ZUSTAND_0_NICHT_DEFINIERT)) {
				code = datum.getVorhersageCode();
			} else if ((datum.getEmpfehlungsCode() != null)
					&& (datum.getEmpfehlungsCode() != AttRdsEmpfehlungsCode.ZUSTAND_0_NICHT_DEFINIERT)) {
				code = datum.getEmpfehlungsCode();
			}
		}
		return code;
	}

	/**
	 * Liefert die mit diesem Ereignis-Code assoziierte Ereignisart zurück.
	 *
	 * @return die mit diesem Ereignis-Code assoziierte Ereignisart
	 *
	 * @see de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmsttmccodes.konfigurationsdaten.KdRdsTmcCode.Daten#getEreignisart()
	 */
	public AttRdsEreignisArt getEreignisart() {
		return datum.getEreignisart();
	}

	/**
	 * Liefert die mit diesem Ereignis-Code assoziierte Dauer zurück.
	 *
	 * @return die mit diesem Ereignis-Code assoziierte Dauer
	 *
	 * @see de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmsttmccodes.konfigurationsdaten.KdRdsTmcCode.Daten#getEreignisdauer()
	 */
	public AttRdsEreignisDauer getEreignisdauer() {
		return datum.getEreignisdauer();
	}

	/**
	 * Liefert das mit diesem Ereignis-Code assoziierte Flag "In Beide Richtungen"
	 * zurück.
	 *
	 * @return das mit diesem Ereignis-Code assoziierte Flag "In Beide Richtungen"
	 *
	 * @see de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmsttmccodes.konfigurationsdaten.KdRdsTmcCode.Daten#getEreignisInBeidenRichtungen()
	 */
	public AttJaNein getEreignisInBeidenRichtungen() {
		return datum.getEreignisInBeidenRichtungen();
	}

	/**
	 * Liefert die mit diesem Ereignis-Code assoziierte Kategorie zurück.
	 *
	 * @return die mit diesem Ereignis-Code assoziierte Kategorie
	 *
	 * @see de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmsttmccodes.konfigurationsdaten.KdRdsTmcCode.Daten#getKategorie()
	 */
	public AttRdsEreignisKategorie getKategorie() {
		return datum.getKategorie();
	}

	/**
	 * Liefert den mit diesem Ereignis-Code assoziierte Phrasencode zurück.
	 *
	 * @return den mit diesem Ereignis-Code assoziierte Phrasencode
	 *
	 * @see de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmsttmccodes.konfigurationsdaten.KdRdsTmcCode.Daten#getPhrasenCode()
	 */
	public String getPhrasenCode() {
		return datum.getPhrasenCode();
	}

	/**
	 * Liefert die mit diesem Ereignis-Code assoziierte Prioritaet zurück.
	 *
	 * @return die mit diesem Ereignis-Code assoziierte Prioritaet
	 *
	 * @see de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmsttmccodes.konfigurationsdaten.KdRdsTmcCode.Daten#getPrioritaet()
	 */
	public AttRdsPrioritaet getPrioritaet() {
		return datum.getPrioritaet();
	}

	/**
	 * Liefert die mit diesem Ereignis-Code assoziierte Quantität zurück.
	 *
	 * @return die mit diesem Ereignis-Code assoziierte Quantität
	 *
	 * @see de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmsttmccodes.konfigurationsdaten.KdRdsTmcCode.Daten#getQuantitaetReferenz()
	 */
	public RdsQuantitaet getQuantitaetReferenz() {
		return datum.getQuantitaetReferenz();
	}

	/**
	 * Liefert den Text dieses Ereignis-Codes zurück.
	 *
	 * @return den Text dieses Ereignis-Codes
	 *
	 * @see de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmsttmccodes.konfigurationsdaten.KdRdsTmcCode.Daten#getText()
	 */
	public String getText() {
		return datum.getText();
	}

	/**
	 * Liefert den Text dieses Ereignis-Codes zurück, welcher bei einer Quantität
	 * mit dem Wert 1 genommen werden soll.
	 *
	 * @return den Text dieses Ereignis-Codes, welcher bei einer Quantität mit dem
	 *         Wert 1 genommen werden soll
	 *
	 * @see de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmsttmccodes.konfigurationsdaten.KdRdsTmcCode.Daten#getTextQuantitaetEinzahl()
	 */
	public String getTextQuantitaetEinzahl() {
		return datum.getTextQuantitaetEinzahl();
	}

	/**
	 * Liefert den Text dieses Ereignis-Codes zurück, welcher bei einer Quantität
	 * mit dem Wert &gt; 1 genommen werden soll.
	 *
	 * @return den Text dieses Ereignis-Codes, welcher bei einer Quantität mit dem
	 *         Wert &gt; 1 genommen werden soll
	 *
	 * @see de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmsttmccodes.konfigurationsdaten.KdRdsTmcCode.Daten#getTextQuantitaetMehrzahl()
	 */
	public String getTextQuantitaetMehrzahl() {
		return datum.getTextQuantitaetMehrzahl();
	}
}
