/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2011 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.bitctrl.modell.util.rdstmc;

import java.util.Scanner;
import java.util.regex.Pattern;

import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.attribute.AttRdsTMCRichtung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcglobal.konfigurationsdaten.KdTmcLinie.Daten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcglobal.objekte.TmcGebiet;
import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcglobal.objekte.TmcLinie;
import de.bsvrz.sys.funclib.bitctrl.modell.util.cache.TmcCache;

/**
 * Wrapper, welcher für ein TmcLinie-SystemObjekt sämtliche Daten vorhält.
 * 
 * @author BitCtrl Systems GmbH, Andreas Meissner
 * 
 * @see TmcLinie
 */
public class TmcLinieWrapper extends TmcLocationCodeWrapper implements Comparable<TmcLinieWrapper> {

	/** Regulärer Ausdruck für eine Nummer. */
	private static final String NUMBER_REGEX = "[0-9]+";

	/** Die in diesem Wrapper gekapselte TmcLinie. */
	private final TmcLinie tmcLinie;

	/** Das zu der gekapseltem TmcLinie gehörige Datum. */
	private final Daten datum;

	/** Die Richtung der TmcLinie. */
	private final AttRdsTMCRichtung richtung;

	/** Der Straßentyp der TmcLinie. */
	private String strassenTyp;

	/** Die Straßennummer der TmcLinie. */
	private Integer strassenNummer;

	/** Der Straßenzusatz der TmcLinie. */
	private String strassenZusatz;

	/**
	 * Liefert die Richtung dieser TMC-Linie zurück.
	 * 
	 * @return die Richtung dieser TMC-Linie
	 */
	public AttRdsTMCRichtung getRichtung() {
		return richtung;
	}

	/**
	 * Der Konstruktor.
	 * 
	 * @param tmcLinie die TMC-Linie, darf nicht <code>null</code> sein
	 */
	public TmcLinieWrapper(final TmcLinie tmcLinie) {
		this(tmcLinie, AttRdsTMCRichtung.ZUSTAND_0_POSITIV);
	}

	/**
	 * Der Konstruktor.
	 * 
	 * @param tmcLinie die TMC-Linie, darf nicht <code>null</code> sein
	 * @param richtung die Richtung
	 */
	public TmcLinieWrapper(final TmcLinie tmcLinie, final AttRdsTMCRichtung richtung) {
		super(tmcLinie);
		assert tmcLinie.getKdTmcLinie() != null;
		assert tmcLinie.getKdTmcLinie().getDatum() != null;

		this.tmcLinie = tmcLinie;
		this.richtung = richtung;
		datum = tmcLinie.getKdTmcLinie().getDatum();

		strassenTyp = null;
		strassenNummer = null;
		strassenZusatz = null;
	}

	/**
	 * Liefert das in diesem Objekt gekapselte TmcLinie-Systemobjekt zurück.
	 * 
	 * @return das in diesem Objekt gekapselte TmcLinie-Systemobjekt, niemals
	 *         <code>null</code>
	 */
	public TmcLinie getTmcLinie() {
		return tmcLinie;
	}

	/**
	 * Attribut zur Referenzierung des Objekttyps: typ.tmcGebiet (Optionale
	 * Referenzierung erlaubt).
	 * 
	 * @return der Wert von IstTeilvonTmcGebiet
	 * 
	 * @see de.bsvrz.sys.funclib.bitctrl.modell.tmtmcglobal.konfigurationsdaten.KdTmcLinie.Daten#getIstTeilvonTmcGebiet()
	 */
	public TmcGebiet getIstTeilvonTmcGebiet() {
		return datum.getIstTeilvonTmcGebiet();
	}

	/**
	 * Attribut zur Referenzierung des Objekttyps: typ.tmcLinie (Optionale
	 * Referenzierung erlaubt).
	 * 
	 * @return der Wert von IstTeilvonTmcLinie
	 * 
	 * @see de.bsvrz.sys.funclib.bitctrl.modell.tmtmcglobal.konfigurationsdaten.KdTmcLinie.Daten#getIstTeilvonTmcLinie()
	 */
	public TmcLinie getIstTeilvonTmcLinie() {
		return datum.getIstTeilvonTmcLinie();
	}

	/**
	 * Attribut zur Referenzierung des Objekttyps: typ.tmcLinie (Optionale
	 * Referenzierung erlaubt).
	 * 
	 * @return der Wert von Nachfolger
	 * 
	 * @see de.bsvrz.sys.funclib.bitctrl.modell.tmtmcglobal.konfigurationsdaten.KdTmcLinie.Daten#getNachfolger()
	 */
	public TmcLinie getNachfolger() {
		return datum.getNachfolger();
	}

	/**
	 * Attribut zur Referenzierung des Objekttyps: typ.tmcLinie (Optionale
	 * Referenzierung erlaubt).
	 * 
	 * @return der Wert von Vorgaenger
	 * 
	 * @see de.bsvrz.sys.funclib.bitctrl.modell.tmtmcglobal.konfigurationsdaten.KdTmcLinie.Daten#getVorgaenger()
	 */
	public TmcLinie getVorgaenger() {
		return datum.getVorgaenger();
	}

	/**
	 * Liefert den Namen der Straße zurück, welche diese TMC-Linie repräsentiert.
	 * 
	 * <p>
	 * Der hier zurückgelieferte Straßenname setzt sich aus den TMC-Informationen zu
	 * Straßennummer sowie dem ersten und zweiten Namen zusammen. Der Straßenname
	 * hat dabei folgende Form: &lt;Straßennummer&gt; von &lt;Erster Name&gt; nach
	 * &lt;Zweiter Name&gt;.
	 * </p>
	 * 
	 * @return den Namen der Straße, welche diese TMC-Linie repräsentiert, kann
	 *         <code>null</code> sein
	 * 
	 * @see TmcLocationCodeWrapper#getTmcStrassenNummer()
	 * @see TmcLocationCodeWrapper#getTmcErsterName()
	 * @see TmcLocationCodeWrapper#getTmcZweiterName()
	 */
	public String getStrassenName() {
		final String tmcStrassenNummer = getTmcStrassenNummer();
		final String tmcStrassenName = getTmcStrassenName();
		final String tmcErsterName = getTmcErsterName();
		final String tmcZweiterName = getTmcZweiterName();
		final StringBuilder sname = new StringBuilder();

		if ((tmcStrassenNummer != null) && !"".equals(tmcStrassenNummer)) {
			sname.append(tmcStrassenNummer);
		} else {
			if ((tmcStrassenName != null) && !"".equals(tmcStrassenName)) {
				sname.append(tmcStrassenName);
			}
		}

		if ((tmcErsterName != null) && !"".equals(tmcErsterName) && (tmcZweiterName != null)
				&& !"".equals(tmcZweiterName)) {
			if (richtung == AttRdsTMCRichtung.ZUSTAND_0_POSITIV) {
				sname.append(" von ").append(tmcErsterName).append(" nach ").append(tmcZweiterName);
			} else {
				sname.append(" von ").append(tmcZweiterName).append(" nach ").append(tmcErsterName);
			}

			return sname.toString();
		}

		return "TMC-Linie " + getAttTmcLocationCode().toString();
	}

	@Override
	public String toString() {
		return getStrassenName();
	}

	/**
	 * Vergleicht diesen TmcLinieWrapper mit einem anderen. Diese Methode ist dafür
	 * gedacht, TmcLinien zu sortieren.
	 * 
	 * @param o der andere TmcLinieWrapper, darf nicht <code>null</code> sein
	 * 
	 * @return 0 bei Gleichheit; negativer Wert, wenn diese TmcLinie vor der
	 *         gegebenen TmcLinie liegen soll, ansonsten positiver Wert
	 */
	@Override
	public int compareTo(final TmcLinieWrapper o) {

		if (o == null) {
			return 0;
		}

		int compare = getRichtung().compareTo(o.getRichtung());
		if (compare == 0) {
			compare = getStrassenTyp().compareTo(o.getStrassenTyp());
		}
		if (compare == 0) {
			compare = new Integer(getStrassenNummer()).compareTo(new Integer(o.getStrassenNummer()));
		}
		if (compare == 0) {
			compare = getStrassenZusatz().compareTo(o.getStrassenZusatz());
		}
		if (compare == 0) {
			compare = getTmcOrtsTyp().getValue().compareTo(o.getTmcOrtsTyp().getValue());
		}
		if (compare == 0) {
			compare = TmcCache.compareTmcLinien(tmcLinie, o.tmcLinie);
		}
		return compare;
	}

	/**
	 * Liefert den Straßentyp dieser TMC-Linie zurück.
	 * 
	 * @return den Straßentyp dieser TMC-Linie
	 */
	private String getStrassenTyp() {
		if (strassenTyp == null) {
			strassenTyp = "Z";
			final String tmcStrassenNummer = getTmcStrassenNummer();
			if ((tmcStrassenNummer != null) && (tmcStrassenNummer.length() > 0)) {
				strassenTyp = tmcStrassenNummer.substring(0, 1);
			}
		}
		return strassenTyp;
	}

	/**
	 * Liefert die Straßennummer dieser TMC-Linie zurück.
	 * 
	 * @return die Straßennummer dieser TMC-Linie
	 */
	private int getStrassenNummer() {
		if (strassenNummer == null) {
			strassenNummer = -1;
			final String tmcStrassenNummer = getTmcStrassenNummer();
			if ((tmcStrassenNummer != null) && (tmcStrassenNummer.length() > 0)) {
				final Scanner s = new Scanner(tmcStrassenNummer.substring(1));
				final Pattern p = Pattern.compile(NUMBER_REGEX);
				final String nummer = s.findInLine(p);
				if ((nummer != null) && (nummer.length() > 0)) {
					try {
						strassenNummer = Integer.parseInt(nummer);
					} catch (final NumberFormatException nfe) {
						// do nothing
					}
				}
			}
		}
		return strassenNummer;
	}

	/**
	 * Liefert den Straßenzusatz dieser TMC-Linie zurück.
	 * 
	 * @return den Straßenzusatz dieser TMC-Linie
	 */
	private String getStrassenZusatz() {
		if (strassenZusatz == null) {
			strassenZusatz = "";
			final String tmcStrassenNummer = getTmcStrassenNummer();
			if ((tmcStrassenNummer != null) && (tmcStrassenNummer.length() > 0)) {
				final Scanner s = new Scanner(tmcStrassenNummer.substring(1));
				try {
					final Pattern p = Pattern.compile(NUMBER_REGEX);
					final String nummer = s.findInLine(p);

					String[] zusaetze;
					if (nummer != null) {
						zusaetze = getTmcStrassenNummer().substring(1).split(nummer);
						for (final String element : zusaetze) {
							strassenZusatz += element.trim();
						}
					} else {
						// FIXME ist sinnlos, weil immer ein leeres Arra zurückkommt
						// zusaetze = getTmcStrassenNummer().substring(1).split(".");
					}
				} finally {
					s.close();
				}

			}
		}
		return strassenZusatz;
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = super.hashCode();
		result = (prime * result) + ((richtung == null) ? 0 : richtung.hashCode());
		result = (prime * result) + ((tmcLinie == null) ? 0 : tmcLinie.hashCode());
		return result;
	}

	@Override
	public boolean equals(final Object obj) {
		if (this == obj) {
			return true;
		}
		if (obj == null) {
			return false;
		}
		if (getClass() != obj.getClass()) {
			return false;
		}
		final TmcLinieWrapper other = (TmcLinieWrapper) obj;
		return super.equals(obj) && richtung.equals(other.getRichtung());
	}
}
