/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2011 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.bitctrl.modell.util.rdstmc;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

import de.bsvrz.sys.funclib.bitctrl.modell.att.Feld;
import de.bsvrz.sys.funclib.bitctrl.modell.metamodellglobal.attribute.AttJaNein;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.attribute.AttWgs84Breite;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.attribute.AttWgs84Laenge;
import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcglobal.attribute.AttTmcLocationCode;
import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcglobal.attribute.AttTmcOrtsTyp;
import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcglobal.konfigurationsdaten.KdTmcLocationCode.Daten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcglobal.objekte.TmcLocationCode;

/**
 * Wrapper, welcher für ein TmcLocationCode-SystemObjekt sämtliche Daten
 * vorhält.
 * 
 * @author BitCtrl Systems GmbH, Andreas Meissner
 * 
 * @see TmcLocationCode
 */
public class TmcLocationCodeWrapper {

	/**
	 * Map mit zu benutzenden Texten (Abk&uuml;rzungen) f&uuml;r TmcOrtsTypen.
	 */
	private static Map<AttTmcOrtsTyp, String> tmcOrtsTypAbkuerzungenMap;

	static {
		tmcOrtsTypAbkuerzungenMap = new HashMap<>();
		tmcOrtsTypAbkuerzungenMap.put(AttTmcOrtsTyp.ZUSTAND_101001_KNOTENPUNKT_AUTOBAHNKREUZ, "Kreuz");
		tmcOrtsTypAbkuerzungenMap.put(AttTmcOrtsTyp.ZUSTAND_101002_KNOTENPUNKT_AUTOBAHNDREIECK, "Dreieck");
		tmcOrtsTypAbkuerzungenMap.put(AttTmcOrtsTyp.ZUSTAND_101003_KNOTENPUNKT_AUTOBAHNANSCHLUSSSTELLE, "AS");
		tmcOrtsTypAbkuerzungenMap.put(AttTmcOrtsTyp.ZUSTAND_101004_KNOTENPUNKT_AUTOBAHNAUSFAHRT, "Ausfahrt");
		tmcOrtsTypAbkuerzungenMap.put(AttTmcOrtsTyp.ZUSTAND_101005_KNOTENPUNKT_AUTOBAHNEINFAHRT, "Einfahrt");
		tmcOrtsTypAbkuerzungenMap.put(AttTmcOrtsTyp.ZUSTAND_101011_KNOTENPUNKT_KREUZUNG, "Kreuzung");
		tmcOrtsTypAbkuerzungenMap.put(AttTmcOrtsTyp.ZUSTAND_101015_KNOTENPUNKT_AUSFAHRT, "Ausfahrt");
		tmcOrtsTypAbkuerzungenMap.put(AttTmcOrtsTyp.ZUSTAND_103003_SONSTIGER_MARKANTER_PUNKT_RASTSTAETTE, "Raststätte");
		tmcOrtsTypAbkuerzungenMap.put(
				AttTmcOrtsTyp.ZUSTAND_103004_SONSTIGER_MARKANTER_PUNKT_RASTPLATZ_MIT_EINZELNEN_EINRICHTUNGEN,
				"Rastplatz");
		tmcOrtsTypAbkuerzungenMap.put(AttTmcOrtsTyp.ZUSTAND_101011_KNOTENPUNKT_KREUZUNG, "Kreuzung");
		tmcOrtsTypAbkuerzungenMap.put(AttTmcOrtsTyp.ZUSTAND_103014_SONSTIGER_MARKANTER_PUNKT_GRENZUEBERGANG,
				"Grenzübergang");
	}

	/** Der in diesem Wrapper gekapselte TmcLocationCode. */
	private final TmcLocationCode tmcLocationCode;

	/** Das zu dem gekapseltem TmcLocationCode gehörige Datum. */
	private final Daten datum;

	/** Die bereinigte Straßennummer. */
	private String strassenNummer;

	/**
	 * Der Konstruktor.
	 * 
	 * @param tmcLocationCode der TMC-LocationCode, darf nicht <code>null</code>
	 *                        sein
	 */
	public TmcLocationCodeWrapper(final TmcLocationCode tmcLocationCode) {
		assert tmcLocationCode != null;
		assert tmcLocationCode.getKdTmcLocationCode() != null;
		assert tmcLocationCode.getKdTmcLocationCode().getDatum() != null;

		this.tmcLocationCode = tmcLocationCode;
		datum = tmcLocationCode.getKdTmcLocationCode().getDatum();
	}

	/**
	 * Liefert das in diesem Objekt gekapselte TmcLocationCode-Systemobjekt zurück.
	 * 
	 * @return das in diesem Objekt gekapselte TmcLocationCode-Systemobjekt, niemals
	 *         <code>null</code>
	 */
	public TmcLocationCode getTmcLocationCode() {
		return tmcLocationCode;
	}

	/**
	 * Liefert den Namen dieses TMC-LocationCodes zurück.
	 * 
	 * @return der Name dieses TMC-LocationCodes
	 */
	public String getName() {
		return tmcLocationCode.getName();
	}

	/**
	 * Eintrag wurde zuletzt geändert am angegebenen Datum.
	 * 
	 * @return der Wert von AenderungsDatum
	 */
	public String getAenderungsDatum() {
		return datum.getAenderungsDatum();
	}

	/**
	 * Ortscode ist aktiv.
	 * 
	 * @return der Wert von Aktiviert
	 */
	public AttJaNein getAktiviert() {
		return datum.getAktiviert();
	}

	/**
	 * Möglichkeit, aus der Straße in negativer Erfassungsrichtung auszufahren.
	 * 
	 * @return der Wert von AusfahrtNegativeRichtung
	 */
	public AttJaNein getAusfahrtNegativeRichtung() {
		return datum.getAusfahrtNegativeRichtung();
	}

	/**
	 * Nummer der Ausfahrt (z. B. "4a" ).
	 * 
	 * @return der Wert von AusfahrtNr
	 */
	public String getAusfahrtNr() {
		return datum.getAusfahrtNr();
	}

	/**
	 * Möglichkeit, aus der Straße in positiver Erfassungsrichtung herauszufahren.
	 * 
	 * @return der Wert von AusfahrtPositiveRichtung
	 */
	public AttJaNein getAusfahrtPositiveRichtung() {
		return datum.getAusfahrtPositiveRichtung();
	}

	/**
	 * Möglichkeit, in die Straße in negativer Erfassungsrichtung einzufahren.
	 * 
	 * @return der Wert von EinfahrtNegativeRichtung
	 */
	public AttJaNein getEinfahrtNegativeRichtung() {
		return datum.getEinfahrtNegativeRichtung();
	}

	/**
	 * Möglichkeit, auf die Straße in positiver Erfassungsrichtung aufzufahren.
	 * 
	 * @return der Wert von EinfahrtPositiveRichtung
	 */
	public AttJaNein getEinfahrtPositiveRichtung() {
		return datum.getEinfahrtPositiveRichtung();
	}

	/**
	 * Daten wurden getestet.
	 * 
	 * @return der Wert von Getestet
	 */
	public AttJaNein getGetestet() {
		return datum.getGetestet();
	}

	/**
	 * Ortscode gehört zum angegebenen Netzknoten A.
	 * 
	 * @return der Wert von NetzKnotenNrA
	 */
	public String getNetzKnotenNrA() {
		return datum.getNetzKnotenNrA();
	}

	/**
	 * Ortscode gehört zum angegebenen Netzknoten B.
	 * 
	 * @return der Wert von NetzKnotenNrB
	 */
	public String getNetzKnotenNrB() {
		return datum.getNetzKnotenNrB();
	}

	/**
	 * Ortscode gehört zur angegebenen Polizeidirektion.
	 * 
	 * @return der Wert von PolizeiDirektion
	 */
	public String getPolizeiDirektion() {
		return datum.getPolizeiDirektion();
	}

	/**
	 * Sondereintrag. Interpretation ist nicht definiert.
	 * 
	 * @return der Wert von Sondereintrag
	 */
	public Feld<String> getSondereintrag() {
		return datum.getSondereintrag();
	}

	/**
	 * Ortscode gehört zur angegebenen Station.
	 * 
	 * @return der Wert von Station
	 */
	public String getStation() {
		return datum.getStation();
	}

	/**
	 * Tern.
	 * 
	 * @return der Wert von Tern
	 */
	public AttJaNein getTern() {
		return datum.getTern();
	}

	/**
	 * Zusätzlicher Name.
	 * 
	 * <p>
	 * Namen von Orten sind im Regelfall in der Sprache des jeweiligen Ortes
	 * anzugeben. Es ist jedoch auch zulässig, Fassungen von Ortstabellen mit (zwei-
	 * oder mehrsprachigen) Ortsnamen, welche in andere Sprachen übersetzt wurden,
	 * zu erstellen. Der Name eines Anschlusses darf eine Beschreibung sein, wie sie
	 * Straßennutzern am besten bekannt ist.
	 * </p>
	 * 
	 * @return der Wert von TmcErsterName
	 */
	public String getTmcErsterName() {
		return datum.getTmcErsterName();
	}

	/**
	 * Liefert den Namen dieses TmcLocationCodes zur&uuml;ck. Dieser Name setzt sich
	 * aus einem m&ouml;glichen Ortstyp-abh&auml;ngigen Pr&auml;fix und dem ersten
	 * Namen zusammen.
	 * 
	 * @return den Namen dieses TmcLocationCodes
	 * 
	 * @see #getTmcOrtsTyp()
	 * @see #getTmcErsterName()
	 */
	public String getTmcName() {
		if ((getTmcErsterName() == null) || "".equals(getTmcErsterName())) {
			return getTmcErsterName();
		}
		final StringBuilder buf = new StringBuilder();
		final AttTmcOrtsTyp tmcOrtsTyp = getTmcOrtsTyp();
		if (tmcOrtsTyp != null) {
			final String abkuerzungOrtsTyp = tmcOrtsTypAbkuerzungenMap.get(tmcOrtsTyp);
			if ((abkuerzungOrtsTyp != null) && !"".equals(abkuerzungOrtsTyp)) {
				buf.append(abkuerzungOrtsTyp).append(" ");
			}
		}
		return buf.append(getTmcErsterName()).toString();
	}

	/**
	 * Ortscode (TMC Location Code).
	 * 
	 * @return der Wert von TmcLocationCode
	 */
	public AttTmcLocationCode getAttTmcLocationCode() {
		return datum.getTmcLocationCode();
	}

	/**
	 * Ortstyp und OrtSubtyp des Ortes entsprechend der Festlegungen der LCL.
	 * 
	 * <p>
	 * Für die Sprachunabhängigkeit der weitergegebenen Informationen und zur
	 * Mitteilung an das empfangende System, welche Datenfelder zu erwarten sind,
	 * sind Ortstypen und Untertypen erforderlich. Innerhalb einer jeden Kategorie
	 * wird (grundsätzlich) nach Ortstypen unterschieden, sofern für einen Ort ein
	 * funktionaler Unterschied besteht in der Art, wie er vom Empfänger einer
	 * Meldung zu behandeln ist.
	 * </p>
	 * 
	 * <p>
	 * Orts- und Untertypen werden gemeinsam kodiert. Dabei gilt folgendes
	 * Kodierschema:
	 * </p>
	 * 
	 * <ul>
	 * <li>Der TMC-Typ 'Punkt', 'Linie' oder 'Gebiet' ergibt sich aus dem
	 * Objekttyp.</li>
	 * <li>Der Name wird gebildet aus 'OrtTypName:UntertypName', wobei
	 * 'UnterTypName' gleich 'Allgemein' verwendet wird, sofern kein Untertyp
	 * vorhanden ist.</li>
	 * <li>Der Wert des Aufzahlungstyps ist eine sechsstellige Zahl der Form TNNSSS
	 * mit T gleich Typ (1=Punkt, 2=Linie, 3=Gebiet), NN gleich Orttypnummer und SSS
	 * gleich Untertypnummer entsprechend LCL, jeweils mit führenden Nullen.</li>
	 * </ul>
	 * 
	 * @return der Wert von TmcOrtsTyp
	 */
	public AttTmcOrtsTyp getTmcOrtsTyp() {
		return datum.getTmcOrtsTyp();
	}

	/**
	 * Gibt an, ob der Verkehr an diesem Punktort hauptsächlich den Charakter von
	 * Stadtverkehr (ja) oder von außerorts Verkehr (nein) hat.
	 * 
	 * @return der Wert von TmcStadtverkehr
	 */
	public AttJaNein getTmcStadtverkehr() {
		return datum.getTmcStadtverkehr();
	}

	/**
	 * Straßenname der Straße.
	 * 
	 * @return der Wert von TmcStrassenName
	 */
	public String getTmcStrassenName() {
		return datum.getTmcStrassenName();
	}

	/**
	 * Straßennummer der Straße, die durch diesen Ortscode referenziert wird.
	 * 
	 * @return der Wert von TmcStrassenNummer
	 */
	public String getTmcStrassenNummer() {
		if (strassenNummer == null) {
			strassenNummer = datum.getTmcStrassenNummer().replace(" ", "");
		}
		return strassenNummer;
	}

	/**
	 * Zusätzlicher Name.
	 * 
	 * <p>
	 * Namen von Orten sind im Regelfall in der Sprache des jeweiligen Ortes
	 * anzugeben. Es ist jedoch auch zulässig, Fassungen von Ortstabellen mit (zwei-
	 * oder mehrsprachigen) Ortsnamen, welche in andere Sprachen übersetzt wurden,
	 * zu erstellen. Der Name eines Anschlusses darf eine Beschreibung sein, wie sie
	 * Straßennutzern am besten bekannt ist.
	 * </p>
	 * 
	 * @return der Wert von TmcZweiterName
	 */
	public String getTmcZweiterName() {
		return datum.getTmcZweiterName();
	}

	/**
	 * Umleitung in negative Richtung über angegebene Umleitungsstrecke.
	 * 
	 * @return der Wert von UmleitungNegativeRichtung
	 */
	public String getUmleitungNegativeRichtung() {
		return datum.getUmleitungNegativeRichtung();
	}

	/**
	 * Umleitung in positive Richtung über angegebene Umleitungsstrecke.
	 * 
	 * @return der Wert von UmleitungPositiveRichtung
	 */
	public String getUmleitungPositiveRichtung() {
		return datum.getUmleitungPositiveRichtung();
	}

	/**
	 * Wurde Datensatz gegenüber letzter LocationCodeListe verändert.
	 * 
	 * @return der Wert von Veraenderung
	 */
	public AttJaNein getVeraenderung() {
		return datum.getVeraenderung();
	}

	/**
	 * Ortscode liegt im angegebenen Verwaltungsbereich.
	 * 
	 * @return der Wert von VerwaltungsBereich
	 */
	public String getVerwaltungsBereich() {
		return datum.getVerwaltungsBereich();
	}

	/**
	 * Zur Angabe, dass ein Ort in negativer Erfassungsrichtung vorhanden ist.
	 * 
	 * @return der Wert von VorhandenNegativeRichtung
	 */
	public AttJaNein getVorhandenNegativeRichtung() {
		return datum.getVorhandenNegativeRichtung();
	}

	/**
	 * Zur Angabe, dass ein Ort in der positiven Erfassungsrichtung vorhanden ist.
	 * 
	 * @return der Wert von VorhandenPositiveRichtung
	 */
	public AttJaNein getVorhandenPositiveRichtung() {
		return datum.getVorhandenPositiveRichtung();
	}

	/**
	 * x-Koordinate (Längenangaben gemäß WGS84).
	 * 
	 * @return der Wert von X
	 */
	public AttWgs84Laenge getX() {
		return datum.getX();
	}

	/**
	 * y-Koordinate (Breitenangabe gemäß WGS84).
	 * 
	 * @return der Wert von Y
	 */
	public AttWgs84Breite getY() {
		return datum.getY();
	}

	@Override
	public String toString() {
		final StringBuilder buf = new StringBuilder(getTmcName());
		if (getAttTmcLocationCode() != null) {
			buf.append(" (").append(getAttTmcLocationCode().getValue()).append(")");
		}
		return buf.toString();
	}

	@Override
	public int hashCode() {
		return Objects.hash(tmcLocationCode);
	}

	@Override
	public boolean equals(final Object obj) {
		if (this == obj) {
			return true;
		}
		if (obj == null) {
			return false;
		}
		if (getClass() != obj.getClass()) {
			return false;
		}
		final TmcLocationCodeWrapper other = (TmcLocationCodeWrapper) obj;
		if (!Objects.equals(tmcLocationCode, other.tmcLocationCode)) {
			return false;
		}
		return true;
	}
}
