/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2011 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.bitctrl.modell.util.test;

import java.util.ArrayList;
import java.util.List;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.sys.funclib.application.StandardApplication;
import de.bsvrz.sys.funclib.application.StandardApplicationRunner;
import de.bsvrz.sys.funclib.bitctrl.modell.AnmeldeException;
import de.bsvrz.sys.funclib.bitctrl.modell.DefaultObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.SystemObjekt;
import de.bsvrz.sys.funclib.bitctrl.modell.att.Zeitstempel;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkextlsglobal.attribute.AttTlsDEFehlerStatus;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkextlsglobal.attribute.AttTlsDEKanalStatus;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkextlsglobal.attribute.AttTlsDEProjektierungsStatus;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkextlsglobal.attribute.AttTlsHersteller;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkextlsglobal.attribute.AttTlsHerstellerDefinierterCode;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkextlsglobal.konfigurationsdaten.KdDe;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkextlsglobal.objekte.DeUfd;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkextlsglobal.objekte.DeVlt;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkextlsglobal.onlinedaten.OdTlsGloDeFehler;
import de.bsvrz.sys.funclib.commandLineArgs.ArgumentList;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Testdatensender für diejenigen Attributgruppen, die durch KzdTestSender und
 * UfdTestSender nicht bedient werden, die aber für eine sinnvolle Simulation
 * erforderlich oder zumindest vorteilhaft sind.
 *
 * Im Einzelnen sind dies:
 * <ul>
 * <li>Globale DE-Fehler FG2,3,6 (kommandozeilenoption
 * -gloDeFehler=[2|3|6|keine])</li>
 * </ul>
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 */
public class RestDatenSender implements StandardApplication {

	/**
	 * Liste aller DE UFD (=FG3), die keine Sammelkanäle sind.
	 */
	private final List<DeUfd> deUfdList = new ArrayList<>();

	/**
	 * Liste aller DE VLT (=FG6), die keine Sammelkanäle sind.
	 */
	private final List<DeVlt> deVltList = new ArrayList<>();

	/**
	 * Liste aller Funktionsgruppen, die bedinet werden. Gültige WErte: 2,3,6.
	 */
	private final List<Integer> funktionsGruppen = new ArrayList<>();

	/**
	 * Objekt für Debug-Ausgaben.
	 */
	private Debug debug;

	@Override
	public void parseArguments(final ArgumentList argumentList) throws Exception {
		debug = Debug.getLogger();
		final String fgListStr = argumentList.fetchArgument("-gloDeFehler=keine").asNonEmptyString();
		if ("keine".equalsIgnoreCase(fgListStr)) {
			debug.warning("Versende keine globalen DE-Fehler");
		} else {
			final String[] fgStr = fgListStr.split(",");
			for (final String s : fgStr) {
				funktionsGruppen.add(Integer.parseInt(s));
			}
			debug.info("Versende globale DE-Fehler für folgende FG", funktionsGruppen);
		}
	}

	@Override
	public void initialize(final ClientDavInterface connection) throws Exception {
		DefaultObjektFactory.getInstanz().setDav(connection);
		doInitDeListen();
		doInitGlobaleDeFehler();
	}

	/**
	 * Befüllt die Listen mit den DE-Modellobjekten, wobei die Sammelkanäle
	 * ignoriert werden.
	 */
	private void doInitDeListen() {
		for (final SystemObjekt o : DefaultObjektFactory.getInstanz().bestimmeModellobjekte("typ.deUfd")) {
			final DeUfd deUfd = (DeUfd) o;
			final KdDe.Daten d = deUfd.getKdDe().getDatum();
			if (d.getDEKanal().getValue() == 255) {
				debug.config("Ignoriere FG3 Samelkanal ", deUfd);
				continue;
			}
			deUfdList.add(deUfd);
		}
		for (final SystemObjekt o : DefaultObjektFactory.getInstanz().bestimmeModellobjekte("typ.deVlt")) {
			final DeVlt deVlt = (DeVlt) o;
			final KdDe.Daten d = deVlt.getKdDe().getDatum();
			if (d.getDEKanal().getValue() == 255) {
				debug.config("Ignoriere FG6 Samelkanal ", deVlt);
				continue;
			}
			deVltList.add(deVlt);
		}
	}

	/**
	 * Anmeldung für globale DE-Fehler an allen DE-Listen, die per Kommandozeile
	 * gefordert wurden. Gleichzeitig Versand der Daten.
	 */
	private void doInitGlobaleDeFehler() {
		final long z = 60000 * (DefaultObjektFactory.getInstanz().getDav().getTime() / 60000);
		for (final Integer i : funktionsGruppen) {
			switch (i) {
			case 3: {
				doAnmeldenUndSendenGloDeFehlerUfd(z);
				break;
			}
			case 6: {
				doAnmeldenUndSendenGloDeFehlerVlt(z);
				break;
			}
			}
		}
	}

	/**
	 * Anmeldung und Versand eines "gut" Datensatezs für alle DE UFD.
	 *
	 * @param z Zeitstempel der Datensätze
	 */
	private void doAnmeldenUndSendenGloDeFehlerUfd(final long z) {
		for (final DeUfd deUfd : deUfdList) {
			final OdTlsGloDeFehler.Daten gloDeFehlerDaten = deUfd.getOdTlsGloDeFehler().createDatum();
			fillGloDeFehlerDatensatzGutmeldung(z, gloDeFehlerDaten);
			try {
				deUfd.getOdTlsGloDeFehler().anmeldenQuelle(OdTlsGloDeFehler.Aspekte.TlsAntwort, gloDeFehlerDaten);
			} catch (final AnmeldeException ex) {
				debug.error("Fehler bei Quellanmeldung an globale DEFehler " + deUfd, ex);
			}
		}
	}

	/**
	 * Anmeldung und Versand eines "gut" Datensatezs für alle DE VLT.
	 *
	 * @param z Zeitstempel der Datensätze
	 */
	private void doAnmeldenUndSendenGloDeFehlerVlt(final long z) {
		for (final DeVlt deUfd : deVltList) {
			final OdTlsGloDeFehler.Daten gloDeFehlerDaten = deUfd.getOdTlsGloDeFehler().createDatum();
			fillGloDeFehlerDatensatzGutmeldung(z, gloDeFehlerDaten);
			try {
				deUfd.getOdTlsGloDeFehler().anmeldenQuelle(OdTlsGloDeFehler.Aspekte.TlsAntwort, gloDeFehlerDaten);
			} catch (final AnmeldeException ex) {
				debug.error("Fehler bei Quellanmeldung an globale DEFehler " + deUfd, ex);
			}
		}
	}

	/**
	 * Befüllt einen Datensatz mit Werten für eine Gutmeldung.
	 *
	 * @param z                Zeitstempel des Datensatzes
	 * @param gloDeFehlerDaten zu befüllender Datensatz
	 */
	private void fillGloDeFehlerDatensatzGutmeldung(final long z, final OdTlsGloDeFehler.Daten gloDeFehlerDaten) {
		gloDeFehlerDaten.dSetZeitstempel(new Zeitstempel(z));
		gloDeFehlerDaten.setDEFehlerStatus(AttTlsDEFehlerStatus.ZUSTAND_0_OK);
		gloDeFehlerDaten.setDEKanalStatus(AttTlsDEKanalStatus.ZUSTAND_0_AKTIV);
		gloDeFehlerDaten.setDEProjektierungsStatus(AttTlsDEProjektierungsStatus.ZUSTAND_0_PROJEKTIERUNG_OK);
		gloDeFehlerDaten.setHersteller(AttTlsHersteller.ZUSTAND_0_HERSTELLER_UNBEKANNT);
		gloDeFehlerDaten.setHerstellerDefinierterCode(new AttTlsHerstellerDefinierterCode((byte) 0));
	}

	/**
	 * Die Hauptfunktion des Restdatensenders.
	 *
	 * @param args Programm-Argumente
	 */
	public static void main(final String[] args) {
		StandardApplicationRunner.run(new RestDatenSender(), args);
	}
}
