/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2011 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.sys.funclib.bitctrl.modell.util;

import de.bsvrz.sys.funclib.bitctrl.modell.att.Feld;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.attribute.AttLaenge;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.konfigurationsdaten.KdPunktLiegtAufLinienObjekt.Daten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.objekte.Linie;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.MessQuerschnittAllgemein;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.StrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.StrassenTeilSegment;

/**
 * Funktionen zum Ermitteln von Daten aus MessquerschnittAllgemein-Objekten.
 * 
 * @author BitCtrl Systems GmbH, Uwe Peuker
 * @version $Id$
 */
public final class MqAllgemeinUtil {

	/**
	 * ermittelt den Offset innerhalb des Straßensegments, auf dem sich der
	 * übergebene Messquerschnitt befindet.
	 * 
	 * Die Funktion ermittelt das in der Konfiguration im Datensatz
	 * PunktLiegtAufLinienObjekt definierten Offset. Kann der Datensatz nicht
	 * ausgewertet werden, liefert die Funktion den Wert 0.
	 * 
	 * @param mq der Messquerschnitt
	 * @return das Straßensegment oder 0
	 */
	private static double getOffset(final MessQuerschnittAllgemein mq) {
		double result = 0;
		final Daten daten = mq.getKdPunktLiegtAufLinienObjekt().getDatum();
		if (daten != null) {
			final AttLaenge offset = daten.getOffset();
			if (offset != null) {
				result = offset.doubleValue();
			}
		}
		return result;
	}

	/**
	 * ermittelt das Straßensegment, auf dem sich der übergebene Messquerschnitt
	 * befindet.
	 * 
	 * Die Funktion ermittelt das in der Konfiguration im Datensatz
	 * PunktLiegtAufLinienObjekt referenzierte Linienobjekt. Dieses sollte ein
	 * Straßensegment sein. Kann der Datensatz nicht ausgewertet werden, liefert die
	 * Funktion den Wert <code>null</code>.
	 * 
	 * @param mq der Messquerschnitt
	 * @return das Straßensegment oder <code>null</code>
	 */
	public static StrassenSegment getStrassenSegment(final MessQuerschnittAllgemein mq) {
		StrassenSegment result = null;
		final Daten daten = mq.getKdPunktLiegtAufLinienObjekt().getDatum();
		if (daten != null) {
			result = (StrassenSegment) daten.getLinienReferenz();
		}
		return result;
	}

	/**
	 * ermittelt das Straßensegmentteilsegment, auf dem sich der übergebene
	 * Messquerschnitt befindet.
	 * 
	 * Die Funktion ermittelt das Teilsegment aus der Menge der für das zugeordnete
	 * Straßensegment definierten Teilsegmente und dem dort definierten Offset für
	 * den Messquerschnitt. Wenn kein gültes Straßenteilsegment ermittelt werden
	 * kann, wird der Wert <code>null</code> geliefert.
	 * 
	 * @param mq der Messquerschnitt
	 * @return das Straßenteilsegment oder <code>null</code>
	 */
	public static StrassenTeilSegment getStrassenTeilSegment(final MessQuerschnittAllgemein mq) {

		StrassenTeilSegment result = null;
		final StrassenSegment segment = getStrassenSegment(mq);

		if (segment != null) {
			final de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.konfigurationsdaten.KdBestehtAusLinienObjekten.Daten daten = segment
					.getKdBestehtAusLinienObjekten().getDatum();

			if (daten != null) {
				final double offset = MqAllgemeinUtil.getOffset(mq);
				final Feld<Linie> linien = daten.getLinienReferenz();
				double teiloffset = 0;
				for (final Linie linie : linien) {
					final de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.konfigurationsdaten.KdLinie.Daten linienDaten = linie
							.getKdLinie().getDatum();
					if ((linienDaten != null) && (linienDaten.getLaenge() != null)) {
						if ((teiloffset + linienDaten.getLaenge().doubleValue()) > offset) {
							result = (StrassenTeilSegment) linie;
							break;
						}
						teiloffset += linienDaten.getLaenge().doubleValue();
					}
				}
			}
		}

		return result;
	}

	/** Konstruktor. */
	private MqAllgemeinUtil() {
		// es gibt keine Instanzen dieser Klasse.
	}

}
